/**
 * PropertyPro - Invoice Email API Route
 * Send lease invoice PDFs via email
 */

import { NextRequest } from "next/server";
import { UserRole } from "@/types";
import {
  createSuccessResponse,
  createErrorResponse,
  handleApiError,
  withRoleAndDB,
  parseRequestBody,
  isValidObjectId,
} from "@/lib/api-utils";
import { EmailService } from "@/lib/email-service";
import { Lease } from "@/models";
import { generateEmailPDF } from "@/lib/email-pdf-service";
import { buildInvoiceEmailHtml } from "@/lib/templates/invoice-email-template";
import { generateInvoicePdfBuffer } from "@/lib/services/invoice-pdf.service";
import { Invoice } from "@/models";
import { getCompanyInfoServer } from "@/lib/utils/company-info";

// ============================================================================
// POST /api/invoices/email - Send invoice PDF via email
// ============================================================================

export const POST = withRoleAndDB([
  UserRole.ADMIN,
  UserRole.MANAGER,
  UserRole.MANAGER,
])(async (user, request: NextRequest) => {
  try {
    const { success, data: body, error } = await parseRequestBody(request);
    if (!success) {
      return createErrorResponse(error!, 400);
    }

    const { leaseId, invoiceId, to, subject, message, invoiceNumber } = body;

    if (!to || !subject || !message) {
      return createErrorResponse(
        "Missing required fields: to, subject, message",
        400
      );
    }

    if (!leaseId && !invoiceId) {
      return createErrorResponse(
        "Either leaseId or invoiceId is required",
        400
      );
    }

    // Validate lease ID
    if (leaseId && !isValidObjectId(leaseId)) {
      return createErrorResponse("Invalid lease ID", 400);
    }

    if (invoiceId && !isValidObjectId(invoiceId)) {
      return createErrorResponse("Invalid invoice ID", 400);
    }

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(to)) {
      return createErrorResponse("Invalid email address", 400);
    }

    // Find the lease to verify it exists and get additional context
    const lease = leaseId
      ? await Lease.findById(leaseId)
          .populate("tenantId", "firstName lastName email")
          .populate("propertyId", "name address")
      : null;

    if (leaseId && !lease) {
      return createErrorResponse("Lease not found", 404);
    }

    let invoice = null;

    if (invoiceId) {
      invoice = await Invoice.findById(invoiceId)
        .populate("tenantId", "firstName lastName email")
        .populate("propertyId", "name address");

      if (!invoice) {
        return createErrorResponse("Invoice not found", 404);
      }
    } else if (leaseId) {
      invoice = await Invoice.findOne({ leaseId })
        .sort({ issueDate: -1 })
        .populate("tenantId", "firstName lastName email")
        .populate("propertyId", "name address");
    }

    // Initialize email service
    const emailService = new EmailService();

    // Generate PDF using the email service
    let pdfBuffer: Buffer;
    let fileName = `invoice_${Date.now()}.pdf`;

    // Fetch company info from display settings
    const companyInfo = await getCompanyInfoServer();

    if (invoice) {
      pdfBuffer = await generateInvoicePdfBuffer(invoice as any);
      fileName = `${invoice.invoiceNumber || invoice._id}.pdf`;
    } else {
      const pdfResult = await generateEmailPDF({
        lease: lease as any,
        invoiceNumber,
        issueDate: new Date(),
        dueDate: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000),
        companyInfo: companyInfo || undefined,
      });

      if (!pdfResult.success || !pdfResult.buffer) {
        return createErrorResponse(
          pdfResult.error || "Failed to generate PDF",
          500
        );
      }

      pdfBuffer = pdfResult.buffer;
      fileName = pdfResult.fileName || fileName;
    }

    // Prepare email content
    const tenantName = invoice?.tenantId
      ? `${invoice.tenantId.firstName || ""} ${
          invoice.tenantId.lastName || ""
        }`.trim() ||
        invoice.tenantId.email ||
        "Tenant"
      : lease?.tenantId
      ? `${lease.tenantId.firstName} ${lease.tenantId.lastName}`
      : "Tenant";

    const propertyName =
      invoice?.propertyId?.name || lease?.propertyId?.name || "Property";

    const emailContent = buildInvoiceEmailHtml({
      message,
      propertyName,
      tenantName,
      invoiceNumber,
    });

    // Send email with PDF attachment
    const emailSent = await emailService.sendEmailWithAttachments(
      to,
      {
        subject,
        html: emailContent,
        text: message, // Fallback text version
      },
      [
        {
          filename: fileName,
          content: pdfBuffer,
          contentType: "application/pdf",
        },
      ]
    );

    if (emailSent) {
      return createSuccessResponse(
        {
          message: "Invoice email sent successfully",
          to,
          subject,
          fileName,
          leaseId,
          invoiceId,
          sentAt: new Date().toISOString(),
        },
        "Invoice emailed successfully"
      );
    } else {
      return createErrorResponse("Failed to send email", 500);
    }
  } catch (error) {
    console.error("Failed to send invoice email:", error);
    return handleApiError(error, "Failed to send invoice email");
  }
});
