/**
 * PropertyPro - Sessions API Route
 * Handle session listing and bulk operations
 */

import { NextRequest } from "next/server";
import { UserRole } from "@/types";
import {
  createSuccessResponse,
  createErrorResponse,
  handleApiError,
  withRoleAndDB,
  parsePaginationParams,
} from "@/lib/api-utils";

// ============================================================================
// GET /api/sessions - Get active sessions
// ============================================================================

export const GET = withRoleAndDB([UserRole.ADMIN, UserRole.MANAGER])(
  async (user, request: NextRequest) => {
    try {
      const { searchParams } = new URL(request.url);
      const paginationParams = parsePaginationParams(searchParams);
      const userId = searchParams.get("userId");
      const deviceType = searchParams.get("deviceType");
      const status = searchParams.get("status"); // active, idle, all

      // In a real implementation, you would:
      // 1. Query your session store (Redis, database, etc.)
      // 2. Apply filters based on query parameters
      // 3. Implement pagination
      // 4. Return session data with proper access control

      // Mock session data for demonstration
      const mockSessions = [
        {
          id: "session-1",
          userId: "user-1",
          userName: "John Doe",
          userEmail: "john.doe@example.com",
          userRole: UserRole.TENANT,
          deviceType: "desktop",
          browser: "Chrome 120.0",
          os: "Windows 11",
          ipAddress: "192.168.1.100",
          location: {
            city: "New York",
            country: "United States",
            region: "NY",
          },
          lastActivity: new Date(Date.now() - 1000 * 60 * 15), // 15 minutes ago
          createdAt: new Date(Date.now() - 1000 * 60 * 60 * 2), // 2 hours ago
          isCurrentSession: false,
        },
        {
          id: "session-2",
          userId: "user-1",
          userName: "John Doe",
          userEmail: "john.doe@example.com",
          userRole: UserRole.TENANT,
          deviceType: "mobile",
          browser: "Safari 17.0",
          os: "iOS 17.1",
          ipAddress: "10.0.0.50",
          location: {
            city: "New York",
            country: "United States",
            region: "NY",
          },
          lastActivity: new Date(Date.now() - 1000 * 60 * 60), // 1 hour ago
          createdAt: new Date(Date.now() - 1000 * 60 * 60 * 24), // 1 day ago
          isCurrentSession: false,
        },
        {
          id: "session-3",
          userId: "user-2",
          userName: "Jane Smith",
          userEmail: "jane.smith@example.com",
          userRole: UserRole.MANAGER,
          deviceType: "desktop",
          browser: "Firefox 121.0",
          os: "macOS 14.0",
          ipAddress: "203.0.113.45",
          location: {
            city: "San Francisco",
            country: "United States",
            region: "CA",
          },
          lastActivity: new Date(Date.now() - 1000 * 60 * 5), // 5 minutes ago
          createdAt: new Date(Date.now() - 1000 * 60 * 30), // 30 minutes ago
          isCurrentSession: false,
        },
        {
          id: "session-4",
          userId: "user-3",
          userName: "Mike Johnson",
          userEmail: "mike.johnson@example.com",
          userRole: UserRole.MANAGER,
          deviceType: "tablet",
          browser: "Chrome 120.0",
          os: "Android 14",
          ipAddress: "198.51.100.25",
          location: {
            city: "Chicago",
            country: "United States",
            region: "IL",
          },
          lastActivity: new Date(Date.now() - 1000 * 60 * 2), // 2 minutes ago
          createdAt: new Date(Date.now() - 1000 * 60 * 45), // 45 minutes ago
          isCurrentSession: false,
        },
      ];

      // Apply filters
      let filteredSessions = mockSessions;

      if (userId) {
        filteredSessions = filteredSessions.filter(
          (session) => session.userId === userId
        );
      }

      if (deviceType) {
        filteredSessions = filteredSessions.filter(
          (session) => session.deviceType === deviceType
        );
      }

      if (status && status !== "all") {
        const now = Date.now();
        filteredSessions = filteredSessions.filter((session) => {
          const minutesAgo = Math.floor(
            (now - session.lastActivity.getTime()) / (1000 * 60)
          );

          switch (status) {
            case "active":
              return minutesAgo < 5;
            case "idle":
              return minutesAgo >= 30;
            default:
              return true;
          }
        });
      }

      // Role-based access control
      if (user.role === UserRole.MANAGER) {
        // Property managers can only see sessions for non-admin users
        filteredSessions = filteredSessions.filter(
          (session) => ![UserRole.ADMIN, UserRole.MANAGER].includes(session.userRole)
        );
      }

      // Apply pagination
      const startIndex = (paginationParams.page - 1) * paginationParams.limit;
      const endIndex = startIndex + paginationParams.limit;
      const paginatedSessions = filteredSessions.slice(startIndex, endIndex);

      const totalCount = filteredSessions.length;
      const totalPages = Math.ceil(totalCount / paginationParams.limit);

      return createSuccessResponse({
        data: paginatedSessions,
        pagination: {
          page: paginationParams.page,
          limit: paginationParams.limit,
          total: totalCount,
          pages: totalPages,
        },
        summary: {
          totalSessions: totalCount,
          activeSessions: filteredSessions.filter((s) => {
            const minutesAgo = Math.floor(
              (Date.now() - s.lastActivity.getTime()) / (1000 * 60)
            );
            return minutesAgo < 5;
          }).length,
          idleSessions: filteredSessions.filter((s) => {
            const minutesAgo = Math.floor(
              (Date.now() - s.lastActivity.getTime()) / (1000 * 60)
            );
            return minutesAgo >= 30;
          }).length,
        },
      });
    } catch (error) {
      return handleApiError(error);
    }
  }
);

// ============================================================================
// DELETE /api/sessions - Bulk terminate sessions
// ============================================================================

export const DELETE = withRoleAndDB([UserRole.ADMIN])(
  async (user, request: NextRequest) => {
    try {
      const { searchParams } = new URL(request.url);
      const sessionIds = searchParams.get("ids")?.split(",") || [];
      const userId = searchParams.get("userId"); // Terminate all sessions for a user

      if (sessionIds.length === 0 && !userId) {
        return createErrorResponse("Session IDs or user ID required", 400);
      }

      let terminatedCount = 0;
      const errors: string[] = [];

      if (userId) {
        // Terminate all sessions for a specific user
        try {
          // In a real implementation, you would query and terminate all sessions for the user

          terminatedCount = 3; // Mock count
        } catch (error) {
          errors.push(`Failed to terminate sessions for user ${userId}`);
        }
      } else {
        // Terminate specific sessions
        for (const sessionId of sessionIds) {
          try {
            // In a real implementation, you would terminate each session

            terminatedCount++;
          } catch (error) {
            errors.push(`Failed to terminate session ${sessionId}`);
          }
        }
      }

      // Log the bulk termination
      const auditLog = {
        action: "bulk_session_termination",
        sessionIds: userId ? `all for user ${userId}` : sessionIds,
        terminatedBy: user.id,
        terminatedByEmail: user.email,
        terminatedCount,
        errors,
        timestamp: new Date(),
        ipAddress: request.headers.get("x-forwarded-for") || "unknown",
      };


      return createSuccessResponse(
        {
          terminatedCount,
          errors,
          terminatedAt: new Date(),
          terminatedBy: user.email,
        },
        `Successfully terminated ${terminatedCount} session(s)`
      );
    } catch (error) {
      return handleApiError(error);
    }
  }
);

// ============================================================================
// OPTIONS - Handle CORS preflight
// ============================================================================

export async function OPTIONS() {
  return new Response(null, {
    status: 200,
    headers: {
      "Access-Control-Allow-Origin": "*",
      "Access-Control-Allow-Methods": "GET, DELETE, OPTIONS",
      "Access-Control-Allow-Headers": "Content-Type, Authorization",
    },
  });
}
