/**
 * PropertyPro - Tenant Notification Settings API
 * Handles CRUD operations for tenant notification preferences
 */

import { NextRequest, NextResponse } from "next/server";
import { auth } from "@/lib/auth";
import connectDB from "@/lib/mongodb";
import { NotificationSettings } from "@/models";
import { UserRole } from "@/types";

// ============================================================================
// GET - Fetch tenant's notification settings
// ============================================================================

export async function GET(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    if (session.user.role !== UserRole.TENANT) {
      return NextResponse.json(
        { error: "Access denied. Tenant role required." },
        { status: 403 }
      );
    }

    await connectDB();

    // Find the tenant's notification settings
    const settings = await NotificationSettings.findOne({
      tenantId: session.user.id,
    }).lean();

    return NextResponse.json({
      success: true,
      data: settings,
    });
  } catch (error) {
    return NextResponse.json(
      {
        error:
          error instanceof Error
            ? error.message
            : "Failed to fetch notification settings",
      },
      { status: 500 }
    );
  }
}

// ============================================================================
// POST - Create new notification settings
// ============================================================================

export async function POST(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    if (session.user.role !== UserRole.TENANT) {
      return NextResponse.json(
        { error: "Access denied. Tenant role required." },
        { status: 403 }
      );
    }

    const body = await request.json();
    const { emailNotifications, reminderSchedule, preferences } = body;

    // Validation
    if (!emailNotifications || !reminderSchedule || !preferences) {
      return NextResponse.json(
        { error: "Missing required fields" },
        { status: 400 }
      );
    }

    if (!preferences.emailAddress) {
      return NextResponse.json(
        { error: "Email address is required" },
        { status: 400 }
      );
    }

    await connectDB();

    // Check if settings already exist
    const existingSettings = await NotificationSettings.findOne({
      tenantId: session.user.id,
    });

    if (existingSettings) {
      return NextResponse.json(
        { error: "Notification settings already exist. Use PUT to update." },
        { status: 409 }
      );
    }

    // Create new notification settings
    const settings = new NotificationSettings({
      tenantId: session.user.id,
      emailNotifications: {
        paymentReminders: emailNotifications.paymentReminders || false,
        paymentConfirmations: emailNotifications.paymentConfirmations || false,
        overdueNotices: emailNotifications.overdueNotices || false,
        receiptDelivery: emailNotifications.receiptDelivery || false,
      },
      reminderSchedule: {
        daysBeforeDue: reminderSchedule.daysBeforeDue || [7, 3, 1],
        overdueReminders: reminderSchedule.overdueReminders || false,
        overdueFrequency: reminderSchedule.overdueFrequency || "weekly",
      },
      preferences: {
        emailAddress: preferences.emailAddress,
        phoneNumber: preferences.phoneNumber || null,
        smsNotifications: preferences.smsNotifications || false,
        timezone: preferences.timezone || "America/New_York",
      },
    });

    await settings.save();

    return NextResponse.json({
      success: true,
      data: settings,
      message: "Notification settings created successfully",
    });
  } catch (error) {
    return NextResponse.json(
      {
        error:
          error instanceof Error
            ? error.message
            : "Failed to create notification settings",
      },
      { status: 500 }
    );
  }
}

// ============================================================================
// PUT - Update notification settings
// ============================================================================

export async function PUT(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    if (session.user.role !== UserRole.TENANT) {
      return NextResponse.json(
        { error: "Access denied. Tenant role required." },
        { status: 403 }
      );
    }

    const body = await request.json();
    const { emailNotifications, reminderSchedule, preferences } = body;

    await connectDB();

    // Find existing settings
    let settings = await NotificationSettings.findOne({
      tenantId: session.user.id,
    });

    if (!settings) {
      // Create new settings if they don't exist
      settings = new NotificationSettings({
        tenantId: session.user.id,
        emailNotifications: {
          paymentReminders: false,
          paymentConfirmations: false,
          overdueNotices: false,
          receiptDelivery: false,
        },
        reminderSchedule: {
          daysBeforeDue: [7, 3, 1],
          overdueReminders: false,
          overdueFrequency: "weekly",
        },
        preferences: {
          emailAddress: session.user.email || "",
          smsNotifications: false,
          timezone: "America/New_York",
        },
      });
    }

    // Update settings
    if (emailNotifications) {
      settings.emailNotifications = {
        paymentReminders:
          emailNotifications.paymentReminders ??
          settings.emailNotifications.paymentReminders,
        paymentConfirmations:
          emailNotifications.paymentConfirmations ??
          settings.emailNotifications.paymentConfirmations,
        overdueNotices:
          emailNotifications.overdueNotices ??
          settings.emailNotifications.overdueNotices,
        receiptDelivery:
          emailNotifications.receiptDelivery ??
          settings.emailNotifications.receiptDelivery,
      };
    }

    if (reminderSchedule) {
      settings.reminderSchedule = {
        daysBeforeDue:
          reminderSchedule.daysBeforeDue ??
          settings.reminderSchedule.daysBeforeDue,
        overdueReminders:
          reminderSchedule.overdueReminders ??
          settings.reminderSchedule.overdueReminders,
        overdueFrequency:
          reminderSchedule.overdueFrequency ??
          settings.reminderSchedule.overdueFrequency,
      };
    }

    if (preferences) {
      settings.preferences = {
        emailAddress:
          preferences.emailAddress ?? settings.preferences.emailAddress,
        phoneNumber:
          preferences.phoneNumber ?? settings.preferences.phoneNumber,
        smsNotifications:
          preferences.smsNotifications ?? settings.preferences.smsNotifications,
        timezone: preferences.timezone ?? settings.preferences.timezone,
      };
    }

    await settings.save();

    return NextResponse.json({
      success: true,
      data: settings,
      message: "Notification settings updated successfully",
    });
  } catch (error) {
    return NextResponse.json(
      {
        error:
          error instanceof Error
            ? error.message
            : "Failed to update notification settings",
      },
      { status: 500 }
    );
  }
}

// ============================================================================
// DELETE - Delete notification settings
// ============================================================================

export async function DELETE(request: NextRequest) {
  try {
    const session = await auth();

    if (!session?.user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    if (session.user.role !== UserRole.TENANT) {
      return NextResponse.json(
        { error: "Access denied. Tenant role required." },
        { status: 403 }
      );
    }

    await connectDB();

    // Delete the notification settings
    const result = await NotificationSettings.findOneAndDelete({
      tenantId: session.user.id,
    });

    if (!result) {
      return NextResponse.json(
        { error: "Notification settings not found" },
        { status: 404 }
      );
    }

    return NextResponse.json({
      success: true,
      message: "Notification settings deleted successfully",
    });
  } catch (error) {
    return NextResponse.json(
      {
        error:
          error instanceof Error
            ? error.message
            : "Failed to delete notification settings",
      },
      { status: 500 }
    );
  }
}
