/**
 * PropertyPro - Tenants API Routes
 * CRUD operations for tenant management
 */

export const dynamic = "force-dynamic";

import { NextRequest } from "next/server";
import { auth } from "@/lib/auth";
import connectDB from "@/lib/mongodb";
import { User } from "@/models";
import { UserRole } from "@/types";
import {
  createSuccessResponse,
  createErrorResponse,
  handleApiError,
  withRoleAndDB,
  parsePaginationParams,
  paginateQuery,
  parseRequestBody,
} from "@/lib/api-utils";
import {
  userSchema,
  paginationSchema,
  validateSchema,
} from "@/lib/validations";
import { z } from "zod";

// Schema for tenants listing filters (extends pagination with status)
const tenantFilterSchema = paginationSchema.extend({
  status: z
    .enum([
      "pending",
      "approved",
      "active",
      "inactive",
      "moved_out",
      "terminated",
      "under_review",
      "application_submitted",
    ])
    .optional(),
  propertyId: z.string().optional(),
});

// ============================================================================
// GET /api/tenants - Get all tenants with pagination and filtering
// ============================================================================

export const GET = withRoleAndDB([UserRole.ADMIN, UserRole.MANAGER])(
  async (user, request: NextRequest) => {
    try {
      const { searchParams } = new URL(request.url);
      const paginationParams = parsePaginationParams(searchParams);

      // Parse filter parameters
      const filterParams = {
        ...paginationParams,
        status: searchParams.get("status") || undefined,
        propertyId: searchParams.get("propertyId") || undefined,
      };

      // Validate filter parameters (use tenant-specific filter schema)
      const validation = validateSchema(tenantFilterSchema, filterParams);
      if (!validation.success) {
        return createErrorResponse(validation.errors.join(", "), 400);
      }

      const filters = validation.data;

      // Build query based on user role and filters
      let query: any = {
        role: UserRole.TENANT, // Only fetch users with tenant role
        deletedAt: null, // Exclude soft-deleted tenants
      };

      // Apply filters - support both legacy and new status filtering
      if (filters.status) {
        switch (filters.status) {
          case "pending":
            // Legacy support: map to new status system
            query.$or = [
              { tenantStatus: "application_submitted" },
              { tenantStatus: "under_review" },
            ];
            break;
          case "approved":
            query.tenantStatus = "approved";
            break;
          case "active":
            query.tenantStatus = "active";
            break;
          case "inactive":
            query.tenantStatus = "inactive";
            break;
          case "moved_out":
            query.tenantStatus = "moved_out";
            break;
          case "terminated":
            query.tenantStatus = "terminated";
            break;
          case "under_review":
            query.tenantStatus = "under_review";
            break;
          case "application_submitted":
            query.tenantStatus = "application_submitted";
            break;
          default:
            // If it's a direct tenant status, use it
            if (
              [
                "application_submitted",
                "under_review",
                "approved",
                "active",
                "inactive",
                "moved_out",
                "terminated",
              ].includes(filters.status)
            ) {
              query.tenantStatus = filters.status;
            }
            break;
        }
      }

      console.log(
        "🔍 Tenant API - Query after status filter:",
        JSON.stringify(query)
      );

      // For admin users, allow "Show All" option with high limit
      let result;
      if (user.role === UserRole.ADMIN && filters.limit >= 1000) {
        // Get all tenants for super admin when "Show All" is selected
        const { sortBy, sortOrder, search } = filters;

        // Add search functionality manually for super admin
        // IMPORTANT: Don't overwrite $or if it exists (for status filtering)
        if (search) {
          if (query.$or) {
            // If $or already exists (from status filter), combine with $and
            query.$and = [
              { $or: query.$or }, // Keep the status filter
              {
                $or: [
                  { firstName: { $regex: search, $options: "i" } },
                  { lastName: { $regex: search, $options: "i" } },
                  { email: { $regex: search, $options: "i" } },
                ],
              },
            ];
            delete query.$or; // Remove the old $or since we're using $and now
          } else {
            // No existing $or, safe to add search $or
            query.$or = [
              { firstName: { $regex: search, $options: "i" } },
              { lastName: { $regex: search, $options: "i" } },
              { email: { $regex: search, $options: "i" } },
            ];
          }
        }

        // Build sort object
        const sort: any = {};
        if (sortBy) {
          sort[sortBy] = sortOrder === "asc" ? 1 : -1;
        } else {
          sort.createdAt = -1; // Default sort by creation date
        }

        console.log(
          "🔍 Tenant API - Final query (admin, limit >= 1000):",
          JSON.stringify(query)
        );

        // Execute query without pagination for super admin "Show All"
        const [data, total] = await Promise.all([
          User.find(query).sort(sort).lean(),
          User.countDocuments(query),
        ]);

        result = {
          data,
          pagination: {
            page: 1,
            limit: total,
            total,
            totalPages: 1,
            hasNext: false,
            hasPrev: false,
          },
        };
      } else {
        // Use normal paginated query for all other cases
        console.log(
          "🔍 Tenant API - Final query (normal pagination):",
          JSON.stringify(query)
        );
        result = await paginateQuery(User, query, filters);
      }

      // No need to populate since we're fetching users directly
      const populatedData = result.data;

      return createSuccessResponse(
        populatedData,
        "Tenants retrieved successfully",
        result.pagination
      );
    } catch (error) {
      return handleApiError(error);
    }
  }
);

// ============================================================================
// POST /api/tenants - Create a new tenant
// ============================================================================

export const POST = withRoleAndDB([UserRole.ADMIN, UserRole.MANAGER])(
  async (user, request: NextRequest) => {
    try {
      const { success, data: body, error } = await parseRequestBody(request);
      if (!success) {
        return createErrorResponse(error!, 400);
      }

      // Transform and prepare tenant data
      const tenantData = {
        ...body,
        role: UserRole.TENANT, // Ensure tenant role
        // Transform date strings to Date objects
        dateOfBirth: body.dateOfBirth ? new Date(body.dateOfBirth) : undefined,
        moveInDate: body.moveInDate ? new Date(body.moveInDate) : undefined,
        applicationDate: new Date(), // Set current date as application date
        // Set default tenant status if not provided
        tenantStatus: body.tenantStatus || "application_submitted",
        // Transform employment info dates
        employmentInfo:
          body.employmentInfo && body.employmentInfo.employer
            ? {
                ...body.employmentInfo,
                startDate: body.employmentInfo.startDate
                  ? new Date(body.employmentInfo.startDate)
                  : new Date(),
              }
            : undefined,
      };

      // Validate the transformed data
      const validation = validateSchema(userSchema, tenantData);
      if (!validation.success) {
        return createErrorResponse(validation.errors.join(", "), 400);
      }

      const userData = validation.data;

      // Check if user with this email already exists
      const existingUser = await User.findOne({ email: userData.email });
      if (existingUser) {
        return createErrorResponse("User with this email already exists", 409);
      }

      // Create the tenant user
      const tenant = new User(userData);
      await tenant.save();

      return createSuccessResponse(
        tenant,
        "Tenant created successfully",
        undefined
      );
    } catch (error) {
      return handleApiError(error);
    }
  }
);

// ============================================================================
// PUT /api/tenants - Bulk update tenants (admin only)
// ============================================================================

export async function PUT(request: NextRequest) {
  try {
    // Connect to database
    await connectDB();

    // Get session
    const session = await auth();
    if (!session?.user) {
      return createErrorResponse("Unauthorized", 401);
    }

    // Check user role
    if (session.user.role !== UserRole.ADMIN) {
      return createErrorResponse("Insufficient permissions", 403);
    }

    const { success, data: body, error } = await parseRequestBody(request);
    if (!success) {
      return createErrorResponse(error!, 400);
    }

    const { tenantIds, updates } = body;

    if (!Array.isArray(tenantIds) || tenantIds.length === 0) {
      return createErrorResponse("Tenant IDs array is required", 400);
    }

    if (!updates || typeof updates !== "object") {
      return createErrorResponse("Updates object is required", 400);
    }

    // Remove fields that shouldn't be bulk updated
    const allowedUpdates = { ...updates };
    delete allowedUpdates._id;
    delete allowedUpdates.userId;
    delete allowedUpdates.createdAt;
    delete allowedUpdates.updatedAt;

    // Perform bulk update on users with tenant role
    const result = await User.updateMany(
      { _id: { $in: tenantIds }, role: UserRole.TENANT },
      { $set: allowedUpdates }
    );

    return createSuccessResponse(
      {
        matchedCount: result.matchedCount,
        modifiedCount: result.modifiedCount,
      },
      `${result.modifiedCount} tenants updated successfully`
    );
  } catch (error) {
    return handleApiError(error);
  }
}

// ============================================================================
// DELETE /api/tenants - Bulk delete tenants (admin only)
// ============================================================================

export async function DELETE(request: NextRequest) {
  try {
    // Connect to database
    await connectDB();

    // Get session
    const session = await auth();
    if (!session?.user) {
      return createErrorResponse("Unauthorized", 401);
    }

    // Check user role
    if (session.user.role !== UserRole.ADMIN) {
      return createErrorResponse("Insufficient permissions", 403);
    }

    const { searchParams } = new URL(request.url);
    const tenantIds = searchParams.get("ids")?.split(",") || [];

    if (tenantIds.length === 0) {
      return createErrorResponse("Tenant IDs are required", 400);
    }

    // Check for active leases
    const Lease = require("@/models/Lease").default;
    const activeLeases = await Lease.find({
      tenantId: { $in: tenantIds },
      status: "active",
    });

    if (activeLeases.length > 0) {
      return createErrorResponse(
        "Cannot delete tenants with active leases. Please terminate leases first.",
        409
      );
    }

    // Perform permanent deletion on users with tenant role
    const result = await User.deleteMany({
      _id: { $in: tenantIds },
      role: UserRole.TENANT,
    });

    // Also delete associated tenant profiles
    const Tenant = require("@/models/Tenant").default;
    await Tenant.deleteMany({
      userId: { $in: tenantIds },
    });

    return createSuccessResponse(
      {
        deletedCount: result.deletedCount,
      },
      `${result.deletedCount} tenants permanently deleted`
    );
  } catch (error) {
    return handleApiError(error);
  }
}
