/**
 * PropertyPro - Users API Routes
 * Handle user-related operations with role-based access control
 */

import { NextRequest } from "next/server";
import { auth } from "@/lib/auth";
import { User } from "@/models";
import { UserRole } from "@/types";
import {
  createSuccessResponse,
  createErrorResponse,
  handleApiError,
  withRoleAndDB,
} from "@/lib/api-utils";
import connectDB from "@/lib/mongodb";

// ============================================================================
// GET /api/users - Get all users with filtering and pagination
// ============================================================================

export const GET = async (request: NextRequest) => {
  try {
    // Connect to database
    await connectDB();

    // Check authentication and authorization
    const session = await auth();
    if (!session?.user) {
      return createErrorResponse("Unauthorized", 401);
    }

    // Check if user has permission to view users
    const canViewUsers = [UserRole.ADMIN, UserRole.MANAGER].includes(
      session.user.role as UserRole
    );

    if (!canViewUsers) {
      return createErrorResponse("Insufficient permissions", 403);
    }

    const { searchParams } = new URL(request.url);

    // Parse query parameters
    const page = parseInt(searchParams.get("page") || "1");
    const limit = parseInt(searchParams.get("limit") || "10");
    const search = searchParams.get("search") || "";
    const role = searchParams.get("role") || "";
    const isActive = searchParams.get("isActive");

    // Build filter query
    const filter: any = {
      // Exclude soft-deleted users
      deletedAt: null,
    };

    // Search filter
    if (search) {
      filter.$or = [
        { firstName: { $regex: search, $options: "i" } },
        { lastName: { $regex: search, $options: "i" } },
        { email: { $regex: search, $options: "i" } },
      ];
    }

    // Role filter - support multiple roles separated by comma
    if (role) {
      const roles = role
        .split(",")
        .map((r) => r.trim())
        .filter((r) => r);
      if (roles.length === 1) {
        filter.role = roles[0];
      } else if (roles.length > 1) {
        filter.role = { $in: roles };
      }
    }

    // Active status filter
    if (isActive !== null && isActive !== undefined && isActive !== "") {
      filter.isActive = isActive === "true";
    }

    // Calculate pagination
    const skip = (page - 1) * limit;

    // Get users with pagination
    const [users, total] = await Promise.all([
      User.find(filter)
        .select("-password -__v") // Exclude sensitive fields
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(limit)
        .lean(),
      User.countDocuments(filter),
    ]);

    // Calculate pagination info
    const pages = Math.ceil(total / limit);
    const hasNext = page < pages;
    const hasPrev = page > 1;

    return createSuccessResponse({
      users: users,
      pagination: {
        page,
        limit,
        total,
        pages,
        hasNext,
        hasPrev,
      },
    });
  } catch (error) {
    return createErrorResponse("Failed to fetch users", 500);
  }
};

// ============================================================================
// POST /api/users - Create a new user (Admin only)
// ============================================================================

export const POST = async (request: NextRequest) => {
  try {
    // Connect to database
    await connectDB();

    const body = await request.json();

    // Basic validation
    if (!body.email || !body.password || !body.firstName || !body.lastName) {
      return createErrorResponse("Missing required fields", 400);
    }

    // Validate role
    if (body.role && !Object.values(UserRole).includes(body.role)) {
      return createErrorResponse("Invalid role specified", 400);
    }

    // Check if user already exists
    const existingUser = await User.findOne({ email: body.email });
    if (existingUser) {
      return createErrorResponse("User with this email already exists", 400);
    }

    // Create new user
    const newUser = new User({
      firstName: body.firstName,
      lastName: body.lastName,
      email: body.email,
      password: body.password, // Will be hashed by the model
      role: body.role || UserRole.TENANT,
      phone: body.phone || undefined,
      avatar: body.avatar || undefined,
      isActive: body.isActive !== undefined ? body.isActive : true,
    });

    const savedUser = await newUser.save();

    // Remove password from response
    const userResponse = savedUser.toObject();
    delete userResponse.password;

    return createSuccessResponse(
      { data: userResponse },
      "User created successfully"
    );
  } catch (error) {
    return handleApiError(error);
  }
};

// ============================================================================
// PUT /api/users - Bulk update users (Admin only)
// ============================================================================

export const PUT = withRoleAndDB([UserRole.ADMIN], async (user, request) => {
  try {
    const body = await request.json();
    const { userIds, updates } = body;

    if (!userIds || !Array.isArray(userIds) || userIds.length === 0) {
      return createErrorResponse("User IDs are required", 400);
    }

    if (!updates || typeof updates !== "object") {
      return createErrorResponse("Updates are required", 400);
    }

    // Remove sensitive fields from updates
    delete updates.password;
    delete updates._id;
    delete updates.__v;

    // Perform bulk update
    const result = await User.updateMany(
      { _id: { $in: userIds } },
      { $set: updates }
    );

    return createSuccessResponse({
      data: {
        matchedCount: result.matchedCount,
        modifiedCount: result.modifiedCount,
      },
    });
  } catch (error) {
    return handleApiError(error);
  }
});

// ============================================================================
// DELETE /api/users - Bulk deactivate users (Admin only)
// ============================================================================

export const DELETE = withRoleAndDB([UserRole.ADMIN])(async (user, request) => {
  try {
    const { searchParams } = new URL(request.url);
    const idsParam = searchParams.get("ids");

    if (!idsParam) {
      return createErrorResponse("User IDs are required", 400);
    }

    const userIds = idsParam.split(",");

    // Deactivate users instead of deleting them
    const result = await User.updateMany(
      { _id: { $in: userIds } },
      { $set: { isActive: false } }
    );

    return createSuccessResponse({
      data: {
        matchedCount: result.matchedCount,
        modifiedCount: result.modifiedCount,
      },
    });
  } catch (error) {
    return handleApiError(error);
  }
});
