"use client";

import { useState } from "react";
import { signIn } from "next-auth/react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Alert, AlertDescription } from "@/components/ui/alert";
import {
  Loader2,
  Building2,
  Mail,
  Lock,
  AlertCircle,
  Copy,
} from "lucide-react";

export default function SignInPage() {
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState("");

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError("");

    try {
      const result = await signIn("credentials", {
        email,
        password,
        redirect: false,
      });

      if (result?.error) {
        setError("Invalid email or password");
        setIsLoading(false);
      } else if (result?.ok) {
        // Successful login - use setTimeout to allow pending requests to complete
        // This prevents the "signal is aborted without reason" error
        setTimeout(() => {
          window.location.href = "/dashboard";
        }, 100);
      }
    } catch (error) {
      console.error("Sign in error:", error);
      setError("An error occurred during sign in");
      setIsLoading(false);
    }
  };

  const handleDemoCredentials = (demoEmail: string, demoPassword: string) => {
    setEmail(demoEmail);
    setPassword(demoPassword);
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gray-50 dark:bg-gray-900 py-12 px-4 sm:px-6 lg:px-8">
      <div className="max-w-lg w-full space-y-8">
        {/* Header */}
        <div className="text-center">
          <div className="flex justify-center">
            <Building2 className="h-12 w-12 text-blue-600 dark:text-blue-400" />
          </div>
          <h2 className="mt-6 text-3xl font-extrabold text-gray-900 dark:text-gray-100">
            Sign in to PropertyPro
          </h2>
          <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
            Manage your properties with ease
          </p>
        </div>

        {/* Sign In Form */}
        <Card>
          <CardHeader>
            <CardTitle>Welcome back</CardTitle>
            <CardDescription>
              Enter your credentials to access your account
            </CardDescription>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              {error && (
                <Alert variant="destructive">
                  <AlertCircle className="h-4 w-4" />
                  <AlertDescription>{error}</AlertDescription>
                </Alert>
              )}

              <div className="space-y-2">
                <Label htmlFor="email">Email address</Label>
                <div className="relative">
                  <Mail className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-gray-400 dark:text-gray-500" />
                  <Input
                    id="email"
                    type="email"
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    placeholder="Enter your email"
                    className="pl-10"
                    required
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="password">Password</Label>
                <div className="relative">
                  <Lock className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-gray-400 dark:text-gray-500" />
                  <Input
                    id="password"
                    type="password"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    placeholder="Enter your password"
                    className="pl-10"
                    required
                  />
                </div>
              </div>
              <Button type="submit" className="w-full" disabled={isLoading}>
                {isLoading ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Signing in...
                  </>
                ) : (
                  "Sign in"
                )}
              </Button>
            </form>
          </CardContent>
        </Card>

        {/* Demo Credentials */}
        <Card className="bg-blue-50 dark:bg-blue-950/30 border-blue-200 dark:border-blue-800">
          <CardContent>
            <h3 className="text-sm font-semibold text-gray-900 dark:text-gray-100 mb-3">
              Demo account login credentials
            </h3>
            <div className="space-y-2">
              {/* Admin Credentials */}
              <div className="flex items-center justify-between bg-white dark:bg-gray-800 rounded-lg px-3 py-1 border border-gray-200 dark:border-gray-700 hover:border-blue-300 dark:hover:border-blue-600 transition-colors gap-2">
                <div className="flex items-center gap-2 sm:gap-4 flex-1 min-w-0 overflow-hidden">
                  <span className="text-xs sm:text-sm font-medium text-gray-700 dark:text-gray-300 w-16 sm:w-20 shrink-0">
                    Admin
                  </span>
                  <span className="text-xs sm:text-sm text-gray-600 dark:text-gray-400 truncate">
                    admin@propertypro.com
                  </span>
                  <span className="text-xs sm:text-sm text-gray-600 dark:text-gray-400 hidden sm:inline shrink-0">
                    Admin123$
                  </span>
                </div>
                <Button
                  type="button"
                  variant="ghost"
                  size="sm"
                  onClick={() =>
                    handleDemoCredentials("admin@propertypro.com", "Admin123$")
                  }
                  className="h-8 w-8 p-0 hover:bg-blue-100 dark:hover:bg-blue-900/30 shrink-0"
                >
                  <Copy className="h-4 w-4 text-gray-500 dark:text-gray-400" />
                </Button>
              </div>

              {/* Manager Credentials */}
              <div className="flex items-center justify-between bg-white dark:bg-gray-800 rounded-lg px-3 py-1 border border-gray-200 dark:border-gray-700 hover:border-blue-300 dark:hover:border-blue-600 transition-colors gap-2">
                <div className="flex items-center gap-2 sm:gap-4 flex-1 min-w-0 overflow-hidden">
                  <span className="text-xs sm:text-sm font-medium text-gray-700 dark:text-gray-300 w-16 sm:w-20 shrink-0">
                    Manager
                  </span>
                  <span className="text-xs sm:text-sm text-gray-600 dark:text-gray-400 truncate">
                    manager@propertypro.com
                  </span>
                  <span className="text-xs sm:text-sm text-gray-600 dark:text-gray-400 hidden sm:inline shrink-0">
                    Manager123$
                  </span>
                </div>
                <Button
                  type="button"
                  variant="ghost"
                  size="sm"
                  onClick={() =>
                    handleDemoCredentials(
                      "manager@propertypro.com",
                      "Manager123$"
                    )
                  }
                  className="h-8 w-8 p-0 hover:bg-blue-100 dark:hover:bg-blue-900/30 shrink-0"
                >
                  <Copy className="h-4 w-4 text-gray-500 dark:text-gray-400" />
                </Button>
              </div>

              {/* Tenant Credentials */}
              <div className="flex items-center justify-between bg-white dark:bg-gray-800 rounded-lg px-3 py-1 border border-gray-200 dark:border-gray-700 hover:border-blue-300 dark:hover:border-blue-600 transition-colors gap-2">
                <div className="flex items-center gap-2 sm:gap-4 flex-1 min-w-0 overflow-hidden">
                  <span className="text-xs sm:text-sm font-medium text-gray-700 dark:text-gray-300 w-16 sm:w-20 shrink-0">
                    Tenant
                  </span>
                  <span className="text-xs sm:text-sm text-gray-600 dark:text-gray-400 truncate">
                    tenant@propertypro.com
                  </span>
                  <span className="text-xs sm:text-sm text-gray-600 dark:text-gray-400 hidden sm:inline shrink-0">
                    Tenant123$
                  </span>
                </div>
                <Button
                  type="button"
                  variant="ghost"
                  size="sm"
                  onClick={() =>
                    handleDemoCredentials(
                      "tenant@propertypro.com",
                      "Tenant123$"
                    )
                  }
                  className="h-8 w-8 p-0 hover:bg-blue-100 dark:hover:bg-blue-900/30 shrink-0"
                >
                  <Copy className="h-4 w-4 text-gray-500 dark:text-gray-400" />
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
