"use client";

import { useState, useEffect, useCallback, useMemo } from "react";
import Link from "next/link";
import { useRouter } from "next/navigation";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { DeleteConfirmationDialog } from "@/components/ui/confirmation-dialog";
import {
  Shield,
  Users,
  Building2,
  DollarSign,
  Activity,
  Settings,
  MoreHorizontal,
  Plus,
  Eye,
  Edit,
  Trash2,
  UserCheck,
  UserX,
  AlertTriangle,
  CheckCircle,
  Clock,
  TrendingUp,
  Database,
  Server,
  RefreshCw,
} from "lucide-react";
import type {
  AdminDashboardResponse,
  AdminDashboardAlert,
  AdminDashboardUserSummary,
} from "@/lib/services/admin-dashboard.service";
import { UserRole } from "@/types";

const roleLabelMap: Record<UserRole, string> = {
  [UserRole.ADMIN]: "Admin",
  [UserRole.MANAGER]: "Manager",
  [UserRole.TENANT]: "Tenant",
};

const healthStatusCopy: Record<
  AdminDashboardResponse["systemStatus"]["status"],
  string
> = {
  healthy: "All systems operational",
  degraded: "Performance requires attention",
  unhealthy: "Immediate action required",
};

const alertColorMap: Record<AdminDashboardAlert["type"], string> = {
  error: "text-red-600",
  warning: "text-orange-500",
  info: "text-blue-600",
};

const alertIconMap: Record<AdminDashboardAlert["type"], typeof AlertTriangle> =
  {
    error: AlertTriangle,
    warning: AlertTriangle,
    info: CheckCircle,
  };

const formatCurrency = (amount?: number | null) => {
  if (amount === undefined || amount === null || Number.isNaN(amount)) {
    return "$0";
  }

  return new Intl.NumberFormat("en-US", {
    style: "currency",
    currency: "USD",
    minimumFractionDigits: 0,
    maximumFractionDigits: 0,
  }).format(amount);
};

const formatNumber = (value?: number | null) => {
  if (value === undefined || value === null || Number.isNaN(value)) {
    return undefined;
  }
  return value.toLocaleString("en-US");
};

const formatDate = (value?: string | null) => {
  if (!value) {
    return "No activity yet";
  }

  return new Date(value).toLocaleString("en-US", {
    year: "numeric",
    month: "short",
    day: "numeric",
    hour: "2-digit",
    minute: "2-digit",
  });
};

const getRoleVariant = (
  role: string
): "default" | "secondary" | "outline" | "destructive" => {
  switch (role) {
    case UserRole.ADMIN:
      return "destructive";
    case UserRole.MANAGER:
      return "default";
    case UserRole.TENANT:
      return "outline";
    default:
      return "outline";
  }
};

const getStatusVariant = (
  status: string
): "secondary" | "outline" | "destructive" => {
  switch (status) {
    case "active":
      return "secondary";
    case "inactive":
      return "destructive";
    case "pending":
    default:
      return "outline";
  }
};

const getAlertIcon = (type: AdminDashboardAlert["type"]) =>
  alertIconMap[type] ?? Clock;

const getAlertColor = (type: AdminDashboardAlert["type"]) =>
  alertColorMap[type] ?? "text-gray-600";

export default function AdminPage() {
  const router = useRouter();
  const [activeTab, setActiveTab] = useState("overview");
  const [dashboardData, setDashboardData] =
    useState<AdminDashboardResponse | null>(null);
  const [isLoading, setIsLoading] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [actionUserId, setActionUserId] = useState<string | null>(null);

  const loadDashboardData = useCallback(
    async ({ isRefresh = false }: { isRefresh?: boolean } = {}) => {
      if (isRefresh) {
        setIsRefreshing(true);
      } else {
        setIsLoading(true);
      }
      setError(null);

      try {
        const response = await fetch("/api/admin/dashboard");

        if (!response.ok) {
          throw new Error(`Failed to load dashboard (${response.status})`);
        }

        const result = await response.json();
        const payload = (result?.data ?? result) as AdminDashboardResponse;

        if (!payload?.stats) {
          throw new Error("Invalid dashboard response");
        }

        setDashboardData(payload);
      } catch (err) {
        const message =
          err instanceof Error ? err.message : "Unable to load dashboard";
        setError(message);
      } finally {
        if (isRefresh) {
          setIsRefreshing(false);
        } else {
          setIsLoading(false);
        }
      }
    },
    []
  );

  useEffect(() => {
    loadDashboardData();
  }, [loadDashboardData]);

  const recentUsers = useMemo(
    () => dashboardData?.recentUsers?.slice(0, 6) ?? [],
    [dashboardData]
  );

  const alerts = useMemo(
    () => dashboardData?.alerts?.slice(0, 6) ?? [],
    [dashboardData]
  );

  const services = useMemo(
    () => dashboardData?.systemStatus?.services ?? [],
    [dashboardData]
  );

  const handleRefresh = useCallback(() => {
    loadDashboardData({ isRefresh: true });
  }, [loadDashboardData]);

  const handleToggleUserStatus = useCallback(
    async (userId: string, shouldActivate: boolean) => {
      try {
        setActionUserId(userId);
        const response = await fetch(`/api/users/${userId}`, {
          method: "PUT",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({ isActive: shouldActivate }),
        });

        const result = await response.json().catch(() => ({}));

        if (!response.ok) {
          throw new Error(
            result?.error || result?.message || "Unable to update user"
          );
        }

        toast.success(
          shouldActivate
            ? "User activated successfully"
            : "User suspended successfully"
        );
        await loadDashboardData({ isRefresh: true });
      } catch (err) {
        const message =
          err instanceof Error ? err.message : "Failed to update user status";
        toast.error(message);
      } finally {
        setActionUserId(null);
      }
    },
    [loadDashboardData]
  );

  const handleDeleteUser = useCallback(
    async (userId: string) => {
      try {
        setActionUserId(userId);
        const response = await fetch(`/api/users/${userId}`, {
          method: "DELETE",
        });

        const result = await response.json().catch(() => ({}));

        if (!response.ok) {
          throw new Error(
            result?.error || result?.message || "Unable to deactivate user"
          );
        }

        toast.success("User deactivated successfully");
        await loadDashboardData({ isRefresh: true });
      } catch (err) {
        const message =
          err instanceof Error ? err.message : "Failed to deactivate user";
        toast.error(message);
      } finally {
        setActionUserId(null);
      }
    },
    [loadDashboardData]
  );

  const systemStats = dashboardData?.stats;
  const systemStatus = dashboardData?.systemStatus;
  const databaseStatus = systemStatus?.databaseStatus;
  const metrics = systemStatus?.metrics;
  const maintenanceOpen = systemStats?.maintenanceOpen ?? 0;
  const isInitialLoading = isLoading && !dashboardData;

  const renderStatValue = (value?: string) =>
    value ? (
      <div className="text-2xl font-bold">{value}</div>
    ) : (
      <Skeleton className="h-7 w-16" />
    );

  return (
    <div className="space-y-6">
      <div className="flex flex-col gap-4 md:flex-row md:items-center md:justify-between">
        <div>
          <h1 className="flex items-center gap-2 text-3xl font-bold tracking-tight">
            <Shield className="h-8 w-8 text-red-600" />
            Admin Dashboard
          </h1>
          <p className="text-muted-foreground">
            System administration and management
          </p>
        </div>
        <div className="flex flex-wrap items-center gap-2">
          <Button
            variant="ghost"
            size="icon"
            onClick={handleRefresh}
            disabled={isRefreshing || isInitialLoading}
            title="Refresh dashboard"
            aria-label="Refresh dashboard"
          >
            <RefreshCw
              className={`h-4 w-4 ${isRefreshing ? "animate-spin" : ""}`}
            />
          </Button>
          <Button variant="outline" asChild>
            <Link href="/dashboard/settings">
              <Settings className="mr-2 h-4 w-4" />
              System Settings
            </Link>
          </Button>
          <Button asChild>
            <Link href="/dashboard/admin/users/new">
              <Plus className="mr-2 h-4 w-4" />
              Add User
            </Link>
          </Button>
        </div>
      </div>

      {error && (
        <Alert variant="destructive">
          <AlertTriangle className="h-4 w-4" />
          <AlertTitle>Dashboard data unavailable</AlertTitle>
          <AlertDescription>
            {error} — try refreshing or check your database connection.
          </AlertDescription>
        </Alert>
      )}

      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-6">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Users</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {renderStatValue(formatNumber(systemStats?.totalUsers))}
            <p className="text-xs text-muted-foreground">
              {systemStats ? (
                `${formatNumber(systemStats.activeUsers)} active`
              ) : (
                <Skeleton className="h-3 w-20" />
              )}
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Properties</CardTitle>
            <Building2 className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {renderStatValue(formatNumber(systemStats?.totalProperties))}
            <p className="text-xs text-muted-foreground">
              {systemStats ? (
                `${formatNumber(systemStats.activeProperties)} active`
              ) : (
                <Skeleton className="h-3 w-20" />
              )}
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Revenue</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {renderStatValue(formatCurrency(systemStats?.totalRevenue))}
            <p className="text-xs text-muted-foreground flex items-center gap-1">
              <TrendingUp className="h-3 w-3" />
              Last 30 days:{" "}
              {systemStats
                ? formatCurrency(systemStats.revenueLast30)
                : "Loading"}
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">System Health</CardTitle>
            <Activity className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {renderStatValue(
              systemStatus ? `${systemStatus.score}%` : undefined
            )}
            <p className="text-xs text-muted-foreground">
              {systemStatus ? (
                healthStatusCopy[systemStatus.status]
              ) : (
                <Skeleton className="h-3 w-24" />
              )}
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">
              Active Sessions
            </CardTitle>
            <Server className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {renderStatValue(formatNumber(systemStats?.activeSessions))}
            <p className="text-xs text-muted-foreground">
              {systemStats ? (
                "Last 24 hours"
              ) : (
                <Skeleton className="h-3 w-16" />
              )}
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Database</CardTitle>
            <Database className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {databaseStatus ? (
              <div
                className={`text-2xl font-bold ${
                  databaseStatus.status === "online"
                    ? "text-green-600"
                    : databaseStatus.status === "degraded"
                    ? "text-orange-500"
                    : "text-red-600"
                }`}
              >
                {databaseStatus.status === "online"
                  ? "Online"
                  : databaseStatus.status === "degraded"
                  ? "Degraded"
                  : "Offline"}
              </div>
            ) : (
              <Skeleton className="h-7 w-16" />
            )}
            <p className="text-xs text-muted-foreground">
              {databaseStatus ? (
                databaseStatus.label
              ) : (
                <Skeleton className="h-3 w-24" />
              )}
            </p>
          </CardContent>
        </Card>
      </div>

      <Tabs
        value={activeTab}
        onValueChange={setActiveTab}
        className="space-y-4"
      >
        <TabsList>
          <TabsTrigger value="overview">Overview</TabsTrigger>
          <TabsTrigger value="users">User Management</TabsTrigger>
          <TabsTrigger value="system">System Monitoring</TabsTrigger>
          <TabsTrigger value="settings">Settings</TabsTrigger>
        </TabsList>

        <TabsContent value="overview" className="space-y-4">
          <div className="grid gap-4 md:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle>Recent User Activity</CardTitle>
                <CardDescription>
                  Latest user registrations and logins
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {isInitialLoading && (
                    <>
                      {Array.from({ length: 3 }).map((_, idx) => (
                        <div
                          key={idx}
                          className="flex items-center justify-between"
                        >
                          <div className="flex items-center space-x-3">
                            <Skeleton className="h-8 w-8 rounded-full" />
                            <div className="space-y-1">
                              <Skeleton className="h-4 w-32" />
                              <Skeleton className="h-3 w-40" />
                            </div>
                          </div>
                          <Skeleton className="h-3 w-20" />
                        </div>
                      ))}
                    </>
                  )}

                  {!isInitialLoading && recentUsers.length === 0 && (
                    <p className="text-sm text-muted-foreground">
                      No recent user activity recorded.
                    </p>
                  )}

                  {recentUsers.map((user) => (
                    <div
                      key={user.id}
                      className="flex items-center justify-between"
                    >
                      <div className="flex items-center space-x-3">
                        <div className="flex h-8 w-8 items-center justify-center rounded-full bg-blue-100">
                          <Users className="h-4 w-4 text-blue-600" />
                        </div>
                        <div>
                          <p className="text-sm font-medium">{user.name}</p>
                          <p className="text-xs text-muted-foreground">
                            {user.email}
                          </p>
                          <p className="text-xs text-muted-foreground">
                            {user.properties} properties linked
                          </p>
                        </div>
                      </div>
                      <div className="text-right space-y-1">
                        <Badge
                          variant={getRoleVariant(user.role)}
                          className="capitalize"
                        >
                          {roleLabelMap[user.role as UserRole] ?? user.role}
                        </Badge>
                        <p className="text-xs text-muted-foreground">
                          {formatDate(user.lastLogin)}
                        </p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>System Alerts</CardTitle>
                <CardDescription>
                  Recent system notifications and alerts
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {isInitialLoading && (
                    <>
                      {Array.from({ length: 3 }).map((_, idx) => (
                        <div key={idx} className="flex items-start space-x-3">
                          <Skeleton className="h-4 w-4" />
                          <div className="space-y-2">
                            <Skeleton className="h-4 w-40" />
                            <Skeleton className="h-3 w-28" />
                          </div>
                        </div>
                      ))}
                    </>
                  )}

                  {!isInitialLoading && alerts.length === 0 && (
                    <p className="text-sm text-muted-foreground">
                      No critical alerts at this time.
                    </p>
                  )}

                  {alerts.map((alert) => {
                    const Icon = getAlertIcon(alert.type);
                    return (
                      <div
                        key={alert.id}
                        className="flex items-start space-x-3"
                      >
                        <Icon
                          className={`h-4 w-4 mt-0.5 ${getAlertColor(
                            alert.type
                          )}`}
                        />
                        <div className="flex-1">
                          <p className="text-sm">{alert.message}</p>
                          <p className="text-xs text-muted-foreground">
                            {formatDate(alert.timestamp)} • {alert.source}
                          </p>
                        </div>
                      </div>
                    );
                  })}
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="users" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>User Management</CardTitle>
              <CardDescription>
                Manage system users and their permissions
              </CardDescription>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>User</TableHead>
                    <TableHead>Role</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Properties</TableHead>
                    <TableHead>Last Activity</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {isInitialLoading && (
                    <>
                      {Array.from({ length: 4 }).map((_, idx) => (
                        <TableRow key={idx}>
                          <TableCell>
                            <Skeleton className="h-4 w-40" />
                            <Skeleton className="mt-1 h-3 w-32" />
                          </TableCell>
                          <TableCell>
                            <Skeleton className="h-6 w-16" />
                          </TableCell>
                          <TableCell>
                            <Skeleton className="h-6 w-16" />
                          </TableCell>
                          <TableCell>
                            <Skeleton className="h-4 w-10" />
                          </TableCell>
                          <TableCell>
                            <Skeleton className="h-4 w-24" />
                          </TableCell>
                          <TableCell className="text-right">
                            <Skeleton className="h-8 w-8 rounded-md" />
                          </TableCell>
                        </TableRow>
                      ))}
                    </>
                  )}

                  {!isInitialLoading && recentUsers.length === 0 && (
                    <TableRow>
                      <TableCell
                        colSpan={6}
                        className="text-center text-sm text-muted-foreground"
                      >
                        No user data available yet.
                      </TableCell>
                    </TableRow>
                  )}

                  {recentUsers.map((user: AdminDashboardUserSummary) => (
                    <TableRow key={user.id}>
                      <TableCell>
                        <div>
                          <div className="font-medium">{user.name}</div>
                          <div className="text-sm text-muted-foreground">
                            {user.email}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge
                          variant={getRoleVariant(user.role)}
                          className="capitalize"
                        >
                          {roleLabelMap[user.role as UserRole] ?? user.role}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <Badge
                          variant={getStatusVariant(user.status)}
                          className="capitalize"
                        >
                          {user.status}
                        </Badge>
                      </TableCell>
                      <TableCell>{user.properties}</TableCell>
                      <TableCell>{formatDate(user.lastLogin)}</TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button
                              variant="ghost"
                              className="h-8 w-8 p-0"
                              aria-label="Open user actions"
                            >
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem
                              onSelect={(event) => {
                                event.preventDefault();
                                router.push(
                                  `/dashboard/admin/users/${user.id}`
                                );
                              }}
                            >
                              <Eye className="mr-2 h-4 w-4" />
                              View Details
                            </DropdownMenuItem>
                            <DropdownMenuItem
                              onSelect={(event) => {
                                event.preventDefault();
                                router.push(
                                  `/dashboard/admin/users/${user.id}/edit`
                                );
                              }}
                            >
                              <Edit className="mr-2 h-4 w-4" />
                              Edit User
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            {user.status === "active" ? (
                              <DropdownMenuItem
                                disabled={actionUserId === user.id}
                                onSelect={(event) => {
                                  event.preventDefault();
                                  handleToggleUserStatus(user.id, false);
                                }}
                                className="text-orange-600"
                              >
                                <UserX className="mr-2 h-4 w-4" />
                                Suspend User
                              </DropdownMenuItem>
                            ) : (
                              <DropdownMenuItem
                                disabled={actionUserId === user.id}
                                onSelect={(event) => {
                                  event.preventDefault();
                                  handleToggleUserStatus(user.id, true);
                                }}
                                className="text-green-600"
                              >
                                <UserCheck className="mr-2 h-4 w-4" />
                                Activate User
                              </DropdownMenuItem>
                            )}
                            <DeleteConfirmationDialog
                              itemName={user.name}
                              itemType="user"
                              onConfirm={() => handleDeleteUser(user.id)}
                              loading={actionUserId === user.id}
                            >
                              <DropdownMenuItem
                                onSelect={(event) => event.preventDefault()}
                                className="text-red-600"
                              >
                                <Trash2 className="mr-2 h-4 w-4" />
                                Deactivate User
                              </DropdownMenuItem>
                            </DeleteConfirmationDialog>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="system" className="space-y-4">
          <div className="grid gap-4 md:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle>System Performance</CardTitle>
                <CardDescription>Real-time system metrics</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-medium">CPU Usage</span>
                    <span className="text-muted-foreground">
                      {metrics ? `${Math.round(metrics.cpuUsage ?? 0)}%` : "--"}
                    </span>
                  </div>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-medium">Memory Usage</span>
                    <span className="text-muted-foreground">
                      {metrics
                        ? `${Math.round(metrics.memoryUsage?.percentage ?? 0)}%`
                        : "--"}
                    </span>
                  </div>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-medium">Active Connections</span>
                    <span className="text-muted-foreground">
                      {metrics
                        ? formatNumber(metrics.activeConnections) ?? "0"
                        : "--"}
                    </span>
                  </div>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-medium">Avg Response Time</span>
                    <span className="text-muted-foreground">
                      {metrics
                        ? `${Math.round(metrics.averageResponseTime ?? 0)} ms`
                        : "--"}
                    </span>
                  </div>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-medium">Requests / Min</span>
                    <span className="text-muted-foreground">
                      {metrics
                        ? formatNumber(
                            Math.round(metrics.requestsPerMinute ?? 0)
                          ) ?? "0"
                        : "--"}
                    </span>
                  </div>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-medium">Error Rate</span>
                    <span className="text-muted-foreground">
                      {metrics
                        ? `${Math.round((metrics.errorRate ?? 0) * 1000) / 10}%`
                        : "--"}
                    </span>
                  </div>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-medium">System Uptime</span>
                    <span className="text-muted-foreground">
                      {systemStatus?.uptime ?? "--"}
                    </span>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Service Status</CardTitle>
                <CardDescription>
                  Service availability and health
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {isInitialLoading && (
                    <>
                      {Array.from({ length: 4 }).map((_, idx) => (
                        <div
                          key={idx}
                          className="flex items-center justify-between"
                        >
                          <Skeleton className="h-4 w-24" />
                          <Skeleton className="h-6 w-16" />
                        </div>
                      ))}
                    </>
                  )}

                  {!isInitialLoading && services.length === 0 && (
                    <p className="text-sm text-muted-foreground">
                      No service checks recorded yet.
                    </p>
                  )}

                  {services.map((service) => (
                    <div
                      key={service.service}
                      className="flex items-center justify-between text-sm"
                    >
                      <span className="font-medium">{service.displayName}</span>
                      <Badge
                        variant={
                          service.status === "healthy"
                            ? "secondary"
                            : service.status === "degraded"
                            ? "outline"
                            : "destructive"
                        }
                        className="capitalize"
                      >
                        {service.status}
                      </Badge>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="settings" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>System Snapshot</CardTitle>
              <CardDescription>
                High-level configuration and operational summary
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid gap-4 md:grid-cols-2">
                <div className="rounded-lg border p-4">
                  <p className="text-sm font-medium">Open Maintenance Items</p>
                  <p className="text-2xl font-bold">
                    {formatNumber(maintenanceOpen) ?? "--"}
                  </p>
                  <p className="text-xs text-muted-foreground">
                    Requests awaiting completion
                  </p>
                </div>
                <div className="rounded-lg border p-4">
                  <p className="text-sm font-medium">Database Status</p>
                  <p className="text-2xl font-bold">
                    {databaseStatus
                      ? databaseStatus.status === "online"
                        ? "Connected"
                        : databaseStatus.status === "degraded"
                        ? "Degraded"
                        : "Offline"
                      : "--"}
                  </p>
                  <p className="text-xs text-muted-foreground">
                    {databaseStatus?.label ?? "Awaiting health check"}
                  </p>
                </div>
                <div className="rounded-lg border p-4">
                  <p className="text-sm font-medium">Total Revenue (YTD)</p>
                  <p className="text-2xl font-bold">
                    {formatCurrency(systemStats?.totalRevenue)}
                  </p>
                  <p className="text-xs text-muted-foreground">
                    Includes completed and paid transactions
                  </p>
                </div>
                <div className="rounded-lg border p-4">
                  <p className="text-sm font-medium">Active User Accounts</p>
                  <p className="text-2xl font-bold">
                    {formatNumber(systemStats?.activeUsers) ?? "--"}
                  </p>
                  <p className="text-xs text-muted-foreground">
                    Active users across the organisation
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
