"use client";

import React, { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { useRouter } from "next/navigation";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Checkbox } from "@/components/ui/checkbox";

import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { DeleteConfirmationDialog } from "@/components/ui/confirmation-dialog";
import {
  Users,
  UserPlus,
  Search,
  Filter,
  MoreHorizontal,
  Eye,
  Edit,
  Trash2,
  Grid3X3,
  List,
  RefreshCw,
  Shield,
  Mail,
  Phone,
  Calendar,
  CheckCircle,
  XCircle,
} from "lucide-react";
import { UserRole, IUser } from "@/types";
import { formatDate } from "@/lib/utils";
import { BulkOperations } from "@/components/users/bulk-operations";
import { RoleBadge } from "@/components/ui/role-badge";

interface UserListPageProps {}

interface UserStats {
  total: number;
  active: number;
  inactive: number;
  byRole: Record<UserRole, number>;
}

export default function UserListPage({}: UserListPageProps) {
  const { data: session } = useSession();
  const router = useRouter();

  // State management
  const [users, setUsers] = useState<IUser[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [roleFilter, setRoleFilter] = useState<string>("all");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [viewMode, setViewMode] = useState<"table" | "grid">("table");
  const [selectedUsers, setSelectedUsers] = useState<string[]>([]);
  const [showDeleteDialog, setShowDeleteDialog] = useState(false);
  const [userToDelete, setUserToDelete] = useState<string | null>(null);
  const [isDeleting, setIsDeleting] = useState(false);
  const [showBulkDialog, setShowBulkDialog] = useState(false);
  const [stats, setStats] = useState<UserStats>({
    total: 0,
    active: 0,
    inactive: 0,
    byRole: {
      [UserRole.ADMIN]: 0,
      [UserRole.MANAGER]: 0,
      [UserRole.TENANT]: 0,
    },
  });

  // Pagination
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage, setItemsPerPage] = useState(10);
  const [totalPages, setTotalPages] = useState(1);
  const [totalUsers, setTotalUsers] = useState(0);

  // Check permissions for single company architecture
  const canManageUsers = session?.user?.role === UserRole.ADMIN;
  const canViewUsers = [UserRole.ADMIN, UserRole.MANAGER].includes(
    session?.user?.role as UserRole
  );

  // Fetch users
  const fetchUsers = async () => {
    try {
      setIsLoading(true);

      // Build query parameters
      const params = new URLSearchParams();
      params.append("page", currentPage.toString());
      params.append("limit", itemsPerPage.toString());

      if (searchTerm) {
        params.append("search", searchTerm);
      }

      // Exclude tenants from Users list by default
      if (roleFilter && roleFilter !== "all") {
        // If somehow 'tenant' is selected, fall back to non-tenant roles
        if (roleFilter === UserRole.TENANT) {
          params.append("role", [UserRole.ADMIN, UserRole.MANAGER].join(","));
        } else {
          params.append("role", roleFilter);
        }
      } else {
        params.append("role", [UserRole.ADMIN, UserRole.MANAGER].join(","));
      }

      if (statusFilter && statusFilter !== "all") {
        params.append("isActive", statusFilter === "active" ? "true" : "false");
      }

      const response = await fetch(`/api/users?${params}`);

      if (!response.ok) {
        throw new Error("Failed to fetch users");
      }

      const data = await response.json();
      // Handle both response formats: direct data or nested data
      const responseData = data.data || data;
      const usersArray = (responseData.users || []) as IUser[];
      const pagination = responseData.pagination || {};
      // No need to filter since API already excludes tenants
      setUsers(usersArray);
      setTotalUsers(pagination.total || usersArray.length);
      setTotalPages(pagination.pages || 1);

      // Fetch all users for stats calculation (without pagination)
      const statsParams = new URLSearchParams();
      if (searchTerm) {
        statsParams.append("search", searchTerm);
      }
      if (roleFilter && roleFilter !== "all") {
        if (roleFilter === UserRole.TENANT) {
          statsParams.append(
            "role",
            [UserRole.ADMIN, UserRole.MANAGER].join(",")
          );
        } else {
          statsParams.append("role", roleFilter);
        }
      } else {
        statsParams.append(
          "role",
          [UserRole.ADMIN, UserRole.MANAGER].join(",")
        );
      }
      if (statusFilter && statusFilter !== "all") {
        statsParams.append(
          "isActive",
          statusFilter === "active" ? "true" : "false"
        );
      }
      // Get all users for stats (set a high limit)
      statsParams.append("limit", "10000");
      statsParams.append("page", "1");

      const statsResponse = await fetch(`/api/users?${statsParams}`);
      if (statsResponse.ok) {
        const statsData = await statsResponse.json();
        // Handle both response formats: direct data or nested data
        const statsResponseData = statsData.data || statsData;
        const allUsers = statsResponseData.users || [];

        // Calculate stats from all users (API already filters out tenants)
        const safeUsers = Array.isArray(allUsers) ? allUsers : [];
        const newStats: UserStats = {
          total: safeUsers.length,
          active: safeUsers.filter((u: IUser) => u.isActive).length,
          inactive: safeUsers.filter((u: IUser) => !u.isActive).length,
          byRole: {
            [UserRole.ADMIN]: safeUsers.filter(
              (u: IUser) => u.role === UserRole.ADMIN
            ).length,
            [UserRole.MANAGER]: safeUsers.filter(
              (u: IUser) => u.role === UserRole.MANAGER
            ).length,
            [UserRole.TENANT]: 0, // Always 0 for users page
          },
        };
        setStats(newStats);
      }
    } catch (error) {
      toast.error("Failed to load users");
    } finally {
      setIsLoading(false);
    }
  };

  // Effects
  useEffect(() => {
    if (canViewUsers) {
      fetchUsers();
    }
  }, [
    currentPage,
    itemsPerPage,
    searchTerm,
    roleFilter,
    statusFilter,
    canViewUsers,
  ]);

  // Handle user deletion
  const handleDeleteUser = async () => {
    if (!userToDelete) return;

    try {
      setIsDeleting(true);
      const response = await fetch(`/api/users/${userToDelete}`, {
        method: "DELETE",
      });

      if (!response.ok) {
        throw new Error("Failed to delete user");
      }

      toast.success("User deactivated successfully");
      fetchUsers(); // Refresh the list
    } catch (error) {
      toast.error("Failed to deactivate user");
    } finally {
      setIsDeleting(false);
      setUserToDelete(null);
    }
  };

  // Handle bulk operations
  const handleBulkDelete = async () => {
    try {
      const response = await fetch(
        `/api/users?ids=${selectedUsers.join(",")}`,
        {
          method: "DELETE",
        }
      );

      if (!response.ok) {
        throw new Error("Failed to delete users");
      }

      toast.success(`${selectedUsers.length} users deactivated successfully`);
      setSelectedUsers([]);
      fetchUsers();
    } catch (error) {
      toast.error("Failed to deactivate users");
    }
  };

  // Access control check
  if (!canViewUsers) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="text-center">
          <Shield className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
          <h3 className="text-lg font-semibold">Access Denied</h3>
          <p className="text-muted-foreground">
            You don&apos;t have permission to view user management.
          </p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">User Management</h1>
          <p className="text-muted-foreground">
            Manage system users and their permissions
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Button
            variant="outline"
            size="sm"
            onClick={() => fetchUsers()}
            disabled={isLoading}
          >
            <RefreshCw
              className={`h-4 w-4 mr-2 ${isLoading ? "animate-spin" : ""}`}
            />
            Refresh
          </Button>
          {canManageUsers && (
            <>
              {selectedUsers.length > 0 && (
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setShowBulkDialog(true)}
                >
                  <Users className="h-4 w-4 mr-2" />
                  Bulk Actions ({selectedUsers.length})
                </Button>
              )}
              <Button
                variant="outline"
                onClick={() => router.push("/dashboard/admin/users/roles")}
              >
                <Shield className="h-4 w-4 mr-2" />
                Manage Roles
              </Button>
              <Button onClick={() => router.push("/dashboard/admin/users/new")}>
                <UserPlus className="h-4 w-4 mr-2" />
                Add User
              </Button>
            </>
          )}
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <Card className="gap-2">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Users</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.total}</div>
            <p className="text-xs text-muted-foreground">
              All registered users
            </p>
          </CardContent>
        </Card>
        <Card className="gap-2">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Active Users</CardTitle>
            <CheckCircle className="h-4 w-4 text-green-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {stats.active}
            </div>
            <p className="text-xs text-muted-foreground">Currently active</p>
          </CardContent>
        </Card>
        <Card className="gap-2">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">
              Inactive Users
            </CardTitle>
            <XCircle className="h-4 w-4 text-red-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">
              {stats.inactive}
            </div>
            <p className="text-xs text-muted-foreground">Deactivated users</p>
          </CardContent>
        </Card>
        <Card className="gap-2">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Admins</CardTitle>
            <Shield className="h-4 w-4 text-purple-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-purple-600">
              {stats.byRole[UserRole.ADMIN] +
                stats.byRole[UserRole.MANAGER]}
            </div>
            <p className="text-xs text-muted-foreground">Admin users</p>
          </CardContent>
        </Card>
      </div>

      {/* Filters and Search */}
      <Card className="gap-2">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Filter className="h-5 w-5" />
            Filters & Search
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                <Input
                  placeholder="Search users by name or email..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>
            <Select value={roleFilter} onValueChange={setRoleFilter}>
              <SelectTrigger className="w-full md:w-48">
                <SelectValue placeholder="Filter by role" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Roles</SelectItem>
                <SelectItem value={UserRole.ADMIN}>
                  Super Admin
                </SelectItem>
                <SelectItem value={UserRole.MANAGER}>
                  Property Manager
                </SelectItem>
                <SelectItem value={UserRole.MANAGER}>Owner</SelectItem>
                <SelectItem value={UserRole.MANAGER}>
                  Leasing Agent
                </SelectItem>
                <SelectItem value={UserRole.MANAGER}>
                  Maintenance Staff
                </SelectItem>
              </SelectContent>
            </Select>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-full md:w-48">
                <SelectValue placeholder="Filter by status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Status</SelectItem>
                <SelectItem value="active">Active</SelectItem>
                <SelectItem value="inactive">Inactive</SelectItem>
              </SelectContent>
            </Select>
            <div className="flex items-center gap-2">
              <Button
                variant={viewMode === "table" ? "default" : "outline"}
                size="sm"
                onClick={() => setViewMode("table")}
              >
                <List className="h-4 w-4" />
              </Button>
              <Button
                variant={viewMode === "grid" ? "default" : "outline"}
                size="sm"
                onClick={() => setViewMode("grid")}
              >
                <Grid3X3 className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Bulk Actions */}
      {selectedUsers.length > 0 && canManageUsers && (
        <Card>
          <CardContent className="pt-6">
            <div className="flex items-center justify-between">
              <p className="text-sm text-muted-foreground">
                {selectedUsers.length} user(s) selected
              </p>
              <div className="flex items-center gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setSelectedUsers([])}
                >
                  Clear Selection
                </Button>
                <Button
                  variant="destructive"
                  size="sm"
                  onClick={handleBulkDelete}
                >
                  <Trash2 className="h-4 w-4 mr-2" />
                  Deactivate Selected
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* User List */}
      <Card className="gap-2">
        <CardHeader>
          <CardTitle>Users ({totalUsers})</CardTitle>
          <CardDescription>
            Showing {(users || []).length} users on page {currentPage} of{" "}
            {totalPages}
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-4">
              {[...Array(5)].map((_, i) => (
                <div
                  key={i}
                  className="h-16 bg-muted rounded-lg animate-pulse"
                />
              ))}
            </div>
          ) : (users || []).length === 0 ? (
            <div className="text-center py-8">
              <Users className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-lg font-semibold mb-2">No users found</h3>
              <p className="text-muted-foreground mb-4">
                {(users || []).length === 0
                  ? "No users have been created yet."
                  : "No users match your current filters."}
              </p>

              {canManageUsers && (users || []).length === 0 && (
                <Button
                  onClick={() => router.push("/dashboard/admin/users/new")}
                >
                  <UserPlus className="h-4 w-4 mr-2" />
                  Add First User
                </Button>
              )}
            </div>
          ) : viewMode === "table" ? (
            <div className="space-y-4">
              <Table>
                <TableHeader className="bg-muted/100">
                  <TableRow>
                    {canManageUsers && (
                      <TableHead className="w-12">
                        <Checkbox
                          checked={
                            selectedUsers.length === (users || []).length
                          }
                          onCheckedChange={(checked) => {
                            if (checked) {
                              setSelectedUsers(
                                (users || []).map((u: IUser) =>
                                  u._id.toString()
                                )
                              );
                            } else {
                              setSelectedUsers([]);
                            }
                          }}
                        />
                      </TableHead>
                    )}
                    <TableHead className="w-16">#</TableHead>
                    <TableHead>User</TableHead>
                    <TableHead>Role</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Contact</TableHead>
                    <TableHead>Created</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {(users || []).map((user: IUser) => (
                    <TableRow key={user._id.toString()}>
                      {canManageUsers && (
                        <TableCell>
                          <Checkbox
                            checked={selectedUsers.includes(
                              user._id.toString()
                            )}
                            onCheckedChange={(checked) => {
                              if (checked) {
                                setSelectedUsers([
                                  ...selectedUsers,
                                  user._id.toString(),
                                ]);
                              } else {
                                setSelectedUsers(
                                  selectedUsers.filter(
                                    (id) => id !== user._id.toString()
                                  )
                                );
                              }
                            }}
                          />
                        </TableCell>
                      )}
                      <TableCell className="text-center text-sm text-muted-foreground">
                        {(currentPage - 1) * itemsPerPage +
                          (users || []).indexOf(user) +
                          1}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-3">
                          <Avatar className="h-8 w-8">
                            <AvatarImage src={user.avatar || ""} />
                            <AvatarFallback>
                              {user.firstName[0]}
                              {user.lastName[0]}
                            </AvatarFallback>
                          </Avatar>
                          <div>
                            <div className="font-medium">
                              {user.firstName} {user.lastName}
                            </div>
                            <div className="text-sm text-muted-foreground">
                              {user.email}
                            </div>
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <RoleBadge role={user.role} />
                      </TableCell>
                      <TableCell>
                        <Badge
                          variant={user.isActive ? "default" : "secondary"}
                        >
                          {user.isActive ? "Active" : "Inactive"}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="space-y-1">
                          {user.phone && (
                            <div className="flex items-center text-sm text-muted-foreground">
                              <Phone className="h-3 w-3 mr-1" />
                              {user.phone}
                            </div>
                          )}
                          <div className="flex items-center text-sm text-muted-foreground">
                            <Mail className="h-3 w-3 mr-1" />
                            {user.email}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center text-sm text-muted-foreground">
                          <Calendar className="h-3 w-3 mr-1" />
                          {formatDate(user.createdAt)}
                        </div>
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem
                              onClick={() =>
                                router.push(
                                  `/dashboard/admin/users/${user._id}`
                                )
                              }
                            >
                              <Eye className="mr-2 h-4 w-4" />
                              View Details
                            </DropdownMenuItem>
                            {canManageUsers && (
                              <>
                                <DropdownMenuItem
                                  onClick={() =>
                                    router.push(
                                      `/dashboard/admin/users/${user._id}/edit`
                                    )
                                  }
                                >
                                  <Edit className="mr-2 h-4 w-4" />
                                  Edit User
                                </DropdownMenuItem>
                                <DropdownMenuSeparator />
                                <DeleteConfirmationDialog
                                  itemName={`${user.firstName} ${user.lastName}`}
                                  itemType="user"
                                  onConfirm={handleDeleteUser}
                                  loading={isDeleting}
                                >
                                  <DropdownMenuItem
                                    onSelect={(e) => {
                                      e.preventDefault();
                                      setUserToDelete(user._id.toString());
                                    }}
                                    className="text-red-600"
                                  >
                                    <Trash2 className="mr-2 h-4 w-4" />
                                    Deactivate
                                  </DropdownMenuItem>
                                </DeleteConfirmationDialog>
                              </>
                            )}
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>

              {/* Pagination */}
              {totalPages > 1 && (
                <div className="flex items-center justify-between">
                  <p className="text-sm text-muted-foreground">
                    Showing {(currentPage - 1) * itemsPerPage + 1} to{" "}
                    {Math.min(currentPage * itemsPerPage, totalUsers)} of{" "}
                    {totalUsers} users
                  </p>
                  <div className="flex items-center space-x-2">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setCurrentPage(currentPage - 1)}
                      disabled={currentPage === 1}
                    >
                      Previous
                    </Button>
                    <div className="flex items-center space-x-1">
                      {Array.from(
                        { length: Math.min(5, totalPages) },
                        (_, i) => {
                          let pageNum;
                          if (totalPages <= 5) {
                            pageNum = i + 1;
                          } else if (currentPage <= 3) {
                            pageNum = i + 1;
                          } else if (currentPage >= totalPages - 2) {
                            pageNum = totalPages - 4 + i;
                          } else {
                            pageNum = currentPage - 2 + i;
                          }

                          return (
                            <Button
                              key={pageNum}
                              variant={
                                currentPage === pageNum ? "default" : "outline"
                              }
                              size="sm"
                              onClick={() => setCurrentPage(pageNum)}
                              className="w-8 h-8 p-0"
                            >
                              {pageNum}
                            </Button>
                          );
                        }
                      )}
                    </div>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => setCurrentPage(currentPage + 1)}
                      disabled={currentPage === totalPages}
                    >
                      Next
                    </Button>
                  </div>
                  {/* Items per page selector */}
                  <div className="flex items-center space-x-2">
                    <span className="text-sm text-muted-foreground">
                      Items per page:
                    </span>
                    <Select
                      value={itemsPerPage.toString()}
                      onValueChange={(value) => {
                        setItemsPerPage(parseInt(value));
                        setCurrentPage(1); // Reset to first page
                      }}
                    >
                      <SelectTrigger className="w-20">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="10">10</SelectItem>
                        <SelectItem value="25">25</SelectItem>
                        <SelectItem value="50">50</SelectItem>
                        <SelectItem value="100">100</SelectItem>
                        {/* Super admin can view all */}
                        {session?.user?.role === UserRole.ADMIN && (
                          <SelectItem value="1000">All</SelectItem>
                        )}
                      </SelectContent>
                    </Select>
                  </div>
                </div>
              )}
            </div>
          ) : (
            // Grid view placeholder
            <div className="text-center py-8">
              <p className="text-muted-foreground">Grid view coming soon...</p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={showDeleteDialog} onOpenChange={setShowDeleteDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Deactivate User</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to deactivate this user? This action will
              disable their access to the system but preserve their data.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => {
                if (userToDelete) {
                  handleDeleteUser();
                  setShowDeleteDialog(false);
                }
              }}
              disabled={isDeleting}
              className="bg-red-600 hover:bg-red-700"
            >
              {isDeleting ? "Deactivating..." : "Deactivate"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Bulk Operations Dialog */}
      <BulkOperations
        selectedUsers={(users || []).filter((user) =>
          selectedUsers.includes(user._id.toString())
        )}
        isOpen={showBulkDialog}
        onClose={() => setShowBulkDialog(false)}
        onSuccess={() => {
          setSelectedUsers([]);
          fetchUsers();
        }}
      />
    </div>
  );
}
