"use client";

import { toast } from "sonner";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import React, { useState, useEffect } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  CustomModal,
  CustomModalHeader,
  CustomModalTitle,
  CustomModalDescription,
  CustomModalBody,
  CustomModalFooter,
} from "@/components/ui/custom-modal";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { DeleteConfirmationDialog } from "@/components/ui/confirmation-dialog";
import {
  ArrowLeft,
  Shield,
  Users,
  Plus,
  Trash2,
  Settings,
  UserCheck,
  Search,
  RefreshCw,
  Eye,
  Lock,
  Unlock,
} from "lucide-react";
import { UserRole, IUser, IRoleConfig } from "@/types";

interface Permission {
  id: string;
  name: string;
  description: string;
  category: string;
}

interface RoleConfig {
  role: string;
  name: string;
  label: string;
  description: string;
  permissions: string[];
  userCount: number;
  color: string;
  isSystem: boolean;
  canEdit: boolean;
  canDelete: boolean;
}

interface UserWithRole {
  _id: string;
  firstName: string;
  lastName: string;
  email: string;
  role: UserRole;
  originalRole: string;
  normalizedOriginalRole: string;
  isActive: boolean;
  lastLogin?: string;
}

// Available permissions in the system
const AVAILABLE_PERMISSIONS: Permission[] = [
  // User Management
  {
    id: "user_management",
    name: "User Management",
    description: "Create, edit, and delete users",
    category: "User Management",
  },
  {
    id: "user_view",
    name: "View Users",
    description: "View user profiles and information",
    category: "User Management",
  },
  {
    id: "role_management",
    name: "Role Management",
    description: "Manage roles and permissions",
    category: "User Management",
  },

  // Property Management
  {
    id: "property_management",
    name: "Property Management",
    description: "Full property CRUD operations",
    category: "Property Management",
  },
  {
    id: "property_view",
    name: "View Properties",
    description: "View property information",
    category: "Property Management",
  },
  {
    id: "property_create",
    name: "Create Properties",
    description: "Add new properties",
    category: "Property Management",
  },
  {
    id: "property_edit",
    name: "Edit Properties",
    description: "Modify property details",
    category: "Property Management",
  },
  {
    id: "property_delete",
    name: "Delete Properties",
    description: "Remove properties",
    category: "Property Management",
  },

  // Tenant Management
  {
    id: "tenant_management",
    name: "Tenant Management",
    description: "Full tenant CRUD operations",
    category: "Tenant Management",
  },
  {
    id: "tenant_view",
    name: "View Tenants",
    description: "View tenant information",
    category: "Tenant Management",
  },
  {
    id: "tenant_create",
    name: "Create Tenants",
    description: "Add new tenants",
    category: "Tenant Management",
  },
  {
    id: "tenant_edit",
    name: "Edit Tenants",
    description: "Modify tenant details",
    category: "Tenant Management",
  },

  // Lease Management
  {
    id: "lease_management",
    name: "Lease Management",
    description: "Full lease CRUD operations",
    category: "Lease Management",
  },
  {
    id: "lease_view",
    name: "View Leases",
    description: "View lease information",
    category: "Lease Management",
  },
  {
    id: "lease_create",
    name: "Create Leases",
    description: "Create new leases",
    category: "Lease Management",
  },
  {
    id: "lease_edit",
    name: "Edit Leases",
    description: "Modify lease terms",
    category: "Lease Management",
  },

  // Maintenance Management
  {
    id: "maintenance_management",
    name: "Maintenance Management",
    description: "Full maintenance operations",
    category: "Maintenance",
  },
  {
    id: "maintenance_assign",
    name: "Assign Maintenance",
    description: "Assign maintenance work to staff",
    category: "Maintenance",
  },
  {
    id: "maintenance_view",
    name: "View Maintenance",
    description: "View maintenance requests",
    category: "Maintenance",
  },
  {
    id: "maintenance_create",
    name: "Create Maintenance",
    description: "Create maintenance requests",
    category: "Maintenance",
  },
  {
    id: "maintenance_requests",
    name: "Submit Maintenance Requests",
    description: "Submit maintenance issues",
    category: "Maintenance",
  },
  {
    id: "work_orders",
    name: "Work Orders",
    description: "Manage work orders",
    category: "Maintenance",
  },
  {
    id: "maintenance_history",
    name: "Maintenance History",
    description: "View historical maintenance activity",
    category: "Maintenance",
  },

  // Financial Management
  {
    id: "financial_management",
    name: "Financial Management",
    description: "Full financial operations",
    category: "Financial",
  },
  {
    id: "financial_reports",
    name: "Financial Reports",
    description: "View financial reports",
    category: "Financial",
  },
  {
    id: "payment_processing",
    name: "Payment Processing",
    description: "Process payments",
    category: "Financial",
  },
  {
    id: "payment_portal",
    name: "Payment Portal",
    description: "Access payment portal",
    category: "Financial",
  },
  {
    id: "payment_history",
    name: "Payment History",
    description: "View payment history",
    category: "Financial",
  },

  // System Administration
  {
    id: "system_settings",
    name: "System Settings",
    description: "Configure system settings",
    category: "System",
  },
  {
    id: "audit_logs",
    name: "Audit Logs",
    description: "View system audit logs",
    category: "System",
  },
  {
    id: "backup_restore",
    name: "Backup & Restore",
    description: "Backup and restore data",
    category: "System",
  },
  {
    id: "bulk_operations",
    name: "Bulk Operations",
    description: "Perform bulk operations",
    category: "System",
  },
  {
    id: "company_settings",
    name: "Company Settings",
    description: "Manage organization-wide settings",
    category: "System",
  },
  {
    id: "data_export",
    name: "Data Export",
    description: "Export system data",
    category: "System",
  },

  // Reports and Analytics
  {
    id: "reports_all",
    name: "All Reports",
    description: "Access all system reports",
    category: "Reports",
  },
  {
    id: "reports_property",
    name: "Property Reports",
    description: "Property-specific reports",
    category: "Reports",
  },
  {
    id: "reports_own",
    name: "Own Reports",
    description: "Personal/owned property reports",
    category: "Reports",
  },
  {
    id: "advanced_analytics",
    name: "Advanced Analytics",
    description: "Access advanced analytics dashboards",
    category: "Reports",
  },

  // Applications and Screening
  {
    id: "application_processing",
    name: "Application Processing",
    description: "Process tenant applications",
    category: "Applications",
  },
  {
    id: "screening_management",
    name: "Screening Management",
    description: "Manage tenant screening",
    category: "Applications",
  },

  // Document Management
  {
    id: "document_access",
    name: "Document Access",
    description: "Access documents",
    category: "Documents",
  },
  {
    id: "document_management",
    name: "Document Management",
    description: "Manage documents",
    category: "Documents",
  },

  // Profile Management
  {
    id: "profile_management",
    name: "Profile Management",
    description: "Manage own profile",
    category: "Profile",
  },
];

const SYSTEM_ROLES = new Set(Object.values(UserRole));

// Legacy role mapping for backward compatibility
const LEGACY_ROLE_LOOKUP: Record<string, UserRole> = {
  administrator: UserRole.ADMIN,
  "property administrator": UserRole.ADMIN,
  "property manager": UserRole.MANAGER,
  property_manager: UserRole.MANAGER,
  renter: UserRole.TENANT,
  resident: UserRole.TENANT,
};

const normalizeRoleToSystem = (role?: string | null): UserRole => {
  if (!role) {
    return UserRole.TENANT;
  }

  const normalized = role.toLowerCase();

  if (SYSTEM_ROLES.has(normalized as UserRole)) {
    return normalized as UserRole;
  }

  return LEGACY_ROLE_LOOKUP[normalized] ?? UserRole.TENANT;
};

const isLegacyOrSystemRole = (role?: string | null) => {
  if (!role) return false;
  const normalized = role.toLowerCase();
  return (
    SYSTEM_ROLES.has(normalized as UserRole) ||
    LEGACY_ROLE_LOOKUP[normalized] !== undefined
  );
};

// Default system role configurations for single company architecture
const defaultRoleConfigs: RoleConfig[] = [
  {
    role: UserRole.ADMIN,
    name: UserRole.ADMIN,
    label: "Property Administrator",
    description: "Full system access and control",
    permissions: [
      "user_management",
      "role_management",
      "property_management",
      "tenant_management",
      "lease_management",
      "maintenance_management",
      "financial_management",
      "system_settings",
      "audit_logs",
      "backup_restore",
      "reports_all",
      "bulk_operations",
      "company_settings",
      "data_export",
      "advanced_analytics",
    ],
    userCount: 0,
    color: "destructive",
    isSystem: true,
    canEdit: false,
    canDelete: false,
  },
  {
    role: UserRole.MANAGER,
    name: UserRole.MANAGER,
    label: "Property Manager",
    description: "Company-wide management access",
    permissions: [
      "property_management",
      "tenant_management",
      "lease_management",
      "maintenance_management",
      "financial_management",
      "reports_all",
      "bulk_operations",
      "property_create",
      "tenant_create",
      "lease_create",
      "maintenance_assign",
      "payment_processing",
      "document_management",
    ],
    userCount: 0,
    color: "default",
    isSystem: true,
    canEdit: false,
    canDelete: false,
  },
  {
    role: UserRole.TENANT,
    name: UserRole.TENANT,
    label: "Tenant",
    description: "Property tenant with access to own data",
    permissions: [
      "profile_management",
      "maintenance_requests",
      "payment_portal",
      "document_access",
      "lease_view",
      "payment_history",
      "maintenance_history",
    ],
    userCount: 0,
    color: "outline",
    isSystem: true,
    canEdit: false,
    canDelete: false,
  },
];

export default function RoleManagementPage() {
  const router = useRouter();
  const { data: session } = useSession();
  const [roleConfigs, setRoleConfigs] =
    useState<RoleConfig[]>(defaultRoleConfigs);
  const [users, setUsers] = useState<UserWithRole[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedRole, setSelectedRole] = useState<string>("all");
  const [activeTab, setActiveTab] = useState("overview");

  // Role creation/editing state
  const [showRoleDialog, setShowRoleDialog] = useState(false);
  const [editingRole, setEditingRole] = useState<RoleConfig | null>(null);
  const [newRoleName, setNewRoleName] = useState("");
  const [newRoleDescription, setNewRoleDescription] = useState("");
  const [selectedPermissions, setSelectedPermissions] = useState<string[]>([]);

  // Check if all permissions are selected
  const allPermissionsSelected =
    selectedPermissions.length === AVAILABLE_PERMISSIONS.length;

  // User assignment state
  const [showUserAssignDialog, setShowUserAssignDialog] = useState(false);
  const [selectedUsers, setSelectedUsers] = useState<string[]>([]);
  const [targetRole, setTargetRole] = useState<string | null>(null);
  const [deleteLoading, setDeleteLoading] = useState<string | null>(null);

  // Check permissions
  const canManageRoles = session?.user?.role === UserRole.ADMIN;

  // Delete role function
  const handleDeleteRole = async (roleConfig: RoleConfig) => {
    try {
      setDeleteLoading(roleConfig.name);

      if (roleConfig.isSystem) {
        toast.error("System roles cannot be deleted");
        return;
      }

      if (roleConfig.userCount > 0) {
        toast.error(
          `Cannot delete role with ${roleConfig.userCount} assigned users. Please reassign users first.`
        );
        return;
      }

      // Call the delete API
      const response = await fetch(`/api/roles/${roleConfig.name}`, {
        method: "DELETE",
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to delete role");
      }

      toast.success("Role deleted successfully");

      // Refresh data
      await fetchRolesAndUsers();
    } catch (error) {
      toast.error(
        error instanceof Error ? error.message : "Failed to delete role"
      );
    } finally {
      setDeleteLoading(null);
    }
  };

  // Fetch roles and users
  const fetchRolesAndUsers = async () => {
    try {
      setIsLoading(true);

      // Fetch both roles and users in parallel
      const [rolesResponse, usersResponse] = await Promise.all([
        fetch("/api/roles?includeSystem=true"),
        fetch("/api/users?limit=1000"),
      ]);

      if (!rolesResponse.ok) {
        throw new Error("Failed to fetch roles");
      }

      if (!usersResponse.ok) {
        throw new Error("Failed to fetch users");
      }

      const rolesData = await rolesResponse.json();
      const usersData = await usersResponse.json();

      const rolePayload =
        rolesData?.data?.roles ?? rolesData?.roles ?? rolesData ?? [];

      const usersPayloadRaw = usersData?.data ?? usersData;
      const usersPayload = Array.isArray(usersPayloadRaw?.users)
        ? usersPayloadRaw.users
        : Array.isArray(usersPayloadRaw)
        ? usersPayloadRaw
        : [];

      const normalizedUsers: UserWithRole[] = (usersPayload as IUser[])
        .map((user) => {
          const originalRoleRaw =
            typeof user.role === "string" ? user.role : "";
          const normalizedOriginalRole = originalRoleRaw
            ? originalRoleRaw.toLowerCase()
            : "";
          const canonicalRole = normalizeRoleToSystem(originalRoleRaw);

          return {
            _id:
              typeof user._id === "string"
                ? user._id
                : (user._id as any)?.toString?.() ?? "",
            firstName: user.firstName || "",
            lastName: user.lastName || "",
            email: user.email || "",
            role: canonicalRole,
            originalRole: originalRoleRaw || canonicalRole,
            normalizedOriginalRole: normalizedOriginalRole || canonicalRole,
            isActive: Boolean(user.isActive),
            lastLogin: user.lastLogin
              ? new Date(user.lastLogin).toISOString()
              : undefined,
          };
        })
        .filter((user) => Boolean(user._id));

      setUsers(normalizedUsers);

      const systemRoleCounts = normalizedUsers.reduce<Record<UserRole, number>>(
        (acc, user) => {
          acc[user.role] = (acc[user.role] || 0) + 1;
          return acc;
        },
        {} as Record<UserRole, number>
      );

      const customRoleCounts = normalizedUsers.reduce<Record<string, number>>(
        (acc, user) => {
          if (!isLegacyOrSystemRole(user.originalRole)) {
            const key = user.normalizedOriginalRole;
            if (key) {
              acc[key] = (acc[key] || 0) + 1;
            }
          }
          return acc;
        },
        {}
      );

      const baseRoleMap = new Map<string, RoleConfig>(
        defaultRoleConfigs.map((role) => [
          role.role,
          {
            ...role,
            userCount: systemRoleCounts[role.role as UserRole] ?? 0,
          },
        ])
      );

      const customRoles: RoleConfig[] = [];

      if (Array.isArray(rolePayload)) {
        rolePayload.forEach((role: IRoleConfig) => {
          const originalName = role.name || "";
          const normalizedRawName = originalName.toLowerCase();
          const canonicalRole = normalizeRoleToSystem(originalName);
          const isSystemRole =
            Boolean(role.isSystem) && isLegacyOrSystemRole(originalName);

          const existing = baseRoleMap.get(canonicalRole);

          const baseConfig: RoleConfig = {
            role: isSystemRole ? canonicalRole : originalName,
            name: originalName || canonicalRole,
            label: role.label,
            description: role.description,
            permissions: role.permissions || [],
            userCount: 0,
            color: role.color || existing?.color || "outline",
            isSystem: Boolean(role.isSystem),
            canEdit: Boolean(role.canEdit),
            canDelete: Boolean(role.canDelete),
          };

          if (isSystemRole && SYSTEM_ROLES.has(canonicalRole)) {
            baseRoleMap.set(canonicalRole, {
              ...(existing ?? baseConfig),
              ...baseConfig,
              role: canonicalRole,
              userCount:
                systemRoleCounts[canonicalRole] ??
                role.userCount ??
                existing?.userCount ??
                0,
              color: baseConfig.color,
            });
          } else {
            const count =
              customRoleCounts[normalizedRawName] ?? role.userCount ?? 0;
            customRoles.push({
              ...baseConfig,
              role: originalName || normalizedRawName,
              userCount: count,
            });
          }
        });
      }

      const sortedCustomRoles = customRoles.sort((a, b) =>
        a.label.localeCompare(b.label)
      );

      setRoleConfigs([...baseRoleMap.values(), ...sortedCustomRoles]);
    } catch (error) {
      toast.error("Failed to load roles and users");
      // Fallback to default system roles if API fails
      setRoleConfigs(defaultRoleConfigs);
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    if (!showUserAssignDialog) {
      setTargetRole(null);
    }
  }, [showUserAssignDialog]);

  // Toggle all permissions
  const handleToggleAllPermissions = () => {
    if (allPermissionsSelected) {
      // Deselect all
      setSelectedPermissions([]);
    } else {
      // Select all
      setSelectedPermissions(AVAILABLE_PERMISSIONS.map((p) => p.id));
    }
  };

  // Handle role creation/editing
  const handleSaveRole = async () => {
    try {
      if (!newRoleName.trim()) {
        toast.error("Role name is required");
        return;
      }

      if (selectedPermissions.length === 0) {
        toast.error("At least one permission is required");
        return;
      }

      const roleName = newRoleName.toLowerCase().replace(/\s+/g, "_");
      const roleData = {
        name: roleName,
        label: newRoleName,
        description: newRoleDescription,
        permissions: selectedPermissions,
        color: "outline" as const,
        isActive: true,
      };

      if (editingRole && !editingRole.isSystem) {
        // Update existing custom role
        const response = await fetch(`/api/roles/${editingRole.name}`, {
          method: "PUT",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            label: newRoleName,
            description: newRoleDescription,
            permissions: selectedPermissions,
            color: "outline",
          }),
        });

        if (!response.ok) {
          const error = await response.json();
          throw new Error(error.message || "Failed to update role");
        }

        toast.success("Role updated successfully");
      } else {
        // Create new role
        const response = await fetch("/api/roles", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify(roleData),
        });

        if (!response.ok) {
          const error = await response.json();
          throw new Error(error.message || "Failed to create role");
        }

        toast.success("Role created successfully");
      }

      // Reset form and refresh data
      setShowRoleDialog(false);
      setEditingRole(null);
      setNewRoleName("");
      setNewRoleDescription("");
      setSelectedPermissions([]);

      // Refresh roles and users
      await fetchRolesAndUsers();
    } catch (error) {
      toast.error(
        error instanceof Error ? error.message : "Failed to save role"
      );
    }
  };

  // Handle user role assignment
  const handleAssignRole = async () => {
    try {
      if (!targetRole || selectedUsers.length === 0) {
        toast.error("Please select users and a role");
        return;
      }

      // Call the role assignment API
      const response = await fetch("/api/users/assign-roles", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          userIds: selectedUsers,
          targetRole,
          reason: "Bulk role assignment from admin panel",
          notifyUsers: false,
        }),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to assign roles");
      }

      const result = await response.json();

      toast.success(
        `Successfully assigned ${result.modifiedCount} users to ${targetRole} role`
      );

      setShowUserAssignDialog(false);
      setSelectedUsers([]);
      setTargetRole(null);

      // Refresh data
      await fetchRolesAndUsers();
    } catch (error) {
      toast.error(
        error instanceof Error ? error.message : "Failed to assign roles"
      );
    }
  };

  useEffect(() => {
    if (canManageRoles) {
      fetchRolesAndUsers();
    }
  }, [canManageRoles]);

  // Filter users based on search and role
  const filteredUsers = (users || []).filter((user) => {
    const matchesSearch =
      searchTerm === "" ||
      `${user.firstName} ${user.lastName}`
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      user.email.toLowerCase().includes(searchTerm.toLowerCase());

    const normalizedSelectedRole =
      selectedRole === "all" ? "all" : selectedRole.toLowerCase();
    const matchesRole =
      normalizedSelectedRole === "all" ||
      (isLegacyOrSystemRole(normalizedSelectedRole) &&
        user.role === normalizeRoleToSystem(normalizedSelectedRole)) ||
      user.normalizedOriginalRole === normalizedSelectedRole;

    return matchesSearch && matchesRole;
  });

  // Group permissions by category
  const permissionsByCategory = AVAILABLE_PERMISSIONS.reduce(
    (acc, permission) => {
      if (!acc[permission.category]) {
        acc[permission.category] = [];
      }
      acc[permission.category].push(permission);
      return acc;
    },
    {} as Record<string, Permission[]>
  );

  if (!canManageRoles) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="text-center">
          <Shield className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
          <h3 className="text-lg font-semibold">Access Denied</h3>
          <p className="text-muted-foreground">
            You don&apos;t have permission to manage user roles.
          </p>
          <Button
            variant="outline"
            onClick={() => router.back()}
            className="mt-4"
          >
            <ArrowLeft className="h-4 w-4 mr-2" />
            Go Back
          </Button>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="space-y-2">
          <h1 className="text-3xl font-bold tracking-tight">
            Role Management System
          </h1>
          <p className="text-muted-foreground">
            Create, manage, and assign roles with granular permissions
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Button
            variant="outline"
            size="sm"
            onClick={fetchRolesAndUsers}
            disabled={isLoading}
          >
            <RefreshCw
              className={`h-4 w-4 mr-2 ${isLoading ? "animate-spin" : ""}`}
            />
            Refresh
          </Button>
          <Button
            onClick={() => {
              setEditingRole(null);
              setNewRoleName("");
              setNewRoleDescription("");
              setSelectedPermissions([]);
              setShowRoleDialog(true);
            }}
          >
            <Plus className="h-4 w-4 mr-2" />
            Create Role
          </Button>
        </div>
      </div>

      {/* Tabs */}
      <Tabs
        value={activeTab}
        onValueChange={setActiveTab}
        className="space-y-6"
      >
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="overview">Overview</TabsTrigger>
          <TabsTrigger value="roles">Manage Roles</TabsTrigger>
          <TabsTrigger value="users">User Assignment</TabsTrigger>
        </TabsList>

        {/* Overview Tab */}
        <TabsContent value="overview" className="space-y-6">
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
            {roleConfigs.map((config) => (
              <Card key={config.name} className="relative">
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">
                    {config.label}
                  </CardTitle>
                  <Badge variant={config.color as any} className="text-xs">
                    {config.userCount} users
                  </Badge>
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{config.userCount}</div>
                  <p className="text-xs text-muted-foreground mb-3">
                    {config.description}
                  </p>
                  <div className="space-y-1">
                    <p className="text-xs font-medium">Key Permissions:</p>
                    <div className="flex flex-wrap gap-1">
                      {config.permissions.slice(0, 3).map((permission) => (
                        <Badge
                          key={permission}
                          variant="outline"
                          className="text-xs"
                        >
                          {permission.replace("_", " ")}
                        </Badge>
                      ))}
                      {config.permissions.length > 3 && (
                        <Badge variant="outline" className="text-xs">
                          +{config.permissions.length - 3} more
                        </Badge>
                      )}
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        {/* Roles Management Tab */}
        <TabsContent value="roles" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Settings className="h-5 w-5" />
                Role Management
              </CardTitle>
              <CardDescription>
                Create, edit, and manage system roles and their permissions
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {roleConfigs.map((config) => (
                  <div
                    key={config.name}
                    className="flex items-center justify-between p-4 border rounded-lg"
                  >
                    <div className="flex items-center gap-4">
                      <div className="flex items-center gap-2">
                        {config.isSystem ? (
                          <Lock className="h-4 w-4 text-muted-foreground" />
                        ) : (
                          <Unlock className="h-4 w-4 text-green-600" />
                        )}
                        <div>
                          <h3 className="font-medium">{config.label}</h3>
                          <p className="text-sm text-muted-foreground">
                            {config.description}
                          </p>
                        </div>
                      </div>
                      <Badge variant={config.color as any}>
                        {config.userCount} users
                      </Badge>
                    </div>
                    <div className="flex items-center gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => {
                          setEditingRole(config);
                          setNewRoleName(config.label);
                          setNewRoleDescription(config.description);
                          setSelectedPermissions(config.permissions);
                          setShowRoleDialog(true);
                        }}
                        disabled={!config.canEdit}
                      >
                        <Eye className="h-4 w-4 mr-2" />
                        {config.canEdit ? "Edit" : "View"}
                      </Button>
                      {config.canDelete && (
                        <DeleteConfirmationDialog
                          itemName={config.name}
                          itemType="role"
                          onConfirm={() => handleDeleteRole(config)}
                          loading={deleteLoading === config.name}
                        >
                          <Button
                            variant="outline"
                            size="sm"
                            disabled={deleteLoading === config.name}
                            className="text-red-600 border-red-200 hover:bg-red-50 hover:text-red-700 hover:border-red-300"
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </DeleteConfirmationDialog>
                      )}
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* User Assignment Tab */}
        <TabsContent value="users" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <UserCheck className="h-5 w-5" />
                User Role Assignment
              </CardTitle>
              <CardDescription>
                Assign roles to users and manage user permissions
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex flex-col sm:flex-row gap-4 mb-6">
                <div className="flex-1">
                  <div className="relative">
                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                    <Input
                      placeholder="Search users by name or email..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="pl-10"
                    />
                  </div>
                </div>
                <Select
                  value={selectedRole}
                  onValueChange={(value) => setSelectedRole(value)}
                >
                  <SelectTrigger className="w-full sm:w-48">
                    <SelectValue placeholder="Filter by role" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Roles</SelectItem>
                    {roleConfigs.map((config) => {
                      const normalizedValue = (
                        config.name || config.role
                      ).toLowerCase();
                      return (
                        <SelectItem
                          key={`${config.name}-${normalizedValue}`}
                          value={normalizedValue}
                        >
                          {config.label}
                        </SelectItem>
                      );
                    })}
                  </SelectContent>
                </Select>
                <Button
                  onClick={() => setShowUserAssignDialog(true)}
                  disabled={selectedUsers.length === 0}
                >
                  <UserCheck className="h-4 w-4 mr-2" />
                  Assign Roles
                </Button>
              </div>

              {/* Users Table */}
              <div className="rounded-md border">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="w-12">
                        <Checkbox
                          checked={
                            selectedUsers.length === filteredUsers.length &&
                            filteredUsers.length > 0
                          }
                          onCheckedChange={(checked) => {
                            if (checked) {
                              setSelectedUsers(filteredUsers.map((u) => u._id));
                            } else {
                              setSelectedUsers([]);
                            }
                          }}
                        />
                      </TableHead>
                      <TableHead>User</TableHead>
                      <TableHead>Email</TableHead>
                      <TableHead>Current Role</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Last Login</TableHead>
                      <TableHead className="text-right">Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {isLoading ? (
                      Array.from({ length: 5 }).map((_, index) => (
                        <TableRow key={index}>
                          <TableCell>
                            <div className="h-4 w-4 bg-muted rounded animate-pulse" />
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center space-x-3">
                              <div className="h-8 w-8 bg-muted rounded-full animate-pulse" />
                              <div className="space-y-1">
                                <div className="h-4 w-32 bg-muted rounded animate-pulse" />
                              </div>
                            </div>
                          </TableCell>
                          <TableCell>
                            <div className="h-4 w-48 bg-muted rounded animate-pulse" />
                          </TableCell>
                          <TableCell>
                            <div className="h-6 w-24 bg-muted rounded animate-pulse" />
                          </TableCell>
                          <TableCell>
                            <div className="h-6 w-16 bg-muted rounded animate-pulse" />
                          </TableCell>
                          <TableCell>
                            <div className="h-4 w-20 bg-muted rounded animate-pulse" />
                          </TableCell>
                          <TableCell>
                            <div className="h-8 w-8 bg-muted rounded animate-pulse ml-auto" />
                          </TableCell>
                        </TableRow>
                      ))
                    ) : filteredUsers.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={7} className="text-center py-8">
                          <div className="flex flex-col items-center gap-2">
                            <Users className="h-8 w-8 text-muted-foreground" />
                            <p className="text-muted-foreground">
                              {searchTerm || selectedRole !== "all"
                                ? "No users found matching your criteria"
                                : "No users found"}
                            </p>
                          </div>
                        </TableCell>
                      </TableRow>
                    ) : (
                      filteredUsers.map((user) => (
                        <TableRow key={user._id.toString()}>
                          <TableCell>
                            <Checkbox
                              checked={selectedUsers.includes(user._id)}
                              onCheckedChange={(checked) => {
                                if (checked) {
                                  setSelectedUsers((prev) => [
                                    ...prev,
                                    user._id,
                                  ]);
                                } else {
                                  setSelectedUsers((prev) =>
                                    prev.filter((id) => id !== user._id)
                                  );
                                }
                              }}
                            />
                          </TableCell>
                          <TableCell>
                            <div className="flex items-center space-x-3">
                              <div className="h-8 w-8 bg-primary/10 rounded-full flex items-center justify-center">
                                <span className="text-sm font-medium">
                                  {user.firstName[0]}
                                  {user.lastName[0]}
                                </span>
                              </div>
                              <div>
                                <p className="font-medium">
                                  {user.firstName} {user.lastName}
                                </p>
                              </div>
                            </div>
                          </TableCell>
                          <TableCell>{user.email}</TableCell>
                          <TableCell>
                            {(() => {
                              const matchedRole =
                                roleConfigs.find(
                                  (config) =>
                                    config.role === user.role ||
                                    config.name.toLowerCase() ===
                                      user.normalizedOriginalRole
                                ) ?? null;
                              const rawLabel = matchedRole
                                ? matchedRole.label
                                : user.originalRole || user.role;
                              const displayLabel = rawLabel.replace(/_/g, " ");

                              return (
                                <Badge
                                  variant={
                                    (matchedRole?.color as any) || "outline"
                                  }
                                  className="capitalize"
                                >
                                  {displayLabel}
                                </Badge>
                              );
                            })()}
                          </TableCell>
                          <TableCell>
                            <Badge
                              variant={user.isActive ? "default" : "secondary"}
                            >
                              {user.isActive ? "Active" : "Inactive"}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            {user.lastLogin
                              ? new Date(user.lastLogin).toLocaleDateString()
                              : "Never"}
                          </TableCell>
                          <TableCell className="text-right">
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() =>
                                router.push(
                                  `/dashboard/admin/users/${user._id}`
                                )
                              }
                            >
                              <Eye className="h-4 w-4" />
                            </Button>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Role Creation/Edit Dialog */}
      <CustomModal
        open={showRoleDialog}
        onOpenChange={setShowRoleDialog}
        size="3/4"
      >
        <CustomModalHeader onClose={() => setShowRoleDialog(false)}>
          <CustomModalTitle>
            {editingRole ? "Edit Role" : "Create New Role"}
          </CustomModalTitle>
          <CustomModalDescription>
            {editingRole
              ? "Modify role permissions and settings"
              : "Create a new role with specific permissions"}
          </CustomModalDescription>
        </CustomModalHeader>

        <CustomModalBody>
          <div className="space-y-6">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="roleName">
                  Role Name <span className="text-red-500">*</span>
                </Label>
                <Input
                  id="roleName"
                  value={newRoleName}
                  onChange={(e) => setNewRoleName(e.target.value)}
                  placeholder="Enter role name"
                  disabled={editingRole?.isSystem}
                  required
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="roleDescription">
                  Description <span className="text-red-500">*</span>
                </Label>
                <Input
                  id="roleDescription"
                  value={newRoleDescription}
                  onChange={(e) => setNewRoleDescription(e.target.value)}
                  placeholder="Enter role description"
                  required
                />
              </div>
            </div>

            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <Label className="font-bold">Permissions</Label>
                <Button
                  type="button"
                  variant="outline"
                  size="sm"
                  onClick={handleToggleAllPermissions}
                  disabled={editingRole?.isSystem && !editingRole?.canEdit}
                >
                  {allPermissionsSelected ? "Deselect All" : "Select All"}
                </Button>
              </div>
              <div className="space-y-4">
                {Object.entries(permissionsByCategory).map(
                  ([category, permissions]) => (
                    <div key={category} className="space-y-2">
                      <h4 className="font-bold text-sm">{category}</h4>
                      <div className="grid grid-cols-6 gap-2">
                        {permissions.map((permission) => (
                          <div
                            key={permission.id}
                            className="flex items-center space-x-2"
                          >
                            <Checkbox
                              id={permission.id}
                              checked={selectedPermissions.includes(
                                permission.id
                              )}
                              onCheckedChange={(checked) => {
                                if (checked) {
                                  setSelectedPermissions((prev) => [
                                    ...prev,
                                    permission.id,
                                  ]);
                                } else {
                                  setSelectedPermissions((prev) =>
                                    prev.filter((p) => p !== permission.id)
                                  );
                                }
                              }}
                              disabled={
                                editingRole?.isSystem && !editingRole?.canEdit
                              }
                            />
                            <Label htmlFor={permission.id} className="text-sm">
                              {permission.name}
                            </Label>
                          </div>
                        ))}
                      </div>
                    </div>
                  )
                )}
              </div>
            </div>
          </div>
        </CustomModalBody>

        <CustomModalFooter>
          <Button variant="outline" onClick={() => setShowRoleDialog(false)}>
            Cancel
          </Button>
          <Button
            onClick={handleSaveRole}
            disabled={!newRoleName.trim() || !newRoleDescription.trim()}
          >
            {editingRole ? "Update Role" : "Create Role"}
          </Button>
        </CustomModalFooter>
      </CustomModal>

      {/* User Role Assignment Dialog */}
      <Dialog
        open={showUserAssignDialog}
        onOpenChange={setShowUserAssignDialog}
      >
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Assign Role to Users</DialogTitle>
            <DialogDescription>
              Assign a role to {selectedUsers.length} selected user(s)
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>Select Role</Label>
              <Select
                value={targetRole || ""}
                onValueChange={(value) => setTargetRole(value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Choose a role" />
                </SelectTrigger>
                <SelectContent>
                  {roleConfigs.map((config) => (
                    <SelectItem key={config.name} value={config.name}>
                      {config.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setShowUserAssignDialog(false)}
            >
              Cancel
            </Button>
            <Button onClick={handleAssignRole} disabled={!targetRole}>
              Assign Role
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
