"use client";

import React, { useState, useEffect } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Alert,
  AlertDescription,
  AlertTitle,
} from "@/components/ui/alert";
import {
  Calendar,
  Bell,
  Palette,
  Clock,
  Eye,
  Save,
  RotateCcw,
  AlertTriangle,
} from "lucide-react";
import { toast } from "sonner";

interface CalendarSettings {
  // Display preferences
  defaultView: string;
  weekends: boolean;
  firstDay: number;
  timezone: string;
  
  // Time settings
  businessHours: {
    enabled: boolean;
    startTime: string;
    endTime: string;
    daysOfWeek: number[];
  };
  
  slotDuration: string;
  snapDuration: string;
  defaultEventDuration: string;
  
  // Event preferences
  defaultEventType: string;
  defaultEventPriority: string;
  defaultReminders: number[];
  
  // Notification preferences
  emailNotifications: {
    invitations: boolean;
    reminders: boolean;
    updates: boolean;
    cancellations: boolean;
  };
  
  // View preferences
  showWeekNumbers: boolean;
  showDeclinedEvents: boolean;
  eventLimit: number;
  
  // Color preferences
  eventColors: Record<string, string>;
}

const defaultSettings: CalendarSettings = {
  defaultView: "dayGridMonth",
  weekends: true,
  firstDay: 0,
  timezone: "local",
  businessHours: {
    enabled: true,
    startTime: "09:00",
    endTime: "17:00",
    daysOfWeek: [1, 2, 3, 4, 5],
  },
  slotDuration: "00:30",
  snapDuration: "00:15",
  defaultEventDuration: "01:00",
  defaultEventType: "GENERAL",
  defaultEventPriority: "MEDIUM",
  defaultReminders: [15],
  emailNotifications: {
    invitations: true,
    reminders: true,
    updates: true,
    cancellations: true,
  },
  showWeekNumbers: false,
  showDeclinedEvents: false,
  eventLimit: 3,
  eventColors: {
    LEASE_RENEWAL: "#3b82f6",
    PROPERTY_INSPECTION: "#10b981",
    MAINTENANCE_APPOINTMENT: "#f59e0b",
    PROPERTY_SHOWING: "#8b5cf6",
    TENANT_MEETING: "#6366f1",
    RENT_COLLECTION: "#059669",
    MOVE_IN: "#06b6d4",
    MOVE_OUT: "#ef4444",
    GENERAL: "#6b7280",
  },
};

export default function CalendarSettingsPage() {
  const [settings, setSettings] = useState<CalendarSettings>(defaultSettings);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [hasChanges, setHasChanges] = useState(false);

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    try {
      const response = await fetch("/api/calendar/settings");
      if (response.ok) {
        const result = await response.json();
        setSettings({ ...defaultSettings, ...result.data });
      }
    } catch (error) {
      console.error("Failed to load settings:", error);
      toast.error("Failed to load calendar settings");
    } finally {
      setLoading(false);
    }
  };

  const updateSetting = (path: string, value: any) => {
    setSettings(prev => {
      const newSettings = { ...prev };
      const keys = path.split('.');
      let current = newSettings;
      
      for (let i = 0; i < keys.length - 1; i++) {
        current = current[keys[i] as keyof typeof current] as any;
      }
      
      current[keys[keys.length - 1] as keyof typeof current] = value;
      return newSettings;
    });
    setHasChanges(true);
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      const response = await fetch("/api/calendar/settings", {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(settings),
      });

      if (response.ok) {
        toast.success("Calendar settings saved successfully!");
        setHasChanges(false);
      } else {
        const result = await response.json();
        toast.error(result.error || "Failed to save settings");
      }
    } catch (error) {
      console.error("Failed to save settings:", error);
      toast.error("Failed to save calendar settings");
    } finally {
      setSaving(false);
    }
  };

  const resetSettings = async () => {
    try {
      const response = await fetch("/api/calendar/settings", {
        method: "DELETE",
      });

      if (response.ok) {
        setSettings(defaultSettings);
        setHasChanges(false);
        toast.success("Settings reset to defaults");
      } else {
        toast.error("Failed to reset settings");
      }
    } catch (error) {
      console.error("Failed to reset settings:", error);
      toast.error("Failed to reset settings");
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-[400px]">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mx-auto mb-4"></div>
          <p className="text-muted-foreground">Loading calendar settings...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-foreground">Calendar Settings</h1>
          <p className="text-muted-foreground">
            Customize your calendar preferences and behavior
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Button
            variant="outline"
            onClick={resetSettings}
            disabled={saving}
          >
            <RotateCcw className="h-4 w-4 mr-2" />
            Reset to Defaults
          </Button>
          <Button
            onClick={saveSettings}
            disabled={!hasChanges || saving}
            className="bg-primary hover:bg-primary/90"
          >
            <Save className="h-4 w-4 mr-2" />
            {saving ? "Saving..." : "Save Changes"}
          </Button>
        </div>
      </div>

      {/* Unsaved Changes Alert */}
      {hasChanges && (
        <Alert>
          <AlertTriangle className="h-4 w-4" />
          <AlertTitle>Unsaved Changes</AlertTitle>
          <AlertDescription>
            You have unsaved changes. Don't forget to save your settings.
          </AlertDescription>
        </Alert>
      )}

      {/* Settings Tabs */}
      <Tabs defaultValue="display" className="space-y-6">
        <TabsList className="grid w-full grid-cols-5">
          <TabsTrigger value="display" className="flex items-center gap-2">
            <Eye className="h-4 w-4" />
            Display
          </TabsTrigger>
          <TabsTrigger value="time" className="flex items-center gap-2">
            <Clock className="h-4 w-4" />
            Time
          </TabsTrigger>
          <TabsTrigger value="events" className="flex items-center gap-2">
            <Calendar className="h-4 w-4" />
            Events
          </TabsTrigger>
          <TabsTrigger value="notifications" className="flex items-center gap-2">
            <Bell className="h-4 w-4" />
            Notifications
          </TabsTrigger>
          <TabsTrigger value="colors" className="flex items-center gap-2">
            <Palette className="h-4 w-4" />
            Colors
          </TabsTrigger>
        </TabsList>

        {/* Display Settings */}
        <TabsContent value="display" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Eye className="h-5 w-5" />
                Display Preferences
              </CardTitle>
              <CardDescription>
                Configure how your calendar appears and behaves
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-2">
                  <Label htmlFor="defaultView">Default View</Label>
                  <Select
                    value={settings.defaultView}
                    onValueChange={(value) => updateSetting('defaultView', value)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="dayGridMonth">Month</SelectItem>
                      <SelectItem value="timeGridWeek">Week</SelectItem>
                      <SelectItem value="timeGridDay">Day</SelectItem>
                      <SelectItem value="listWeek">Agenda</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="firstDay">Week Starts On</Label>
                  <Select
                    value={settings.firstDay.toString()}
                    onValueChange={(value) => updateSetting('firstDay', parseInt(value))}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="0">Sunday</SelectItem>
                      <SelectItem value="1">Monday</SelectItem>
                      <SelectItem value="2">Tuesday</SelectItem>
                      <SelectItem value="3">Wednesday</SelectItem>
                      <SelectItem value="4">Thursday</SelectItem>
                      <SelectItem value="5">Friday</SelectItem>
                      <SelectItem value="6">Saturday</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="weekends">Show Weekends</Label>
                  <p className="text-sm text-muted-foreground">
                    Display Saturday and Sunday in calendar views
                  </p>
                </div>
                <Switch
                  id="weekends"
                  checked={settings.weekends}
                  onCheckedChange={(checked) => updateSetting('weekends', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="showWeekNumbers">Show Week Numbers</Label>
                  <p className="text-sm text-muted-foreground">
                    Display week numbers in calendar views
                  </p>
                </div>
                <Switch
                  id="showWeekNumbers"
                  checked={settings.showWeekNumbers}
                  onCheckedChange={(checked) => updateSetting('showWeekNumbers', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="showDeclinedEvents">Show Declined Events</Label>
                  <p className="text-sm text-muted-foreground">
                    Display events you've declined in the calendar
                  </p>
                </div>
                <Switch
                  id="showDeclinedEvents"
                  checked={settings.showDeclinedEvents}
                  onCheckedChange={(checked) => updateSetting('showDeclinedEvents', checked)}
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="eventLimit">Events Per Day (Month View)</Label>
                <Input
                  id="eventLimit"
                  type="number"
                  min="1"
                  max="10"
                  value={settings.eventLimit}
                  onChange={(e) => updateSetting('eventLimit', parseInt(e.target.value))}
                  className="w-32"
                />
                <p className="text-sm text-muted-foreground">
                  Maximum number of events to show per day in month view
                </p>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Time Settings */}
        <TabsContent value="time" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Clock className="h-5 w-5" />
                Time Configuration
              </CardTitle>
              <CardDescription>
                Configure time slots and business hours
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="businessHours">Business Hours</Label>
                  <p className="text-sm text-muted-foreground">
                    Highlight business hours in calendar views
                  </p>
                </div>
                <Switch
                  id="businessHours"
                  checked={settings.businessHours.enabled}
                  onCheckedChange={(checked) => updateSetting('businessHours.enabled', checked)}
                />
              </div>

              {settings.businessHours.enabled && (
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 pl-6 border-l-2 border-muted">
                  <div className="space-y-2">
                    <Label htmlFor="startTime">Start Time</Label>
                    <Input
                      id="startTime"
                      type="time"
                      value={settings.businessHours.startTime}
                      onChange={(e) => updateSetting('businessHours.startTime', e.target.value)}
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="endTime">End Time</Label>
                    <Input
                      id="endTime"
                      type="time"
                      value={settings.businessHours.endTime}
                      onChange={(e) => updateSetting('businessHours.endTime', e.target.value)}
                    />
                  </div>
                </div>
              )}

              <Separator />

              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="space-y-2">
                  <Label htmlFor="slotDuration">Time Slot Duration</Label>
                  <Select
                    value={settings.slotDuration}
                    onValueChange={(value) => updateSetting('slotDuration', value)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="00:15">15 minutes</SelectItem>
                      <SelectItem value="00:30">30 minutes</SelectItem>
                      <SelectItem value="01:00">1 hour</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="snapDuration">Snap Duration</Label>
                  <Select
                    value={settings.snapDuration}
                    onValueChange={(value) => updateSetting('snapDuration', value)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="00:15">15 minutes</SelectItem>
                      <SelectItem value="00:30">30 minutes</SelectItem>
                      <SelectItem value="01:00">1 hour</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="defaultDuration">Default Event Duration</Label>
                  <Select
                    value={settings.defaultEventDuration}
                    onValueChange={(value) => updateSetting('defaultEventDuration', value)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="00:30">30 minutes</SelectItem>
                      <SelectItem value="01:00">1 hour</SelectItem>
                      <SelectItem value="02:00">2 hours</SelectItem>
                      <SelectItem value="04:00">4 hours</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Event Settings */}
        <TabsContent value="events" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Calendar className="h-5 w-5" />
                Event Defaults
              </CardTitle>
              <CardDescription>
                Set default values for new events
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-2">
                  <Label htmlFor="defaultEventType">Default Event Type</Label>
                  <Select
                    value={settings.defaultEventType}
                    onValueChange={(value) => updateSetting('defaultEventType', value)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="GENERAL">General</SelectItem>
                      <SelectItem value="PROPERTY_SHOWING">Property Showing</SelectItem>
                      <SelectItem value="PROPERTY_INSPECTION">Property Inspection</SelectItem>
                      <SelectItem value="MAINTENANCE_APPOINTMENT">Maintenance</SelectItem>
                      <SelectItem value="TENANT_MEETING">Tenant Meeting</SelectItem>
                      <SelectItem value="LEASE_RENEWAL">Lease Renewal</SelectItem>
                      <SelectItem value="RENT_COLLECTION">Rent Collection</SelectItem>
                      <SelectItem value="MOVE_IN">Move In</SelectItem>
                      <SelectItem value="MOVE_OUT">Move Out</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="defaultEventPriority">Default Priority</Label>
                  <Select
                    value={settings.defaultEventPriority}
                    onValueChange={(value) => updateSetting('defaultEventPriority', value)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="LOW">Low</SelectItem>
                      <SelectItem value="MEDIUM">Medium</SelectItem>
                      <SelectItem value="HIGH">High</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="space-y-2">
                <Label>Default Reminders</Label>
                <div className="flex flex-wrap gap-2">
                  {[5, 10, 15, 30, 60, 120, 1440].map((minutes) => {
                    const isSelected = settings.defaultReminders.includes(minutes);
                    const label = minutes < 60 
                      ? `${minutes} min` 
                      : minutes === 60 
                        ? "1 hour" 
                        : minutes === 120 
                          ? "2 hours" 
                          : "1 day";
                    
                    return (
                      <Badge
                        key={minutes}
                        variant={isSelected ? "default" : "outline"}
                        className="cursor-pointer"
                        onClick={() => {
                          const newReminders = isSelected
                            ? settings.defaultReminders.filter(r => r !== minutes)
                            : [...settings.defaultReminders, minutes].sort((a, b) => a - b);
                          updateSetting('defaultReminders', newReminders);
                        }}
                      >
                        {label}
                      </Badge>
                    );
                  })}
                </div>
                <p className="text-sm text-muted-foreground">
                  Click to toggle reminder times for new events
                </p>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Notification Settings */}
        <TabsContent value="notifications" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Bell className="h-5 w-5" />
                Email Notifications
              </CardTitle>
              <CardDescription>
                Configure when to receive email notifications
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4">
                <div className="flex items-center justify-between">
                  <div className="space-y-0.5">
                    <Label htmlFor="invitations">Event Invitations</Label>
                    <p className="text-sm text-muted-foreground">
                      Receive emails when invited to events
                    </p>
                  </div>
                  <Switch
                    id="invitations"
                    checked={settings.emailNotifications.invitations}
                    onCheckedChange={(checked) => updateSetting('emailNotifications.invitations', checked)}
                  />
                </div>

                <div className="flex items-center justify-between">
                  <div className="space-y-0.5">
                    <Label htmlFor="reminders">Event Reminders</Label>
                    <p className="text-sm text-muted-foreground">
                      Receive reminder emails before events
                    </p>
                  </div>
                  <Switch
                    id="reminders"
                    checked={settings.emailNotifications.reminders}
                    onCheckedChange={(checked) => updateSetting('emailNotifications.reminders', checked)}
                  />
                </div>

                <div className="flex items-center justify-between">
                  <div className="space-y-0.5">
                    <Label htmlFor="updates">Event Updates</Label>
                    <p className="text-sm text-muted-foreground">
                      Receive emails when events are modified
                    </p>
                  </div>
                  <Switch
                    id="updates"
                    checked={settings.emailNotifications.updates}
                    onCheckedChange={(checked) => updateSetting('emailNotifications.updates', checked)}
                  />
                </div>

                <div className="flex items-center justify-between">
                  <div className="space-y-0.5">
                    <Label htmlFor="cancellations">Event Cancellations</Label>
                    <p className="text-sm text-muted-foreground">
                      Receive emails when events are cancelled
                    </p>
                  </div>
                  <Switch
                    id="cancellations"
                    checked={settings.emailNotifications.cancellations}
                    onCheckedChange={(checked) => updateSetting('emailNotifications.cancellations', checked)}
                  />
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Color Settings */}
        <TabsContent value="colors" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Palette className="h-5 w-5" />
                Event Colors
              </CardTitle>
              <CardDescription>
                Customize colors for different event types
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              {Object.entries(settings.eventColors).map(([type, color]) => (
                <div key={type} className="flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div
                      className="w-6 h-6 rounded border"
                      style={{ backgroundColor: color }}
                    />
                    <Label className="font-medium">
                      {type.replace(/_/g, " ")}
                    </Label>
                  </div>
                  <Input
                    type="color"
                    value={color}
                    onChange={(e) => updateSetting(`eventColors.${type}`, e.target.value)}
                    className="w-16 h-8 p-1 border rounded"
                  />
                </div>
              ))}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
