"use client";

import { useState, useEffect, useCallback } from "react";
import { useRouter } from "next/navigation";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  FileText,
  Plus,
  Search,
  Filter,
  RefreshCw,
  LayoutGrid,
  List,
  CheckCircle,
  Calendar,
  DollarSign,
  Home,
  Eye,
  Edit,
  MoreHorizontal,
} from "lucide-react";

import {
  leaseService,
  LeaseResponse,
  PaginatedLeasesResponse,
  LeaseQueryParams,
} from "@/lib/services/lease.service";
import { LeaseCard } from "@/components/leases/LeaseCard";

export default function ActiveLeasesPage() {
  const router = useRouter();
  const [leases, setLeases] = useState<LeaseResponse[]>([]);
  const [loading, setLoading] = useState(true);
  const [pagination, setPagination] = useState({
    page: 1,
    limit: 12,
    total: 0,
    pages: 0,
    hasNext: false,
    hasPrev: false,
  });
  const [filters, setFilters] = useState<Omit<LeaseQueryParams, "status">>({
    page: 1,
    limit: 12,
    search: "",
    sortBy: "endDate",
    sortOrder: "asc",
  });
  const [viewMode, setViewMode] = useState<"table" | "cards">("table");
  const [stats, setStats] = useState({
    total: 0,
    expiringThisMonth: 0,
    expiringNext30Days: 0,
    totalRentValue: 0,
  });

  const fetchActiveLeases = useCallback(async () => {
    try {
      setLoading(true);
      const response: PaginatedLeasesResponse =
        await leaseService.getActiveLeases(filters);
      setLeases(Array.isArray(response.data) ? response.data : []);
      setPagination(response.pagination);
    } catch (error) {
      console.error("Error fetching active leases:", error);
      toast.error("Failed to fetch active leases");
      setLeases([]);
    } finally {
      setLoading(false);
    }
  }, [filters]);

  const fetchStats = useCallback(async () => {
    try {
      const statsData = await leaseService.getLeaseStats();
      const totalRentValue = leases.reduce(
        (sum, lease) => sum + lease.terms.rentAmount,
        0
      );
      setStats({
        total: statsData.active,
        expiringThisMonth: statsData.expiringThisMonth,
        expiringNext30Days: statsData.expiringThisMonth,
        totalRentValue,
      });
    } catch {
      setStats({
        total: 0,
        expiringThisMonth: 0,
        expiringNext30Days: 0,
        totalRentValue: 0,
      });
    }
  }, [leases]);

  useEffect(() => {
    let isMounted = true;

    const loadData = async () => {
      if (isMounted) {
        await fetchActiveLeases();
        await fetchStats();
      }
    };

    loadData();

    return () => {
      isMounted = false;
    };
  }, [fetchActiveLeases, fetchStats]);

  const handleSearch = (value: string) => {
    setFilters((prev) => ({ ...prev, search: value, page: 1 }));
  };

  const handleSort = (sortBy: string, sortOrder: "asc" | "desc") => {
    setFilters((prev) => ({ ...prev, sortBy, sortOrder, page: 1 }));
  };

  const handlePageChange = (page: number) => {
    setFilters((prev) => ({ ...prev, page }));
  };

  const handleLeaseUpdate = () => {
    fetchActiveLeases();
    fetchStats();
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  if (loading && leases.length === 0) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">Active Leases</h1>
            <p className="text-muted-foreground">
              Manage your active lease agreements
            </p>
          </div>
          <Skeleton className="h-10 w-32" />
        </div>

        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          {Array.from({ length: 4 }).map((_, i) => (
            <Skeleton key={i} className="h-24" />
          ))}
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {Array.from({ length: 6 }).map((_, i) => (
            <Skeleton key={i} className="h-64" />
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Active Leases</h1>
          <p className="text-muted-foreground">
            Manage your currently active lease agreements
          </p>
        </div>
        <Button onClick={() => router.push("/dashboard/leases/new")}>
          <Plus className="mr-2 h-4 w-4" />
          Create Lease
        </Button>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        {loading && leases.length === 0 ? (
          [...Array(4)].map((_, i) => (
            <Card key={i} className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="space-y-2">
                    <Skeleton className="h-4 w-20" />
                    <Skeleton className="h-8 w-16" />
                  </div>
                  <Skeleton className="h-8 w-8 rounded" />
                </div>
              </CardContent>
            </Card>
          ))
        ) : (
          <>
            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="space-y-2">
                    <p className="text-sm font-medium text-muted-foreground">
                      Active Leases
                    </p>
                    <p className="text-2xl font-bold text-green-600">
                      {stats.total}
                    </p>
                  </div>
                  <CheckCircle className="h-8 w-8 text-green-600" />
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4 ">
                <div className="flex items-center justify-between">
                  <div className="space-y-2">
                    <p className="text-sm font-medium text-muted-foreground">
                      Expiring Soon
                    </p>
                    <p className="text-2xl font-bold text-orange-600">
                      {stats.expiringNext30Days}
                    </p>
                  </div>
                  <Calendar className="h-8 w-8 text-orange-600" />
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="space-y-2">
                    <p className="text-sm font-medium text-muted-foreground">
                      Monthly Rent
                    </p>
                    <p className="text-2xl font-bold text-blue-600">
                      {formatCurrency(stats.totalRentValue)}
                    </p>
                  </div>
                  <DollarSign className="h-8 w-8 text-blue-600" />
                </div>
              </CardContent>
            </Card>

            <Card className="py-0">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="space-y-2">
                    <p className="text-sm font-medium text-muted-foreground">
                      This Month
                    </p>
                    <p className="text-2xl font-bold text-purple-600">
                      {stats.expiringThisMonth}
                    </p>
                  </div>
                  <FileText className="h-8 w-8 text-purple-600" />
                </div>
              </CardContent>
            </Card>
          </>
        )}
      </div>

      {/* Filters and Actions */}
      <Card>
        <CardContent className="p-4">
          <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
            <div className="flex flex-1 gap-4 items-center">
              <div className="relative flex-1 max-w-sm">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground h-4 w-4" />
                <Input
                  placeholder="Search active leases..."
                  value={filters.search}
                  onChange={(e) => handleSearch(e.target.value)}
                  className="pl-10"
                />
              </div>

              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline">
                    <Filter className="mr-2 h-4 w-4" />
                    Sort
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent>
                  <DropdownMenuLabel>Sort by</DropdownMenuLabel>
                  <DropdownMenuItem
                    onClick={() => handleSort("endDate", "asc")}
                  >
                    End Date (Soonest)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("endDate", "desc")}
                  >
                    End Date (Latest)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("startDate", "desc")}
                  >
                    Start Date (Latest)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("terms.rentAmount", "desc")}
                  >
                    Rent (High to Low)
                  </DropdownMenuItem>
                  <DropdownMenuItem
                    onClick={() => handleSort("terms.rentAmount", "asc")}
                  >
                    Rent (Low to High)
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            </div>

            <div className="flex items-center gap-2">
              <Button variant="outline" size="sm" onClick={fetchActiveLeases}>
                <RefreshCw className="h-4 w-4" />
              </Button>

              <div className="flex items-center gap-2">
                <Button
                  variant={viewMode === "table" ? "default" : "outline"}
                  size="sm"
                  onClick={() => setViewMode("table")}
                >
                  <List className="h-4 w-4 mr-2" />
                  Table
                </Button>
                <Button
                  variant={viewMode === "cards" ? "default" : "outline"}
                  size="sm"
                  onClick={() => setViewMode("cards")}
                >
                  <LayoutGrid className="h-4 w-4 mr-2" />
                  Cards
                </Button>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Active Leases List */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Active Leases ({leases.length})</CardTitle>
              <CardDescription>
                A list of all currently active lease agreements
              </CardDescription>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {viewMode === "cards" ? (
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {loading && leases.length === 0 ? (
                [...Array(6)].map((_, i) => (
                  <Card key={i} className="hover:shadow-md transition-shadow">
                    <CardHeader className="pb-3">
                      <div className="flex items-start justify-between">
                        <div className="space-y-1 flex-1">
                          <div className="flex items-center gap-2">
                            <Skeleton className="h-5 w-5 rounded" />
                            <Skeleton className="h-5 w-32" />
                          </div>
                          <div className="flex items-center gap-1">
                            <Skeleton className="h-4 w-24" />
                            <span>•</span>
                            <Skeleton className="h-4 w-20" />
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          <Skeleton className="h-6 w-16 rounded-full" />
                          <Skeleton className="h-8 w-8 rounded" />
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div className="flex items-center gap-3">
                        <Skeleton className="h-10 w-10 rounded-full" />
                        <div className="space-y-1">
                          <Skeleton className="h-4 w-24" />
                          <Skeleton className="h-3 w-32" />
                        </div>
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <div className="space-y-1">
                          <Skeleton className="h-3 w-16" />
                          <Skeleton className="h-4 w-20" />
                        </div>
                        <div className="space-y-1">
                          <Skeleton className="h-3 w-12" />
                          <Skeleton className="h-4 w-16" />
                        </div>
                      </div>
                      <div className="grid grid-cols-2 gap-4">
                        <div className="space-y-1">
                          <Skeleton className="h-3 w-14" />
                          <Skeleton className="h-4 w-18" />
                        </div>
                        <div className="space-y-1">
                          <Skeleton className="h-3 w-12" />
                          <Skeleton className="h-4 w-16" />
                        </div>
                      </div>
                      <div className="flex items-center justify-between pt-2">
                        <Skeleton className="h-6 w-20 rounded-full" />
                        <Skeleton className="h-8 w-20 rounded" />
                      </div>
                    </CardContent>
                  </Card>
                ))
              ) : leases.length === 0 ? (
                <div className="col-span-full text-center py-8">
                  <div className="text-muted-foreground">
                    {filters.search
                      ? "No active leases match your search criteria"
                      : "No active leases found"}
                  </div>
                </div>
              ) : (
                leases.map((lease) => (
                  <LeaseCard
                    key={lease._id}
                    lease={lease}
                    onUpdate={handleLeaseUpdate}
                    onDelete={() => {}}
                  />
                ))
              )}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Property</TableHead>
                  <TableHead>Tenant</TableHead>
                  <TableHead>Rent Amount</TableHead>
                  <TableHead>Start Date</TableHead>
                  <TableHead>End Date</TableHead>
                  <TableHead>Days Remaining</TableHead>
                  <TableHead>Urgency</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {loading && leases.length === 0 ? (
                  [...Array(8)].map((_, i) => (
                    <TableRow key={i}>
                      <TableCell>
                        <div className="flex items-center space-x-3">
                          <div className="flex-shrink-0">
                            <Skeleton className="h-8 w-8 rounded" />
                          </div>
                          <div className="space-y-1">
                            <Skeleton className="h-4 w-32" />
                            <Skeleton className="h-3 w-40" />
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-3">
                          <Skeleton className="h-8 w-8 rounded-full" />
                          <div className="space-y-1">
                            <Skeleton className="h-4 w-24" />
                            <Skeleton className="h-3 w-32" />
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="space-y-1">
                          <Skeleton className="h-4 w-20" />
                          <Skeleton className="h-3 w-16" />
                        </div>
                      </TableCell>
                      <TableCell>
                        <Skeleton className="h-4 w-20" />
                      </TableCell>
                      <TableCell>
                        <Skeleton className="h-4 w-20" />
                      </TableCell>
                      <TableCell>
                        <Skeleton className="h-4 w-16" />
                      </TableCell>
                      <TableCell>
                        <Skeleton className="h-6 w-16 rounded-full" />
                      </TableCell>
                      <TableCell className="text-right">
                        <Skeleton className="h-8 w-8 rounded ml-auto" />
                      </TableCell>
                    </TableRow>
                  ))
                ) : leases.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={8} className="text-center py-8">
                      <div className="text-muted-foreground">
                        {filters.search
                          ? "No active leases match your search criteria"
                          : "No active leases found"}
                      </div>
                    </TableCell>
                  </TableRow>
                ) : (
                  leases.map((lease) => {
                    const getDaysRemaining = () => {
                      const endDate = new Date(lease.endDate);
                      const now = new Date();
                      const diffTime = endDate.getTime() - now.getTime();
                      const diffDays = Math.ceil(
                        diffTime / (1000 * 60 * 60 * 24)
                      );
                      return diffDays;
                    };

                    const daysRemaining = getDaysRemaining();
                    const getUrgencyColor = () => {
                      if (daysRemaining <= 7) return "text-red-600";
                      if (daysRemaining <= 30) return "text-orange-600";
                      return "text-green-600";
                    };

                    return (
                      <TableRow key={lease._id}>
                        <TableCell>
                          <div className="flex items-center space-x-3">
                            <div className="flex-shrink-0">
                              <Home className="h-8 w-8 text-muted-foreground" />
                            </div>
                            <div>
                              <div className="font-medium">
                                {lease.propertyId.name}
                              </div>
                              <div className="text-sm text-muted-foreground">
                                {lease.propertyId.address.street},{" "}
                                {lease.propertyId.address.city}
                              </div>
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="flex items-center space-x-3">
                            <Avatar className="h-8 w-8">
                              <AvatarFallback>
                                {lease.tenantId?.firstName?.[0] || "T"}
                                {lease.tenantId?.lastName?.[0] || ""}
                              </AvatarFallback>
                            </Avatar>
                            <div>
                              <div className="font-medium">
                                {lease.tenantId?.firstName}{" "}
                                {lease.tenantId?.lastName}
                              </div>
                              <div className="text-sm text-muted-foreground">
                                {lease.tenantId?.email}
                              </div>
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="font-medium">
                            ${lease.terms.rentAmount.toLocaleString()}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            per month
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="text-sm">
                            {new Date(lease.startDate).toLocaleDateString()}
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="text-sm">
                            {new Date(lease.endDate).toLocaleDateString()}
                          </div>
                        </TableCell>
                        <TableCell>
                          <div
                            className={`text-sm font-medium ${getUrgencyColor()}`}
                          >
                            {daysRemaining} days
                          </div>
                        </TableCell>
                        <TableCell>
                          <Badge
                            variant={
                              daysRemaining <= 7
                                ? "destructive"
                                : daysRemaining <= 30
                                ? "default"
                                : "secondary"
                            }
                          >
                            {daysRemaining <= 7
                              ? "Urgent"
                              : daysRemaining <= 30
                              ? "Warning"
                              : "Normal"}
                          </Badge>
                        </TableCell>
                        <TableCell className="text-right">
                          <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                              <Button variant="ghost" size="sm">
                                <MoreHorizontal className="h-4 w-4" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                              <DropdownMenuItem
                                onClick={() =>
                                  router.push(`/dashboard/leases/${lease._id}`)
                                }
                              >
                                <Eye className="h-4 w-4 mr-2" />
                                View Details
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                onClick={() =>
                                  router.push(
                                    `/dashboard/leases/${lease._id}/edit`
                                  )
                                }
                              >
                                <Edit className="h-4 w-4 mr-2" />
                                Edit
                              </DropdownMenuItem>
                            </DropdownMenuContent>
                          </DropdownMenu>
                        </TableCell>
                      </TableRow>
                    );
                  })
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Pagination */}
      {pagination.pages > 1 && (
        <div className="flex items-center justify-center gap-2">
          <Button
            variant="outline"
            disabled={!pagination.hasPrev}
            onClick={() => handlePageChange(pagination.page - 1)}
          >
            Previous
          </Button>
          <span className="text-sm text-muted-foreground">
            Page {pagination.page} of {pagination.pages}
          </span>
          <Button
            variant="outline"
            disabled={!pagination.hasNext}
            onClick={() => handlePageChange(pagination.page + 1)}
          >
            Next
          </Button>
        </div>
      )}
    </div>
  );
}
