"use client";

import React, { useCallback, useEffect, useState } from "react";
import { useSession } from "next-auth/react";
import { redirect } from "next/navigation";
import { UserRole } from "@/types";
import { Button } from "@/components/ui/button";
import DocumentManagement, {
  TenantDocument,
  TenantLeaseSummary,
  normalizeTenantDocument,
} from "@/components/tenant/DocumentManagement";
import TenantDocumentUploadDialog from "@/components/tenant/TenantDocumentUploadDialog";
import { RefreshCw, Upload, Loader2 } from "lucide-react";
import { toast } from "sonner";

export default function LeasesDocumentsPage() {
  const { data: session, status } = useSession();
  const [documents, setDocuments] = useState<TenantDocument[]>([]);
  const [leases, setLeases] = useState<TenantLeaseSummary[]>([]);
  const [documentsLoading, setDocumentsLoading] = useState(true);
  const [uploadDialogOpen, setUploadDialogOpen] = useState(false);

  useEffect(() => {
    if (status === "loading") {
      return;
    }

    if (!session?.user) {
      redirect("/auth/signin");
      return;
    }

    if (session.user.role !== UserRole.TENANT) {
      redirect("/dashboard/leases");
    }
  }, [session, status]);

  const fetchLeases = useCallback(async () => {
    try {
      const response = await fetch("/api/tenant/dashboard");
      const payload = await response.json();

      if (!response.ok || !payload?.success) {
        throw new Error(payload?.message ?? "Failed to load lease details");
      }

      setLeases(payload?.data?.allLeases ?? []);
    } catch (error) {
      toast.error("Failed to load lease details", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
    }
  }, []);

  const fetchDocuments = useCallback(async () => {
    try {
      setDocumentsLoading(true);

      const response = await fetch("/api/tenant/documents");
      const payload = await response.json();

      if (!response.ok || !payload?.success) {
        throw new Error(payload?.message ?? "Failed to load documents");
      }

      const normalized: TenantDocument[] = (payload?.data?.documents ?? []).map(
        normalizeTenantDocument
      );

      setDocuments(normalized);
      return true;
    } catch (error) {
      toast.error("Failed to load documents", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
      return false;
    } finally {
      setDocumentsLoading(false);
    }
  }, []);

  useEffect(() => {
    if (status !== "authenticated" || session?.user?.role !== UserRole.TENANT) {
      return;
    }

    void fetchLeases();
    void fetchDocuments();
  }, [status, session?.user?.role, fetchLeases, fetchDocuments]);

  const handleRefresh = useCallback(async () => {
    const success = await fetchDocuments();
    if (success) {
      toast.success("Documents updated");
    }
  }, [fetchDocuments]);

  const handleOpenUpload = useCallback(() => {
    setUploadDialogOpen(true);
  }, []);

  const handleUploadComplete = useCallback(
    async () => fetchDocuments(),
    [fetchDocuments]
  );

  if (status === "loading") {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-24 w-24 border-b-2 border-primary" />
      </div>
    );
  }

  return (
    <div className="container mx-auto py-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">My Documents</h1>
          <p className="text-muted-foreground">
            Access and manage your lease-related documents
          </p>
        </div>

        <div className="flex items-center gap-2">
          <Button
            variant="outline"
            size="sm"
            className="gap-2"
            onClick={handleRefresh}
            disabled={documentsLoading}
          >
            {documentsLoading ? (
              <Loader2 className="h-4 w-4 animate-spin" />
            ) : (
              <RefreshCw className="h-4 w-4" />
            )}
            Refresh
          </Button>
          <Button size="sm" className="gap-2" onClick={handleOpenUpload}>
            <Upload className="h-4 w-4" />
            Upload Document
          </Button>
        </div>
      </div>

      <DocumentManagement
        documents={documents}
        leases={leases}
        loading={documentsLoading}
      />

      <TenantDocumentUploadDialog
        open={uploadDialogOpen}
        onOpenChange={setUploadDialogOpen}
        onUploadComplete={async (_uploadedDocuments, summary) => {
          const refreshed = await handleUploadComplete();
          if (!summary) {
            return;
          }
          const message =
            summary?.message ??
            `Uploaded ${summary.uploaded} of ${summary.total} document${
              summary.total === 1 ? "" : "s"
            }`;
          if (refreshed !== false) {
            toast.success(message);
          }
        }}
      />
    </div>
  );
}
