"use client";

import { useState, useEffect, useCallback } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { useSession } from "next-auth/react";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  FileText,
  Plus,
  Search,
  MoreHorizontal,
  Eye,
  Edit,
  Trash2,
  Download,
  RefreshCw,
  LayoutGrid,
  List,
  Calendar,
  DollarSign,
  AlertTriangle,
  CheckCircle,
  Clock,
  Send,
  CreditCard,
  CheckSquare,
  Printer,
} from "lucide-react";
import { UserRole } from "@/types";
import PaymentRecordDialog from "@/components/invoice/PaymentRecordDialog";
import BulkOperationsDialog from "@/components/invoice/BulkOperationsDialog";
import {
  printInvoice,
  downloadInvoiceAsPDF,
  type PrintableInvoice,
} from "@/lib/invoice-print";
import { normalizeInvoiceForPrint } from "@/lib/invoice/invoice-shared";
import { useLocalizationContext } from "@/components/providers/LocalizationProvider";

interface Invoice {
  _id: string;
  invoiceNumber: string;
  tenantId: {
    _id: string;
    firstName: string;
    lastName: string;
    email: string;
  };
  propertyId: {
    name: string;
    address:
      | {
          street: string;
          city: string;
          state: string;
          zipCode: string;
          country: string;
        }
      | string;
  };
  leaseId: {
    _id: string;
    startDate: string;
    endDate: string;
    propertyId: {
      name: string;
      address:
        | {
            street: string;
            city: string;
            state: string;
            zipCode: string;
            country: string;
          }
        | string;
    };
  };
  issueDate: string;
  dueDate: string;
  status: string;
  totalAmount: number;
  amountPaid: number;
  balanceRemaining: number;
  daysOverdue?: number;
  items: Array<{
    description: string;
    amount: number;
    type: string;
  }>;
  createdAt: string;
  updatedAt: string;
}

interface InvoiceQueryParams {
  page: number;
  limit: number;
  search?: string;
  status?: string;
  sortBy?: string;
  sortOrder?: "asc" | "desc";
  tenantId?: string;
  propertyId?: string;
  leaseId?: string;
}

// Helper function to format address
const formatAddress = (
  address:
    | string
    | {
        street: string;
        city: string;
        state: string;
        zipCode: string;
        country: string;
      }
    | undefined
): string => {
  if (typeof address === "string") {
    return address;
  }
  if (address && typeof address === "object") {
    const { street, city, state, zipCode } = address;
    return `${street}, ${city}, ${state} ${zipCode}`;
  }
  return "N/A";
};

export default function LeaseInvoicesPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const { data: session } = useSession();
  const { formatCurrency: formatCurrencyLocalized } = useLocalizationContext();
  const initialLeaseId = searchParams.get("leaseId") || undefined;
  const initialPropertyId = searchParams.get("propertyId") || undefined;
  const [invoices, setInvoices] = useState<Invoice[]>([]);
  const [loading, setLoading] = useState(true);
  const [paymentDialogOpen, setPaymentDialogOpen] = useState(false);
  const [bulkDialogOpen, setBulkDialogOpen] = useState(false);
  const [selectedInvoiceForPayment, setSelectedInvoiceForPayment] =
    useState<Invoice | null>(null);
  const [pagination, setPagination] = useState({
    page: 1,
    limit: 12,
    total: 0,
    pages: 0,
    hasNext: false,
    hasPrev: false,
  });
  const [filters, setFilters] = useState<InvoiceQueryParams>({
    page: 1,
    limit: 12,
    search: "",
    status: undefined,
    sortBy: "createdAt",
    sortOrder: "desc",
    ...(initialPropertyId ? { propertyId: initialPropertyId } : {}),
    ...(initialLeaseId ? { leaseId: initialLeaseId } : {}),
  });
  const [viewMode, setViewMode] = useState<"table" | "cards">("table");
  const [selectedInvoices, setSelectedInvoices] = useState<string[]>([]);
  const [stats, setStats] = useState({
    total: 0,
    scheduled: 0,
    issued: 0,
    paid: 0,
    partial: 0,
    overdue: 0,
    cancelled: 0,
    totalAmount: 0,
    paidAmount: 0,
    overdueAmount: 0,
  });

  useEffect(() => {
    fetchInvoices();
    fetchStats();
  }, [filters]);

  const fetchInvoices = async () => {
    try {
      setLoading(true);

      if (session?.user?.role === UserRole.TENANT) {
        // For tenants, use the tenant invoices API
        const response = await fetch("/api/tenant/invoices");
        const data = await response.json();

        if (data.success) {
          setInvoices(data.data.invoices || []);
          setPagination({
            page: 1,
            limit: data.data.invoices?.length || 0,
            total: data.data.invoices?.length || 0,
            pages: 1,
            hasNext: false,
            hasPrev: false,
          });

          // Update stats from tenant invoices summary
          if (data.data.summary) {
            const summary = data.data.summary;
            setStats({
              total: summary.total || 0,
              scheduled: summary.byStatus?.scheduled || 0,
              issued: summary.byStatus?.issued || 0,
              paid: summary.byStatus?.paid || 0,
              partial: summary.byStatus?.partial || 0,
              overdue: summary.byStatus?.overdue || 0,
              cancelled: summary.byStatus?.cancelled || 0,
              totalAmount: summary.totalAmount || 0,
              paidAmount: summary.amountPaid || 0,
              overdueAmount: summary.balanceRemaining || 0,
            });
          }
        } else {
          toast.error("Failed to fetch your invoices");
        }
      } else {
        // For admin/manager, use the existing invoices API
        const params = new URLSearchParams();

        Object.entries(filters).forEach(([key, value]) => {
          if (value !== undefined && value !== "") {
            params.append(key, value.toString());
          }
        });

        const response = await fetch(`/api/invoices?${params}`);
        const data = await response.json();

        if (data.success) {
          setInvoices(data.data.invoices || []);
          setPagination(data.data.pagination || pagination);
        } else {
          toast.error("Failed to fetch invoices");
        }
      }
    } catch (error) {
      toast.error("Failed to fetch invoices");
    } finally {
      setLoading(false);
    }
  };

  const fetchStats = async () => {
    try {
      // Skip fetching stats for tenants - already fetched with invoices
      if (session?.user?.role === UserRole.TENANT) {
        return;
      }

      const params = new URLSearchParams();
      if (filters.propertyId)
        params.append("propertyId", String(filters.propertyId));
      if (filters.tenantId) params.append("tenantId", String(filters.tenantId));
      if (filters.leaseId) params.append("leaseId", String(filters.leaseId));
      const url = `/api/invoices/stats${
        params.toString() ? `?${params.toString()}` : ""
      }`;
      const response = await fetch(url);
      const data = await response.json();

      if (data.success) {
        setStats(data.data);
      }
    } catch (error) {
      // Failed to fetch invoice stats
    }
  };

  const handleSearch = (value: string) => {
    setFilters((prev) => ({ ...prev, search: value, page: 1 }));
  };

  const handleStatusFilter = (status: string) => {
    setFilters((prev) => ({
      ...prev,
      status: status === "all" ? undefined : status,
      page: 1,
    }));
  };

  const handleSort = (sortBy: string, sortOrder: "asc" | "desc") => {
    setFilters((prev) => ({ ...prev, sortBy, sortOrder, page: 1 }));
  };

  const handlePageChange = (page: number) => {
    setFilters((prev) => ({ ...prev, page }));
  };

  const handleDownloadInvoice = async (inv: Invoice) => {
    try {
      // Fetch company info from display settings
      const { getCompanyInfo } = await import("@/lib/utils/company-info");
      const companyInfo = await getCompanyInfo();

      const printable = normalizeInvoiceForPrint(inv, {
        companyInfo: companyInfo || undefined,
      }) as PrintableInvoice;
      await downloadInvoiceAsPDF(printable);
      toast.success("Invoice downloaded");
    } catch (error) {
      toast.error("Failed to download invoice");
    }
  };

  const handlePrintInvoice = async (inv: Invoice) => {
    try {
      // Fetch company info from display settings
      const { getCompanyInfo } = await import("@/lib/utils/company-info");
      const companyInfo = await getCompanyInfo();

      const printable = normalizeInvoiceForPrint(inv, {
        companyInfo: companyInfo || undefined,
      }) as PrintableInvoice;
      printInvoice(printable);
    } catch (error) {
      toast.error("Failed to print invoice");
    }
  };

  const handleSendInvoice = async (inv: Invoice) => {
    try {
      const body = {
        leaseId: inv.leaseId?._id || (inv as any).leaseId,
        to: inv.tenantId?.email,
        invoiceNumber: inv.invoiceNumber,
      };
      const res = await fetch("/api/invoices/email", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(body),
      });
      const data = await res.json();
      if (res.ok && data.success) {
        toast.success("Invoice emailed to tenant");
      } else {
        throw new Error(data.error || "Failed to send invoice email");
      }
    } catch {
      toast.error("Failed to email invoice");
    }
  };

  const formatCurrency = useCallback(
    (amount: number) => formatCurrencyLocalized(amount ?? 0),
    [formatCurrencyLocalized]
  );

  const getStatusBadge = (status: string, daysOverdue?: number) => {
    const statusConfig = {
      scheduled: {
        variant: "secondary" as const,
        label: "Scheduled",
        icon: Calendar,
      },
      issued: { variant: "outline" as const, label: "Issued", icon: Send },
      paid: { variant: "default" as const, label: "Paid", icon: CheckCircle },
      partial: { variant: "secondary" as const, label: "Partial", icon: Clock },
      overdue: {
        variant: "destructive" as const,
        label: `Overdue${daysOverdue ? ` (${daysOverdue}d)` : ""}`,
        icon: AlertTriangle,
      },
      cancelled: {
        variant: "outline" as const,
        label: "Cancelled",
        icon: Trash2,
      },
    };

    const config = statusConfig[status as keyof typeof statusConfig] || {
      variant: "outline" as const,
      label: status,
      icon: FileText,
    };

    return (
      <Badge variant={config.variant} className="flex items-center gap-1">
        <config.icon className="h-3 w-3" />
        {config.label}
      </Badge>
    );
  };

  const handleProcessLateFees = async () => {
    try {
      const response = await fetch("/api/invoices/late-fees", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ dryRun: false }),
      });

      const data = await response.json();
      if (data.success) {
        toast.success(`Applied late fees to ${data.data.feesApplied} invoices`);
        fetchInvoices();
        fetchStats();
      } else {
        toast.error("Failed to process late fees");
      }
    } catch (error) {
      toast.error("Failed to process late fees");
    }
  };

  const handleBulkAction = async (action: string) => {
    if (selectedInvoices.length === 0) {
      toast.error("Please select invoices first");
      return;
    }

    // Open bulk operations dialog instead of direct action
    setBulkDialogOpen(true);
  };

  const handleRecordPayment = (invoice: Invoice) => {
    setSelectedInvoiceForPayment(invoice);
    setPaymentDialogOpen(true);
  };

  const handleViewInvoice = (invoiceId: string) => {
    router.push(`/dashboard/leases/invoices/${invoiceId}`);
  };

  const handleEditInvoice = (invoiceId: string) => {
    router.push(`/dashboard/leases/invoices/${invoiceId}/edit`);
  };

  if (loading && invoices.length === 0) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <Skeleton className="h-8 w-48" />
            <Skeleton className="h-4 w-64 mt-2" />
          </div>
          <Skeleton className="h-10 w-32" />
        </div>

        {/* Stats Cards Skeleton - Hidden for Tenants */}
        {session?.user?.role !== UserRole.TENANT && (
          <div className="grid grid-cols-1 md:grid-cols-4 lg:grid-cols-7 gap-4">
            {[...Array(7)].map((_, i) => (
              <Card key={i} className="py-0">
                <CardContent className="p-4">
                  <div className="flex items-center justify-between">
                    <div className="space-y-2">
                      <Skeleton className="h-4 w-12" />
                      <Skeleton className="h-8 w-16" />
                    </div>
                    <Skeleton className="h-8 w-8 rounded" />
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}

        {/* Table Skeleton */}
        <Card>
          <CardHeader>
            <Skeleton className="h-6 w-32" />
            <Skeleton className="h-4 w-48" />
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex gap-4">
                <Skeleton className="h-10 flex-1" />
                <Skeleton className="h-10 w-32" />
                <Skeleton className="h-10 w-32" />
              </div>
              <div className="space-y-2">
                {[...Array(5)].map((_, i) => (
                  <Skeleton key={i} className="h-16 w-full" />
                ))}
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">
            {session?.user?.role === UserRole.TENANT
              ? "My Invoices"
              : "Lease Invoices"}
          </h1>
          <p className="text-muted-foreground">
            {session?.user?.role === UserRole.TENANT
              ? "View and manage your lease invoices and payments"
              : "Automatically generated invoices from lease agreements"}
          </p>
        </div>
        <div className="flex gap-2">
          <Button
            variant="outline"
            size="sm"
            onClick={() => {
              fetchInvoices();
              fetchStats();
              toast.success("Invoice list refreshed");
            }}
          >
            <RefreshCw className="mr-2 h-4 w-4" />
            Refresh
          </Button>
          {session?.user?.role !== UserRole.TENANT && (
            <>
              <Button variant="outline" size="sm" onClick={handleProcessLateFees}>
                <AlertTriangle className="mr-2 h-4 w-4" />
                Process Late Fees
              </Button>
              <Button size="sm" onClick={() => router.push("/dashboard/leases/new")}>
                <Plus className="mr-2 h-4 w-4" />
                Create Lease
              </Button>
            </>
          )}
        </div>
      </div>

      {/* Stats Cards - Hidden for Tenants */}
      {session?.user?.role !== UserRole.TENANT && (
        <div className="grid grid-cols-1 md:grid-cols-4 lg:grid-cols-7 gap-4">
          <Card className="py-0">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">
                    Total
                  </p>
                  <p className="text-2xl font-bold">{stats.total}</p>
                </div>
                <FileText className="h-8 w-8 text-muted-foreground" />
              </div>
            </CardContent>
          </Card>

          <Card className="py-0">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">
                    Paid
                  </p>
                  <p className="text-2xl font-bold text-green-600">
                    {stats.paid}
                  </p>
                </div>
                <div className="h-8 w-8 rounded-full bg-green-100 flex items-center justify-center">
                  <CheckCircle className="h-4 w-4 text-green-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="py-0">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">
                    Issued
                  </p>
                  <p className="text-2xl font-bold text-blue-600">
                    {stats.issued}
                  </p>
                </div>
                <div className="h-8 w-8 rounded-full bg-blue-100 flex items-center justify-center">
                  <Send className="h-4 w-4 text-blue-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="py-0">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">
                    Overdue
                  </p>
                  <p className="text-2xl font-bold text-red-600">
                    {stats.overdue}
                  </p>
                </div>
                <div className="h-8 w-8 rounded-full bg-red-100 flex items-center justify-center">
                  <AlertTriangle className="h-4 w-4 text-red-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="py-0">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">
                    Partial
                  </p>
                  <p className="text-2xl font-bold text-yellow-600">
                    {stats.partial}
                  </p>
                </div>
                <div className="h-8 w-8 rounded-full bg-yellow-100 flex items-center justify-center">
                  <Clock className="h-4 w-4 text-yellow-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="py-0">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">
                    Total Value
                  </p>
                  <p className="text-lg font-bold">
                    {formatCurrency(stats.totalAmount)}
                  </p>
                </div>
                <DollarSign className="h-8 w-8 text-muted-foreground" />
              </div>
            </CardContent>
          </Card>

          <Card className="py-0">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">
                    Collected
                  </p>
                  <p className="text-lg font-bold text-green-600">
                    {formatCurrency(stats.paidAmount)}
                  </p>
                </div>
                <div className="h-8 w-8 rounded-full bg-green-100 flex items-center justify-center">
                  <DollarSign className="h-4 w-4 text-green-600" />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Filters and Actions */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Invoice List</CardTitle>
              <CardDescription>
                View and manage all lease-related invoices
              </CardDescription>
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() =>
                  setViewMode(viewMode === "table" ? "cards" : "table")
                }
              >
                {viewMode === "table" ? (
                  <LayoutGrid className="h-4 w-4" />
                ) : (
                  <List className="h-4 w-4" />
                )}
              </Button>
              <Button variant="outline" size="sm" onClick={fetchInvoices}>
                <RefreshCw className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col gap-4 md:flex-row md:items-center md:justify-between">
            <div className="flex flex-1 gap-4">
              <div className="relative flex-1 max-w-sm">
                <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Search invoices..."
                  value={filters.search}
                  onChange={(e) => handleSearch(e.target.value)}
                  className="pl-8"
                />
              </div>

              <Select
                value={filters.status || "all"}
                onValueChange={handleStatusFilter}
              >
                <SelectTrigger className="w-[180px]">
                  <SelectValue placeholder="Filter by status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Statuses</SelectItem>
                  <SelectItem value="scheduled">Scheduled</SelectItem>
                  <SelectItem value="issued">Issued</SelectItem>
                  <SelectItem value="partial">Partial</SelectItem>
                  <SelectItem value="paid">Paid</SelectItem>
                  <SelectItem value="overdue">Overdue</SelectItem>
                  <SelectItem value="cancelled">Cancelled</SelectItem>
                </SelectContent>
              </Select>

              <Select
                value={`${filters.sortBy}-${filters.sortOrder}`}
                onValueChange={(value) => {
                  const [sortBy, sortOrder] = value.split("-");
                  handleSort(sortBy, sortOrder as "asc" | "desc");
                }}
              >
                <SelectTrigger className="w-[180px]">
                  <SelectValue placeholder="Sort by" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="createdAt-desc">Newest First</SelectItem>
                  <SelectItem value="createdAt-asc">Oldest First</SelectItem>
                  <SelectItem value="dueDate-asc">
                    Due Date (Earliest)
                  </SelectItem>
                  <SelectItem value="dueDate-desc">
                    Due Date (Latest)
                  </SelectItem>
                  <SelectItem value="totalAmount-desc">
                    Amount (High to Low)
                  </SelectItem>
                  <SelectItem value="totalAmount-asc">
                    Amount (Low to High)
                  </SelectItem>
                  <SelectItem value="status-asc">Status (A-Z)</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {selectedInvoices.length > 0 && (
              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">
                  {selectedInvoices.length} selected
                </span>
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="outline" size="sm">
                      Bulk Actions
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent>
                    <DropdownMenuItem onClick={() => setBulkDialogOpen(true)}>
                      <CheckSquare className="mr-2 h-4 w-4" />
                      Bulk Operations
                    </DropdownMenuItem>
                    <DropdownMenuSeparator />
                    <DropdownMenuItem onClick={() => handleBulkAction("send")}>
                      <Send className="mr-2 h-4 w-4" />
                      Send Invoices
                    </DropdownMenuItem>
                    <DropdownMenuItem
                      onClick={() => handleBulkAction("download")}
                    >
                      <Download className="mr-2 h-4 w-4" />
                      Download PDFs
                    </DropdownMenuItem>
                    <DropdownMenuItem
                      onClick={() => handleBulkAction("mark_paid")}
                    >
                      <CreditCard className="mr-2 h-4 w-4" />
                      Mark as Paid
                    </DropdownMenuItem>
                    <DropdownMenuSeparator />
                    <DropdownMenuItem
                      onClick={() => handleBulkAction("cancel")}
                      className="text-red-600"
                    >
                      <Trash2 className="mr-2 h-4 w-4" />
                      Cancel Invoices
                    </DropdownMenuItem>
                  </DropdownMenuContent>
                </DropdownMenu>
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Invoice Table */}
      <Card className="p-4">
        <CardContent className="p-0">
          {loading ? (
            <div className="p-6">
              <div className="space-y-4">
                {[...Array(5)].map((_, i) => (
                  <div key={i} className="flex items-center space-x-4">
                    <Skeleton className="h-12 w-12 rounded" />
                    <div className="space-y-2 flex-1">
                      <Skeleton className="h-4 w-[250px]" />
                      <Skeleton className="h-4 w-[200px]" />
                    </div>
                    <Skeleton className="h-4 w-[100px]" />
                    <Skeleton className="h-4 w-[80px]" />
                  </div>
                ))}
              </div>
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  {/* Checkbox column - Hidden for Tenants */}
                  {session?.user?.role !== UserRole.TENANT && (
                    <TableHead className="w-12">
                      <input
                        type="checkbox"
                        checked={
                          selectedInvoices.length === invoices.length &&
                          invoices.length > 0
                        }
                        onChange={(e) => {
                          if (e.target.checked) {
                            setSelectedInvoices(invoices.map((inv) => inv._id));
                          } else {
                            setSelectedInvoices([]);
                          }
                        }}
                        className="rounded border-gray-300"
                      />
                    </TableHead>
                  )}
                  <TableHead>Invoice #</TableHead>
                  {session?.user?.role !== UserRole.TENANT && (
                    <TableHead>Tenant</TableHead>
                  )}
                  <TableHead>Property</TableHead>
                  <TableHead>Issue Date</TableHead>
                  <TableHead>Due Date</TableHead>
                  <TableHead>Amount</TableHead>
                  <TableHead>Paid</TableHead>
                  <TableHead>Balance</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead className="w-12"></TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {invoices.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={11} className="text-center py-8">
                      <div className="flex flex-col items-center gap-2">
                        <FileText className="h-8 w-8 text-muted-foreground" />
                        <p className="text-muted-foreground">
                          No invoices found
                        </p>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => router.push("/dashboard/leases/new")}
                        >
                          Create your first lease
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ) : (
                  invoices.map((invoice) => (
                    <TableRow key={invoice._id}>
                      {/* Checkbox cell - Hidden for Tenants */}
                      {session?.user?.role !== UserRole.TENANT && (
                        <TableCell>
                          <input
                            type="checkbox"
                            checked={selectedInvoices.includes(invoice._id)}
                            onChange={(e) => {
                              if (e.target.checked) {
                                setSelectedInvoices((prev) => [
                                  ...prev,
                                  invoice._id,
                                ]);
                              } else {
                                setSelectedInvoices((prev) =>
                                  prev.filter((id) => id !== invoice._id)
                                );
                              }
                            }}
                            className="rounded border-gray-300"
                          />
                        </TableCell>
                      )}
                      <TableCell>
                        <div className="font-mono text-sm">
                          {invoice.invoiceNumber}
                        </div>
                      </TableCell>
                      {session?.user?.role !== UserRole.TENANT && (
                        <TableCell>
                          <div className="flex items-center gap-3">
                            <Avatar className="h-8 w-8">
                              <AvatarFallback>
                                {invoice.tenantId?.firstName?.[0] || "T"}
                                {invoice.tenantId?.lastName?.[0] || "U"}
                              </AvatarFallback>
                            </Avatar>
                            <div>
                              <div className="font-medium">
                                {invoice.tenantId?.firstName || "Unknown"}{" "}
                                {invoice.tenantId?.lastName || "User"}
                              </div>
                              <div className="text-sm text-muted-foreground">
                                {invoice.tenantId?.email || "No email"}
                              </div>
                            </div>
                          </div>
                        </TableCell>
                      )}
                      <TableCell>
                        <div>
                          <div className="font-medium">
                            {invoice.leaseId?.propertyId?.name ||
                              invoice.propertyId?.name ||
                              "N/A"}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            {formatAddress(
                              invoice.leaseId?.propertyId?.address ||
                                invoice.propertyId?.address
                            )}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        {new Date(invoice.issueDate).toLocaleDateString()}
                      </TableCell>
                      <TableCell>
                        <div
                          className={`${
                            new Date(invoice.dueDate) < new Date() &&
                            invoice.status !== "paid"
                              ? "text-red-600 font-medium"
                              : ""
                          }`}
                        >
                          {new Date(invoice.dueDate).toLocaleDateString()}
                        </div>
                      </TableCell>
                      <TableCell className="font-medium">
                        {formatCurrency(invoice.totalAmount)}
                      </TableCell>
                      <TableCell className="text-green-600">
                        {formatCurrency(invoice.amountPaid)}
                      </TableCell>
                      <TableCell
                        className={`font-medium ${
                          invoice.balanceRemaining > 0
                            ? "text-red-600"
                            : "text-green-600"
                        }`}
                      >
                        {formatCurrency(invoice.balanceRemaining)}
                      </TableCell>
                      <TableCell>
                        {getStatusBadge(invoice.status, invoice.daysOverdue)}
                      </TableCell>
                      <TableCell>
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem
                              onClick={() => handleViewInvoice(invoice._id)}
                            >
                              <Eye className="mr-2 h-4 w-4" />
                              View Details
                            </DropdownMenuItem>

                            {/* Tenant Payment Button */}
                            {session?.user?.role === UserRole.TENANT &&
                              invoice.status !== "paid" &&
                              invoice.balanceRemaining > 0 && (
                                <>
                                  <DropdownMenuSeparator />
                                  <DropdownMenuItem
                                    onClick={() =>
                                      router.push(
                                        `/dashboard/leases/invoices/${invoice._id}`
                                      )
                                    }
                                    className="text-blue-600 font-medium"
                                  >
                                    <CreditCard className="mr-2 h-4 w-4" />
                                    Pay Now
                                  </DropdownMenuItem>
                                  <DropdownMenuSeparator />
                                </>
                              )}

                            <DropdownMenuItem
                              onClick={() => handlePrintInvoice(invoice)}
                            >
                              <Printer className="mr-2 h-4 w-4" />
                              Print Invoice
                            </DropdownMenuItem>
                            <DropdownMenuItem
                              onClick={() => handleDownloadInvoice(invoice)}
                            >
                              <Download className="mr-2 h-4 w-4" />
                              Download PDF
                            </DropdownMenuItem>

                            {/* Admin/Manager Only Actions */}
                            {session?.user?.role !== UserRole.TENANT && (
                              <>
                                {invoice.status !== "paid" && (
                                  <DropdownMenuItem
                                    onClick={() => handleRecordPayment(invoice)}
                                  >
                                    <CreditCard className="mr-2 h-4 w-4" />
                                    Record Payment
                                  </DropdownMenuItem>
                                )}
                                <DropdownMenuItem
                                  onClick={() => handleSendInvoice(invoice)}
                                >
                                  <Send className="mr-2 h-4 w-4" />
                                  Send to Tenant
                                </DropdownMenuItem>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem
                                  onClick={() => handleEditInvoice(invoice._id)}
                                >
                                  <Edit className="mr-2 h-4 w-4" />
                                  Edit Invoice
                                </DropdownMenuItem>
                                <DropdownMenuItem className="text-red-600">
                                  <Trash2 className="mr-2 h-4 w-4" />
                                  Cancel Invoice
                                </DropdownMenuItem>
                              </>
                            )}
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  ))
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Pagination */}
      {pagination.pages > 1 && (
        <div className="flex items-center justify-between">
          <div className="text-sm text-muted-foreground">
            Showing {(pagination.page - 1) * pagination.limit + 1} to{" "}
            {Math.min(pagination.page * pagination.limit, pagination.total)} of{" "}
            {pagination.total} invoices
          </div>
          <div className="flex items-center gap-2">
            <Button
              variant="outline"
              size="sm"
              onClick={() => handlePageChange(pagination.page - 1)}
              disabled={!pagination.hasPrev}
            >
              Previous
            </Button>
            <div className="flex items-center gap-1">
              {[...Array(Math.min(5, pagination.pages))].map((_, i) => {
                const page = i + 1;
                return (
                  <Button
                    key={page}
                    variant={page === pagination.page ? "default" : "outline"}
                    size="sm"
                    onClick={() => handlePageChange(page)}
                    className="w-8 h-8 p-0"
                  >
                    {page}
                  </Button>
                );
              })}
            </div>
            <Button
              variant="outline"
              size="sm"
              onClick={() => handlePageChange(pagination.page + 1)}
              disabled={!pagination.hasNext}
            >
              Next
            </Button>
          </div>
        </div>
      )}

      {/* Payment Record Dialog */}
      <PaymentRecordDialog
        open={paymentDialogOpen}
        onOpenChange={setPaymentDialogOpen}
        invoice={selectedInvoiceForPayment}
        onPaymentRecorded={() => {
          fetchInvoices();
          setSelectedInvoiceForPayment(null);
        }}
      />

      {/* Bulk Operations Dialog */}
      <BulkOperationsDialog
        open={bulkDialogOpen}
        onOpenChange={setBulkDialogOpen}
        selectedInvoices={selectedInvoices}
        onOperationComplete={() => {
          fetchInvoices();
          setSelectedInvoices([]);
        }}
      />
    </div>
  );
}
