"use client";

import React, { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { redirect } from "next/navigation";
import { UserRole } from "@/types";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { toast } from "sonner";
import LeaseTable from "@/components/tenant/LeaseTable";
import { RefreshCw, Search, Home, Calendar, DollarSign } from "lucide-react";

interface Lease {
  _id: string;
  propertyId: {
    _id: string;
    name: string;
    address: {
      street: string;
      city: string;
      state: string;
      zipCode: string;
    };
  };
  unit?: {
    unitNumber: string;
    bedrooms: number;
    bathrooms: number;
    rentAmount: number;
  };
  startDate: string;
  endDate: string;
  status: string;
  terms: {
    rentAmount: number;
    securityDeposit: number;
    leaseDuration: number;
  };
  renewalOptions?: {
    available: boolean;
    deadline: string;
  };
}

export default function MyLeasesPage() {
  const { data: session, status } = useSession();
  const [leases, setLeases] = useState<Lease[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState("all");

  // Show loading state while session is being fetched
  if (status === "loading") {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-primary"></div>
      </div>
    );
  }

  // Redirect if not authenticated
  if (status === "unauthenticated") {
    redirect("/auth/signin");
  }

  // Only allow tenants to access this page
  if (session?.user?.role !== UserRole.TENANT) {
    redirect("/dashboard");
  }

  useEffect(() => {
    if (session?.user) {
      fetchLeases();
    }
  }, [session]);

  const fetchLeases = async () => {
    try {
      setLoading(true);
      const response = await fetch("/api/tenant/dashboard");
      const data = await response.json();

      if (data.success) {
        setLeases(data.data.allLeases || []);
      } else {
        toast.error("Failed to load leases");
      }
    } catch (error) {
      toast.error("Failed to load leases");
    } finally {
      setLoading(false);
    }
  };

  const handleLeaseAction = (action: string, lease: Lease) => {
    switch (action) {
      case "view-details":
        toast.info("Opening lease details...");
        // Navigate to lease details page
        break;
      case "download-agreement":
        toast.info("Downloading lease agreement...");
        // Implement download functionality
        break;
      case "contact-manager":
        toast.info("Opening communication with property manager...");
        // Navigate to communication center
        break;
      case "request-renewal":
        toast.info("Opening renewal request...");
        // Implement renewal request functionality
        break;
      default:
      // Unknown lease action
    }
  };

  const filteredLeases = leases.filter((lease) => {
    const matchesSearch =
      lease.propertyId.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      lease.propertyId.address.street
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      lease.propertyId.address.city
        .toLowerCase()
        .includes(searchTerm.toLowerCase());

    const matchesStatus =
      statusFilter === "all" || lease.status === statusFilter;

    return matchesSearch && matchesStatus;
  });

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const getLeaseStats = () => {
    const now = new Date();
    const stats = {
      total: filteredLeases.length,
      active: filteredLeases.filter((lease) => lease.status === "active")
        .length,
      expiring: filteredLeases.filter((lease) => {
        const endDate = new Date(lease.endDate);
        const daysUntilExpiration = Math.ceil(
          (endDate.getTime() - now.getTime()) / (1000 * 60 * 60 * 24)
        );
        return daysUntilExpiration <= 60 && daysUntilExpiration > 0;
      }).length,
      totalRent: filteredLeases.reduce(
        (sum, lease) =>
          sum + (lease.unit?.rentAmount || lease.terms.rentAmount),
        0
      ),
    };

    return stats;
  };

  const leaseStats = getLeaseStats();

  return (
    <div className="container mx-auto py-6 space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">My Leases</h1>
            <p className="text-muted-foreground">
              View and manage all your active and past leases
            </p>
          </div>
        </div>
        <Button
          variant="outline"
          size="sm"
          onClick={fetchLeases}
          className="gap-2"
        >
          <RefreshCw className="h-4 w-4" />
          Refresh
        </Button>
      </div>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Leases</CardTitle>
            <Home className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{leaseStats.total}</div>
            <p className="text-xs text-muted-foreground">
              {leaseStats.active} currently active
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Active Leases</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {leaseStats.active}
            </div>
            <p className="text-xs text-muted-foreground">Currently occupied</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Expiring Soon</CardTitle>
            <Calendar className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">
              {leaseStats.expiring}
            </div>
            <p className="text-xs text-muted-foreground">Within 60 days</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">
              Total Monthly Rent
            </CardTitle>
            <DollarSign className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(leaseStats.totalRent)}
            </div>
            <p className="text-xs text-muted-foreground">
              Combined rent amount
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Filters */}
      <div className="flex flex-col sm:flex-row gap-4 items-center justify-between">
        <div className="flex flex-1 items-center space-x-2">
          <div className="relative flex-1 max-w-sm">
            <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search properties..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-8"
            />
          </div>
          <Select value={statusFilter} onValueChange={setStatusFilter}>
            <SelectTrigger className="w-32">
              <SelectValue placeholder="Status" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Status</SelectItem>
              <SelectItem value="active">Active</SelectItem>
              <SelectItem value="pending">Pending</SelectItem>
              <SelectItem value="expired">Expired</SelectItem>
              <SelectItem value="terminated">Terminated</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Leases Table */}
      {loading ? (
        <Card>
          <CardHeader>
            <CardTitle>Loading Leases...</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {[...Array(5)].map((_, i) => (
                <div key={i} className="h-16 bg-muted rounded animate-pulse" />
              ))}
            </div>
          </CardContent>
        </Card>
      ) : (
        <LeaseTable leases={filteredLeases} onLeaseAction={handleLeaseAction} />
      )}
    </div>
  );
}
