"use client";

import { useState, useEffect, useCallback, useMemo } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import Link from "next/link";
import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { Input } from "@/components/ui/input";
import { Checkbox } from "@/components/ui/checkbox";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import { Textarea } from "@/components/ui/textarea";
import {
  AlertTriangle,
  Clock,
  User,
  Plus,
  RefreshCw,
  Filter,
  Search,
  TrendingUp,
  AlertCircle,
  CheckCircle,
  XCircle,
  Zap,
  Timer,
  List,
  LayoutGrid,
  MoreHorizontal,
  UserPlus,
  FileText,
  Download,
  Settings,
  Eye,
  Edit,
  ExternalLink,
} from "lucide-react";
import { toast } from "sonner";
import { formatDistanceToNow } from "date-fns";
import { MaintenanceStatus } from "@/types";
import { formatAddress, cn } from "@/lib/utils";
import EmergencyCard from "@/components/emergency/EmergencyCard";

interface EmergencyRequest {
  _id: string;
  title: string;
  description: string;
  priority: string;
  status: MaintenanceStatus;
  category: string;
  createdAt: string;
  updatedAt: string;
  hoursSinceCreation: number;
  isOverdue: boolean;
  urgencyLevel: "normal" | "overdue" | "critical" | "completed";
  property: {
    _id: string;
    name: string;
    address:
      | string
      | {
          street: string;
          city: string;
          state: string;
          zipCode: string;
          country: string;
        };
  };
  tenant: {
    _id: string;
    firstName: string;
    lastName: string;
    email: string;
    phone: string;
    avatar?: string;
  };
  assignedUser?: {
    _id: string;
    firstName: string;
    lastName: string;
    email: string;
    avatar?: string;
    phone?: string;
  };
  estimatedCost?: number;
  actualCost?: number;
  scheduledDate?: string;
  completedAt?: string;
  images?: string[];
  notes?: string;
  escalationLevel?: number;
  lastContactedAt?: string;
}

interface EmergencyStats {
  total: number;
  active: number;
  overdue: number;
  completed: number;
  unassigned: number;
  avgResponseTime: number;
  critical: number;
  escalated: number;
  todayCreated: number;
  weekCreated: number;
}

interface BulkAction {
  id: string;
  label: string;
  icon: React.ComponentType<any>;
  action: (selectedIds: string[]) => void;
  requiresConfirmation?: boolean;
  confirmationMessage?: string;
}

interface QuickAssignData {
  requestId: string;
  assigneeId: string;
  notes?: string;
}

interface EmergencyPageData {
  requests: EmergencyRequest[];
  pagination: {
    page: number;
    limit: number;
    total: number;
    pages: number;
  };
  statistics: EmergencyStats;
}

export default function EmergencyMaintenancePage() {
  const router = useRouter();
  const searchParams = useSearchParams();

  const [data, setData] = useState<EmergencyPageData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [refreshing, setRefreshing] = useState(false);

  // Filter states
  const [searchTerm, setSearchTerm] = useState(
    searchParams.get("search") || ""
  );
  const [statusFilter, setStatusFilter] = useState(
    searchParams.get("status") || "all"
  );
  const [responseTimeFilter, setResponseTimeFilter] = useState(
    searchParams.get("responseTime") || "all"
  );
  const [propertyFilter, setPropertyFilter] = useState(
    searchParams.get("propertyId") || ""
  );
  const [assignedFilter, setAssignedFilter] = useState(
    searchParams.get("assignedTo") || "all"
  );
  const [viewMode, setViewMode] = useState<"table" | "cards">("table");

  // Enhanced state management
  const [selectedRequests, setSelectedRequests] = useState<string[]>([]);
  const [showBulkActions, setShowBulkActions] = useState(false);
  const [quickAssignDialog, setQuickAssignDialog] = useState(false);
  const [escalateDialog, setEscalateDialog] = useState(false);
  const [selectedRequestForAction, setSelectedRequestForAction] = useState<
    string | null
  >(null);
  const [assigneeOptions, setAssigneeOptions] = useState<
    Array<{ id: string; name: string; email: string }>
  >([]);
  const [bulkActionLoading, setBulkActionLoading] = useState(false);
  const [autoRefresh, setAutoRefresh] = useState(true);
  const [sortBy, setSortBy] = useState("createdAt");
  const [sortOrder, setSortOrder] = useState<"asc" | "desc">("desc");
  const [showFilters, setShowFilters] = useState(false);

  // Enhanced data fetching with better error handling and caching
  const fetchEmergencyRequests = useCallback(async () => {
    try {
      const params = new URLSearchParams();
      if (searchTerm) params.set("search", searchTerm);
      if (statusFilter !== "all") params.set("status", statusFilter);
      if (responseTimeFilter !== "all")
        params.set("responseTime", responseTimeFilter);
      if (propertyFilter) params.set("propertyId", propertyFilter);
      if (assignedFilter !== "all") params.set("assignedTo", assignedFilter);
      if (sortBy) params.set("sortBy", sortBy);
      if (sortOrder) params.set("sortOrder", sortOrder);

      const response = await fetch(
        `/api/maintenance/emergency?${params.toString()}`,
        {
          headers: {
            "Cache-Control": "no-cache",
          },
        }
      );
      const result = await response.json();

      if (!response.ok) {
        throw new Error(result.error || "Failed to fetch emergency requests");
      }

      setData(result.data);
      setError(null);

      // Clear selections when data changes
      setSelectedRequests([]);
    } catch (err) {
      setError(err instanceof Error ? err.message : "An error occurred");
      toast.error("Failed to load emergency requests");
    } finally {
      setLoading(false);
      setRefreshing(false);
    }
  }, [
    searchTerm,
    statusFilter,
    responseTimeFilter,
    propertyFilter,
    assignedFilter,
    sortBy,
    sortOrder,
  ]);

  // Fetch available assignees for quick assignment
  const fetchAssignees = useCallback(async () => {
    try {
      const response = await fetch("/api/users?role=manager&limit=50");
      const result = await response.json();

      if (response.ok && result.users) {
        setAssigneeOptions(
          result.users.map((user: any) => ({
            id: user._id,
            name: `${user.firstName} ${user.lastName}`,
            email: user.email,
          }))
        );
      }
    } catch {
      toast.error("Unable to load assignee options");
    }
  }, []);

  const handleRefresh = () => {
    setRefreshing(true);
    fetchEmergencyRequests();
  };

  // Bulk action handlers
  const handleSelectAll = (checked: boolean) => {
    if (checked && data?.requests) {
      setSelectedRequests(data.requests.map((req) => req._id));
    } else {
      setSelectedRequests([]);
    }
  };

  const handleSelectRequest = (requestId: string, checked: boolean) => {
    if (checked) {
      setSelectedRequests((prev) => [...prev, requestId]);
    } else {
      setSelectedRequests((prev) => prev.filter((id) => id !== requestId));
    }
  };

  const handleBulkAssign = async (assigneeId: string) => {
    if (selectedRequests.length === 0) return;

    setBulkActionLoading(true);
    try {
      const response = await fetch("/api/maintenance/bulk-assign", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          requestIds: selectedRequests,
          assigneeId,
        }),
      });

      if (response.ok) {
        toast.success(
          `Assigned ${selectedRequests.length} requests successfully`
        );
        setSelectedRequests([]);
        fetchEmergencyRequests();
      } else {
        throw new Error("Failed to assign requests");
      }
    } catch {
      toast.error("Failed to assign requests");
    } finally {
      setBulkActionLoading(false);
    }
  };

  const handleBulkStatusUpdate = async (status: MaintenanceStatus) => {
    if (selectedRequests.length === 0) return;

    setBulkActionLoading(true);
    try {
      const response = await fetch("/api/maintenance/bulk-status", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          requestIds: selectedRequests,
          status,
        }),
      });

      if (response.ok) {
        toast.success(
          `Updated ${selectedRequests.length} requests successfully`
        );
        setSelectedRequests([]);
        fetchEmergencyRequests();
      } else {
        throw new Error("Failed to update requests");
      }
    } catch {
      toast.error("Failed to update requests");
    } finally {
      setBulkActionLoading(false);
    }
  };

  const handleEscalate = async (requestId: string, notes?: string) => {
    try {
      const response = await fetch(`/api/maintenance/emergency/escalate`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          requestId,
          notes,
        }),
      });

      if (response.ok) {
        toast.success("Request escalated successfully");
        fetchEmergencyRequests();
      } else {
        throw new Error("Failed to escalate request");
      }
    } catch {
      toast.error("Failed to escalate request");
    }
  };

  const handleQuickContact = async (
    requestId: string,
    method: "phone" | "email"
  ) => {
    const request = data?.requests.find((r) => r._id === requestId);
    if (!request) return;

    if (method === "phone" && request.tenant.phone) {
      window.open(`tel:${request.tenant.phone}`);
    } else if (method === "email" && request.tenant.email) {
      window.open(
        `mailto:${request.tenant.email}?subject=Emergency Maintenance Request - ${request.title}`
      );
    }

    // Log the contact attempt
    try {
      await fetch(`/api/maintenance/${requestId}/contact`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          method,
          contactedAt: new Date().toISOString(),
        }),
      });
    } catch {
      toast.warning("Unable to log the contact attempt");
    }
  };

  const handleDeleteRequest = useCallback(
    async (requestId: string) => {
      if (!requestId) {
        return;
      }

      try {
        const response = await fetch(`/api/maintenance/${requestId}`, {
          method: "DELETE",
        });

        const isJson = response.headers
          .get("content-type")
          ?.includes("application/json");
        const payload = isJson ? await response.json() : null;

        if (!response.ok) {
          throw new Error(
            payload?.error || "Failed to delete emergency request"
          );
        }

        toast.success("Emergency request deleted");
        setRefreshing(true);
        setSelectedRequests((prev) =>
          prev.filter((selectedId) => selectedId !== requestId)
        );
        await fetchEmergencyRequests();
      } catch (err) {
        toast.error(
          err instanceof Error
            ? err.message
            : "Failed to delete emergency request"
        );
      }
    },
    [fetchEmergencyRequests]
  );

  const getUrgencyBadge = (urgencyLevel: string, isOverdue: boolean) => {
    if (urgencyLevel === "critical") {
      return (
        <Badge variant="destructive" className="animate-pulse">
          <AlertTriangle className="w-3 h-3 mr-1" />
          Critical
        </Badge>
      );
    }
    if (urgencyLevel === "overdue" || isOverdue) {
      return (
        <Badge variant="destructive">
          <Clock className="w-3 h-3 mr-1" />
          Overdue
        </Badge>
      );
    }
    if (urgencyLevel === "completed") {
      return (
        <Badge variant="default" className="bg-green-100 text-green-800">
          <CheckCircle className="w-3 h-3 mr-1" />
          Completed
        </Badge>
      );
    }
    return (
      <Badge variant="secondary">
        <Timer className="w-3 h-3 mr-1" />
        Normal
      </Badge>
    );
  };

  const getStatusBadge = (status: MaintenanceStatus) => {
    const statusConfig = {
      [MaintenanceStatus.SUBMITTED]: {
        variant: "secondary" as const,
        icon: AlertCircle,
      },
      [MaintenanceStatus.ASSIGNED]: { variant: "default" as const, icon: User },
      [MaintenanceStatus.IN_PROGRESS]: {
        variant: "default" as const,
        icon: RefreshCw,
      },
      [MaintenanceStatus.COMPLETED]: {
        variant: "default" as const,
        icon: CheckCircle,
      },
      [MaintenanceStatus.CANCELLED]: {
        variant: "destructive" as const,
        icon: XCircle,
      },
    };

    const config = statusConfig[status];
    const Icon = config.icon;

    return (
      <Badge variant={config.variant}>
        <Icon className="w-3 h-3 mr-1" />
        {status.replace("_", " ").toUpperCase()}
      </Badge>
    );
  };

  // Computed values
  const criticalRequests = useMemo(() => {
    return (
      data?.requests.filter(
        (req) =>
          req.urgencyLevel === "critical" ||
          (req.isOverdue && req.hoursSinceCreation > 4)
      ) || []
    );
  }, [data?.requests]);

  const unassignedRequests = useMemo(() => {
    return data?.requests.filter((req) => !req.assignedUser) || [];
  }, [data?.requests]);

  const isAllSelected = useMemo(() => {
    return (
      data?.requests &&
      data.requests.length > 0 &&
      selectedRequests.length === data.requests.length
    );
  }, [selectedRequests.length, data?.requests]);

  const isSomeSelected = useMemo(() => {
    return (
      selectedRequests.length > 0 &&
      selectedRequests.length < (data?.requests.length || 0)
    );
  }, [selectedRequests.length, data?.requests]);

  useEffect(() => {
    fetchEmergencyRequests();
    fetchAssignees();
  }, [fetchEmergencyRequests, fetchAssignees]);

  // Enhanced auto-refresh with configurable interval
  useEffect(() => {
    if (!autoRefresh) return;

    const interval = setInterval(() => {
      if (!loading && !refreshing && !bulkActionLoading) {
        fetchEmergencyRequests();
      }
    }, 30000); // 30 seconds

    return () => clearInterval(interval);
  }, [
    fetchEmergencyRequests,
    loading,
    refreshing,
    bulkActionLoading,
    autoRefresh,
  ]);

  // Update URL params when filters change
  useEffect(() => {
    const params = new URLSearchParams();
    if (searchTerm) params.set("search", searchTerm);
    if (statusFilter !== "all") params.set("status", statusFilter);
    if (responseTimeFilter !== "all")
      params.set("responseTime", responseTimeFilter);
    if (propertyFilter) params.set("propertyId", propertyFilter);
    if (assignedFilter !== "all") params.set("assignedTo", assignedFilter);

    const newUrl = `${window.location.pathname}?${params.toString()}`;
    window.history.replaceState({}, "", newUrl);
  }, [
    searchTerm,
    statusFilter,
    responseTimeFilter,
    propertyFilter,
    assignedFilter,
  ]);

  // Show bulk actions when requests are selected
  useEffect(() => {
    setShowBulkActions(selectedRequests.length > 0);
  }, [selectedRequests.length]);

  if (loading) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight text-red-600">
              Emergency Maintenance
            </h1>
            <p className="text-muted-foreground">
              Critical maintenance requests requiring immediate attention
            </p>
          </div>
        </div>
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          {[...Array(4)].map((_, i) => (
            <Card key={i}>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <div className="h-4 bg-gray-200 rounded w-20 animate-pulse" />
                <div className="h-4 w-4 bg-gray-200 rounded animate-pulse" />
              </CardHeader>
              <CardContent>
                <div className="h-8 bg-gray-200 rounded w-16 animate-pulse mb-2" />
                <div className="h-3 bg-gray-200 rounded w-24 animate-pulse" />
              </CardContent>
            </Card>
          ))}
        </div>
        <div className="space-y-4">
          {[...Array(3)].map((_, i) => (
            <Card key={i}>
              <CardContent className="p-6">
                <div className="space-y-3">
                  <div className="h-4 bg-gray-200 rounded w-3/4 animate-pulse" />
                  <div className="h-3 bg-gray-200 rounded w-1/2 animate-pulse" />
                  <div className="h-3 bg-gray-200 rounded w-2/3 animate-pulse" />
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight text-red-600">
              Emergency Maintenance
            </h1>
            <p className="text-muted-foreground">
              Critical maintenance requests requiring immediate attention
            </p>
          </div>
        </div>
        <Card>
          <CardContent className="flex flex-col items-center justify-center min-h-[400px] space-y-4">
            <AlertTriangle className="h-12 w-12 text-red-500" />
            <h2 className="text-xl font-semibold">
              Failed to Load Emergency Requests
            </h2>
            <p className="text-muted-foreground text-center">{error}</p>
            <Button onClick={handleRefresh}>
              <RefreshCw className="mr-2 h-4 w-4" />
              Try Again
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Enhanced Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight text-red-600 flex items-center gap-2">
            <Zap className="h-8 w-8" />
            Emergency Maintenance
            {criticalRequests.length > 0 && (
              <Badge variant="destructive" className="animate-pulse">
                {criticalRequests.length} Critical
              </Badge>
            )}
          </h1>
          <p className="text-muted-foreground">
            Critical maintenance requests requiring immediate attention
            {unassignedRequests.length > 0 && (
              <span className="text-orange-600 font-medium ml-2">
                • {unassignedRequests.length} unassigned
              </span>
            )}
          </p>
        </div>
        <div className="flex items-center gap-2">
          <div className="flex items-center gap-2 mr-4">
            <label className="text-sm font-medium">Auto-refresh:</label>
            <Checkbox checked={autoRefresh} onCheckedChange={setAutoRefresh} />
          </div>
          <Button
            variant="outline"
            size="sm"
            onClick={handleRefresh}
            disabled={refreshing}
          >
            <RefreshCw
              className={`mr-2 h-4 w-4 ${refreshing ? "animate-spin" : ""}`}
            />
            Refresh
          </Button>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="outline" size="sm">
                <Download className="mr-2 h-4 w-4" />
                Export
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent>
              <DropdownMenuItem onClick={() => window.print()}>
                <FileText className="mr-2 h-4 w-4" />
                Print Report
              </DropdownMenuItem>
              <DropdownMenuItem>
                <Download className="mr-2 h-4 w-4" />
                Export CSV
              </DropdownMenuItem>
              <DropdownMenuItem>
                <Download className="mr-2 h-4 w-4" />
                Export PDF
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
          <Link href="/dashboard/maintenance/new?priority=emergency">
            <Button className="bg-red-600 hover:bg-red-700">
              <Plus className="mr-2 h-4 w-4" />
              New Emergency
            </Button>
          </Link>
        </div>
      </div>

      {/* Enhanced Statistics Cards */}
      {data?.statistics && (
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-6">
          <Card className="border-red-200 hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                Total Emergencies
              </CardTitle>
              <AlertTriangle className="h-4 w-4 text-red-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-red-600">
                {data.statistics.total}
              </div>
              <p className="text-xs text-muted-foreground">
                All time • {data.statistics.todayCreated || 0} today
              </p>
            </CardContent>
          </Card>

          <Card className="border-orange-200 hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Active</CardTitle>
              <Clock className="h-4 w-4 text-orange-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-orange-600">
                {data.statistics.active}
              </div>
              <p className="text-xs text-muted-foreground">Needs attention</p>
            </CardContent>
          </Card>

          <Card className="border-red-200 hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Critical</CardTitle>
              <Zap className="h-4 w-4 text-red-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-red-600">
                {data.statistics.critical || criticalRequests.length}
              </div>
              <p className="text-xs text-muted-foreground">&gt;4 hours old</p>
            </CardContent>
          </Card>

          <Card className="border-yellow-200 hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Unassigned</CardTitle>
              <UserPlus className="h-4 w-4 text-yellow-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-yellow-600">
                {data.statistics.unassigned || unassignedRequests.length}
              </div>
              <p className="text-xs text-muted-foreground">Need assignment</p>
            </CardContent>
          </Card>

          <Card className="border-green-200 hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Completed</CardTitle>
              <CheckCircle className="h-4 w-4 text-green-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600">
                {data.statistics.completed}
              </div>
              <p className="text-xs text-muted-foreground">Resolved</p>
            </CardContent>
          </Card>

          <Card className="border-blue-200 hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                Avg Response
              </CardTitle>
              <TrendingUp className="h-4 w-4 text-blue-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-blue-600">
                {data.statistics.avgResponseTime
                  ? `${Math.round(data.statistics.avgResponseTime)}h`
                  : "N/A"}
              </div>
              <p className="text-xs text-muted-foreground">Response time</p>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Enhanced Filters */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <Filter className="h-5 w-5" />
              Filters
              {(searchTerm ||
                statusFilter !== "all" ||
                responseTimeFilter !== "all" ||
                propertyFilter ||
                assignedFilter !== "all") && (
                <Badge variant="secondary" className="ml-2">
                  Active
                </Badge>
              )}
            </CardTitle>
            <Button
              variant="ghost"
              size="sm"
              onClick={() => setShowFilters(!showFilters)}
            >
              {showFilters ? "Hide" : "Show"} Filters
            </Button>
          </div>
        </CardHeader>
        {showFilters && (
          <CardContent>
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-6">
              <div className="space-y-2">
                <label className="text-sm font-medium">Search</label>
                <div className="relative">
                  <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
                  <Input
                    placeholder="Search requests..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-8"
                  />
                </div>
              </div>

              <div className="space-y-2">
                <label className="text-sm font-medium">Status</label>
                <Select value={statusFilter} onValueChange={setStatusFilter}>
                  <SelectTrigger>
                    <SelectValue placeholder="All statuses" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Statuses</SelectItem>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="submitted">Submitted</SelectItem>
                    <SelectItem value="assigned">Assigned</SelectItem>
                    <SelectItem value="in_progress">In Progress</SelectItem>
                    <SelectItem value="completed">Completed</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <label className="text-sm font-medium">Response Time</label>
                <Select
                  value={responseTimeFilter}
                  onValueChange={setResponseTimeFilter}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="All times" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Times</SelectItem>
                    <SelectItem value="normal">Normal (&lt;2h)</SelectItem>
                    <SelectItem value="overdue">Overdue (&gt;2h)</SelectItem>
                    <SelectItem value="critical">Critical (&gt;4h)</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <label className="text-sm font-medium">Assignment</label>
                <Select
                  value={assignedFilter}
                  onValueChange={setAssignedFilter}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="All assignments" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Assignments</SelectItem>
                    <SelectItem value="unassigned">Unassigned</SelectItem>
                    <SelectItem value="assigned">Assigned</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <label className="text-sm font-medium">Sort By</label>
                <Select value={sortBy} onValueChange={setSortBy}>
                  <SelectTrigger>
                    <SelectValue placeholder="Sort by" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="createdAt">Created Date</SelectItem>
                    <SelectItem value="hoursSinceCreation">
                      Response Time
                    </SelectItem>
                    <SelectItem value="priority">Priority</SelectItem>
                    <SelectItem value="status">Status</SelectItem>
                    <SelectItem value="property.name">Property</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="flex items-end">
                <Button
                  variant="outline"
                  onClick={() => {
                    setSearchTerm("");
                    setStatusFilter("all");
                    setResponseTimeFilter("all");
                    setPropertyFilter("");
                    setAssignedFilter("all");
                    setSortBy("createdAt");
                    setSortOrder("desc");
                  }}
                  className="w-full"
                >
                  Clear All
                </Button>
              </div>
            </div>
          </CardContent>
        )}
      </Card>

      {/* Bulk Actions Bar */}
      {showBulkActions && (
        <Card className="border-blue-200 bg-blue-50/50">
          <CardContent className="py-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-4">
                <span className="text-sm font-medium">
                  {selectedRequests.length} request(s) selected
                </span>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setSelectedRequests([])}
                >
                  Clear Selection
                </Button>
              </div>
              <div className="flex items-center gap-2">
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button size="sm" disabled={bulkActionLoading}>
                      <UserPlus className="mr-2 h-4 w-4" />
                      Assign To
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent>
                    {assigneeOptions.map((assignee) => (
                      <DropdownMenuItem
                        key={assignee.id}
                        onClick={() => handleBulkAssign(assignee.id)}
                      >
                        {assignee.name}
                      </DropdownMenuItem>
                    ))}
                  </DropdownMenuContent>
                </DropdownMenu>

                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button
                      size="sm"
                      variant="outline"
                      disabled={bulkActionLoading}
                    >
                      <Settings className="mr-2 h-4 w-4" />
                      Update Status
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent>
                    <DropdownMenuItem
                      onClick={() =>
                        handleBulkStatusUpdate(MaintenanceStatus.ASSIGNED)
                      }
                    >
                      Mark as Assigned
                    </DropdownMenuItem>
                    <DropdownMenuItem
                      onClick={() =>
                        handleBulkStatusUpdate(MaintenanceStatus.IN_PROGRESS)
                      }
                    >
                      Mark as In Progress
                    </DropdownMenuItem>
                    <DropdownMenuItem
                      onClick={() =>
                        handleBulkStatusUpdate(MaintenanceStatus.COMPLETED)
                      }
                    >
                      Mark as Completed
                    </DropdownMenuItem>
                  </DropdownMenuContent>
                </DropdownMenu>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Emergency Requests List */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>
                Emergency Requests ({data?.requests?.length || 0})
              </CardTitle>
              <CardDescription>
                Critical maintenance requests requiring immediate attention
              </CardDescription>
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant={viewMode === "table" ? "default" : "outline"}
                size="sm"
                onClick={() => setViewMode("table")}
              >
                <List className="h-4 w-4 mr-2" />
                Table
              </Button>
              <Button
                variant={viewMode === "cards" ? "default" : "outline"}
                size="sm"
                onClick={() => setViewMode("cards")}
              >
                <LayoutGrid className="h-4 w-4 mr-2" />
                Cards
              </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          {viewMode === "cards" ? (
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {loading ? (
                [...Array(6)].map((_, i) => (
                  <div
                    key={i}
                    className="h-64 bg-muted rounded-lg animate-pulse"
                  />
                ))
              ) : data?.requests && data.requests.length > 0 ? (
                data.requests.map((request) => (
                  <EmergencyCard
                    key={request._id}
                    request={request}
                    onEdit={(id) =>
                      window.open(`/dashboard/maintenance/${id}/edit`, "_blank")
                    }
                    onDelete={handleDeleteRequest}
                  />
                ))
              ) : (
                <div className="col-span-full text-center py-8">
                  <div className="flex flex-col items-center space-y-4">
                    <CheckCircle className="h-12 w-12 text-green-500" />
                    <h2 className="text-xl font-semibold">
                      No Emergency Requests
                    </h2>
                    <p className="text-muted-foreground text-center">
                      Great! There are currently no emergency maintenance
                      requests.
                    </p>
                  </div>
                </div>
              )}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-12">
                    <Checkbox
                      checked={isAllSelected}
                      onCheckedChange={handleSelectAll}
                      aria-label="Select all requests"
                    />
                  </TableHead>
                  <TableHead className="w-16">#</TableHead>
                  <TableHead>Request</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Property</TableHead>
                  <TableHead>Tenant</TableHead>
                  <TableHead>Assigned To</TableHead>
                  <TableHead>Created</TableHead>
                  <TableHead>Elapsed</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {loading ? (
                  <TableRow>
                    <TableCell colSpan={10} className="text-center py-8">
                      <div className="flex items-center justify-center">
                        <RefreshCw className="h-4 w-4 animate-spin mr-2" />
                        Loading emergency requests...
                      </div>
                    </TableCell>
                  </TableRow>
                ) : data?.requests && data.requests.length > 0 ? (
                  data.requests.map((request, index) => {
                    const getUrgencyColor = () => {
                      if (
                        request.urgencyLevel === "critical" ||
                        request.isOverdue
                      ) {
                        return "text-red-600";
                      }
                      if (request.urgencyLevel === "overdue") {
                        return "text-orange-600";
                      }
                      return "text-gray-600";
                    };

                    return (
                      <TableRow
                        key={request._id}
                        className={cn(
                          "hover:bg-muted/50 transition-colors",
                          request.urgencyLevel === "critical" &&
                            "bg-red-50 border-red-200",
                          request.urgencyLevel === "overdue" &&
                            "bg-orange-50 border-orange-200",
                          selectedRequests.includes(request._id) &&
                            "bg-blue-50 border-blue-200"
                        )}
                      >
                        <TableCell>
                          <Checkbox
                            checked={selectedRequests.includes(request._id)}
                            onCheckedChange={(checked) =>
                              handleSelectRequest(
                                request._id,
                                checked as boolean
                              )
                            }
                            aria-label={`Select request ${request.title}`}
                          />
                        </TableCell>
                        <TableCell className="font-medium text-muted-foreground">
                          {index + 1}
                        </TableCell>
                        <TableCell>
                          <div className="space-y-1">
                            <div className={`font-medium ${getUrgencyColor()}`}>
                              <Link
                                href={`/dashboard/maintenance/${request._id}`}
                                className="hover:underline"
                              >
                                {request.title}
                              </Link>
                            </div>
                            <div className="text-sm text-muted-foreground">
                              ID: {request._id.slice(-8)}
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="space-y-1">
                            {getUrgencyBadge(
                              request.urgencyLevel,
                              request.isOverdue
                            )}
                            {getStatusBadge(request.status)}
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="space-y-1">
                            <div className="font-medium">
                              {request.property.name}
                            </div>
                            <div className="text-sm text-muted-foreground">
                              {formatAddress(request.property.address)}
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="flex items-center space-x-3">
                            <Avatar className="h-8 w-8">
                              <AvatarFallback>
                                {request.tenant.firstName[0]}
                                {request.tenant.lastName[0]}
                              </AvatarFallback>
                            </Avatar>
                            <div>
                              <div className="font-medium">
                                {request.tenant.firstName}{" "}
                                {request.tenant.lastName}
                              </div>
                              <div className="text-sm text-muted-foreground">
                                {request.tenant.phone}
                              </div>
                            </div>
                          </div>
                        </TableCell>
                        <TableCell>
                          {request.assignedUser ? (
                            <div className="flex items-center gap-2">
                              <User className="h-4 w-4 text-green-600" />
                              <span className="text-sm">
                                {request.assignedUser.firstName}{" "}
                                {request.assignedUser.lastName}
                              </span>
                            </div>
                          ) : (
                            <div className="flex items-center gap-2">
                              <AlertCircle className="h-4 w-4 text-orange-600" />
                              <span className="text-sm text-orange-600">
                                Unassigned
                              </span>
                            </div>
                          )}
                        </TableCell>
                        <TableCell>
                          <div className="text-sm">
                            {formatDistanceToNow(new Date(request.createdAt))}{" "}
                            ago
                          </div>
                        </TableCell>
                        <TableCell>
                          <div className="text-sm font-medium">
                            {Math.round(request.hoursSinceCreation)}h
                          </div>
                        </TableCell>
                        <TableCell className="text-right">
                          <div className="flex items-center justify-end space-x-1">
                            <DropdownMenu>
                              <DropdownMenuTrigger asChild>
                                <Button variant="ghost" size="sm">
                                  <MoreHorizontal className="h-4 w-4" />
                                </Button>
                              </DropdownMenuTrigger>
                              <DropdownMenuContent align="end">
                                <DropdownMenuLabel>
                                  Quick Actions
                                </DropdownMenuLabel>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem
                                  onClick={() => {
                                    setSelectedRequestForAction(request._id);
                                    setQuickAssignDialog(true);
                                  }}
                                >
                                  <UserPlus className="mr-2 h-4 w-4" />
                                  Quick Assign
                                </DropdownMenuItem>
                                <DropdownMenuItem
                                  onClick={() => {
                                    setSelectedRequestForAction(request._id);
                                    setEscalateDialog(true);
                                  }}
                                  disabled={(request.escalationLevel || 0) >= 3}
                                >
                                  <AlertTriangle className="mr-2 h-4 w-4" />
                                  Escalate
                                </DropdownMenuItem>
                                                                <DropdownMenuItem
                                  onClick={() =>
                                    router.push(
                                      `/dashboard/maintenance/${request._id}`
                                    )
                                  }
                                >
                                  <Eye className="mr-2 h-4 w-4" />
                                  View
                                </DropdownMenuItem>
                                <DropdownMenuItem
                                  onClick={() =>
                                    router.push(
                                      `/dashboard/maintenance/${request._id}/edit`
                                    )
                                  }
                                >
                                  <Edit className="mr-2 h-4 w-4" />
                                  Edit
                                </DropdownMenuItem>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem
                                  onClick={() =>
                                    router.push(
                                      `/dashboard/properties/${request.property._id}`
                                    )
                                  }
                                >
                                  <ExternalLink className="mr-2 h-4 w-4" />
                                  View Property
                                </DropdownMenuItem>
                              </DropdownMenuContent>
                            </DropdownMenu>
                          </div>
                        </TableCell>
                      </TableRow>
                    );
                  })
                ) : (
                  <TableRow>
                    <TableCell colSpan={10} className="text-center py-8">
                      <div className="flex flex-col items-center space-y-4">
                        <CheckCircle className="h-12 w-12 text-green-500" />
                        <h2 className="text-xl font-semibold">
                          No Emergency Requests
                        </h2>
                        <p className="text-muted-foreground text-center">
                          Great! There are currently no emergency maintenance
                          requests.
                        </p>
                      </div>
                    </TableCell>
                  </TableRow>
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Pagination */}
      {data?.pagination && data.pagination.pages > 1 && (
        <div className="flex items-center justify-center space-x-2">
          <Button
            variant="outline"
            size="sm"
            disabled={data.pagination.page === 1}
            onClick={() => {
              const params = new URLSearchParams(searchParams.toString());
              params.set("page", (data.pagination.page - 1).toString());
              router.push(`?${params.toString()}`);
            }}
          >
            Previous
          </Button>
          <span className="text-sm text-muted-foreground">
            Page {data.pagination.page} of {data.pagination.pages}
          </span>
          <Button
            variant="outline"
            size="sm"
            disabled={data.pagination.page === data.pagination.pages}
            onClick={() => {
              const params = new URLSearchParams(searchParams.toString());
              params.set("page", (data.pagination.page + 1).toString());
              router.push(`?${params.toString()}`);
            }}
          >
            Next
          </Button>
        </div>
      )}

      {/* Quick Assign Dialog */}
      <Dialog open={quickAssignDialog} onOpenChange={setQuickAssignDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Quick Assign Request</DialogTitle>
            <DialogDescription>
              Assign this emergency request to a maintenance staff member.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <label className="text-sm font-medium">Assign To</label>
              <Select>
                <SelectTrigger>
                  <SelectValue placeholder="Select staff member" />
                </SelectTrigger>
                <SelectContent>
                  {assigneeOptions.map((assignee) => (
                    <SelectItem key={assignee.id} value={assignee.id}>
                      {assignee.name} - {assignee.email}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <label className="text-sm font-medium">Notes (Optional)</label>
              <Textarea
                placeholder="Add any special instructions or notes..."
                rows={3}
              />
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setQuickAssignDialog(false)}
            >
              Cancel
            </Button>
            <Button onClick={() => setQuickAssignDialog(false)}>
              Assign Request
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Escalate Dialog */}
      <Dialog open={escalateDialog} onOpenChange={setEscalateDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Escalate Emergency Request</DialogTitle>
            <DialogDescription>
              Escalate this request to higher priority and notify management.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
              <div className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5 text-yellow-600" />
                <span className="font-medium text-yellow-800">
                  Escalation Warning
                </span>
              </div>
              <p className="text-sm text-yellow-700 mt-1">
                This will notify management and increase the priority level of
                this request.
              </p>
            </div>
            <div className="space-y-2">
              <label className="text-sm font-medium">Escalation Reason</label>
              <Textarea
                placeholder="Explain why this request needs to be escalated..."
                rows={4}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setEscalateDialog(false)}>
              Cancel
            </Button>
            <Button
              variant="destructive"
              onClick={() => {
                if (selectedRequestForAction) {
                  handleEscalate(selectedRequestForAction);
                }
                setEscalateDialog(false);
              }}
            >
              <AlertTriangle className="mr-2 h-4 w-4" />
              Escalate Request
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
