"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription } from "@/components/ui/alert";
import {
  CreditCard,
  Download,
  Filter,
  Search,
  Calendar,
  ArrowLeft,
  Receipt,
  CheckCircle,
  XCircle,
  Clock,
  AlertTriangle,
  DollarSign,
  TrendingUp,
  Eye,
  Loader2,
  ExternalLink,
} from "lucide-react";
import Link from "next/link";
import { UserRole, PaymentStatus, PaymentType } from "@/types";
import { downloadInvoiceAsPDF, PrintableInvoice } from "@/lib/invoice-print";
import { normalizeInvoiceForPrint } from "@/lib/invoice/invoice-shared";

// ============================================================================
// INTERFACES
// ============================================================================

interface PaymentHistoryItem {
  _id: string;
  amount: number;
  amountPaid?: number;
  type: PaymentType;
  status: PaymentStatus;
  paymentMethod?: string;
  dueDate: string;
  paidDate?: string;
  description?: string;
  notes?: string;
  receiptUrl?: string;
  stripePaymentIntentId?: string;
  lateFeeApplied?: number;
  propertyId: {
    _id?: string;
    name?: string;
    address?:
      | {
          street?: string;
          city?: string;
          state?: string;
          zipCode?: string;
          [key: string]: any;
        }
      | string;
  };
  leaseId?: {
    _id: string;
    startDate: string;
    endDate: string;
  };
  invoiceId?: {
    _id: string;
    invoiceNumber: string;
    issueDate: string;
    dueDate: string;
    status: string;
    subtotal?: number;
    taxAmount?: number;
    totalAmount: number;
    amountPaid: number;
    balanceRemaining: number;
    notes?: string;
    lineItems?: Array<{
      description: string;
      amount?: number;
      type?: string;
      quantity?: number;
      unitPrice?: number;
    }>;
    propertyId?: {
      name?: string;
      address?:
        | {
            street?: string;
            city?: string;
            state?: string;
            zipCode?: string;
            [key: string]: any;
          }
        | string;
    };
    tenantId?: {
      firstName?: string;
      lastName?: string;
      email?: string;
    };
  };
  createdAt: string;
  updatedAt: string;
}

interface PaginationInfo {
  page: number;
  limit: number;
  total: number;
  pages: number;
}

interface PaymentSummary {
  totalPaid: number;
  totalPending: number;
  totalOverdue: number;
  paymentsThisMonth: number;
  paymentsThisYear: number;
  averagePaymentAmount: number;
  onTimePaymentRate: number;
}

// ============================================================================
// MAIN COMPONENT
// ============================================================================

export default function PaymentHistoryPage() {
  const router = useRouter();
  const { data: session, status } = useSession();

  // State management
  const [payments, setPayments] = useState<PaymentHistoryItem[]>([]);
  const [paymentSummary, setPaymentSummary] = useState<PaymentSummary | null>(
    null
  );
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [pagination, setPagination] = useState<PaginationInfo>({
    page: 1,
    limit: 10,
    total: 0,
    pages: 0,
  });

  // Filter state
  const [filters, setFilters] = useState({
    status: "all",
    type: "all",
    search: "",
    startDate: "",
    endDate: "",
  });

  // Modal state
  const [selectedPayment, setSelectedPayment] =
    useState<PaymentHistoryItem | null>(null);
  const [showPaymentDetails, setShowPaymentDetails] = useState(false);
  const [isDownloadingReceipt, setIsDownloadingReceipt] = useState<
    string | null
  >(null);

  // Authentication check
  useEffect(() => {
    if (status === "loading") return;

    if (!session) {
      router.push("/auth/signin");
      return;
    }

    if (session.user?.role !== UserRole.TENANT) {
      toast.error("Access denied. This page is for tenants only.");
      router.push("/dashboard");
      return;
    }
  }, [session, status, router]);

  // Fetch payment history
  useEffect(() => {
    if (session?.user?.role === UserRole.TENANT) {
      fetchPaymentHistory();
    }
  }, [session, pagination.page, filters]);

  const fetchPaymentHistory = async () => {
    try {
      setIsLoading(true);
      setError(null);

      const searchParams = new URLSearchParams({
        page: pagination.page.toString(),
        limit: pagination.limit.toString(),
      });

      // Add filters to search params
      if (filters.status !== "all") {
        searchParams.append("status", filters.status);
      }
      if (filters.type !== "all") {
        searchParams.append("type", filters.type);
      }
      if (filters.startDate) {
        searchParams.append("startDate", filters.startDate);
      }
      if (filters.endDate) {
        searchParams.append("endDate", filters.endDate);
      }
      if (filters.search) {
        searchParams.append("search", filters.search);
      }

      // Fetch payment history
      const historyResponse = await fetch(
        `/api/tenant/payments?${searchParams}`
      );
      if (!historyResponse.ok) {
        throw new Error("Failed to fetch payment history");
      }
      const historyData = await historyResponse.json();

      // Fetch payment summary (only on first load or when filters change significantly)
      let summaryData = null;
      if (pagination.page === 1) {
        try {
          const summaryResponse = await fetch("/api/tenant/payments/summary");
          if (summaryResponse.ok) {
            summaryData = await summaryResponse.json();
            setPaymentSummary(summaryData.data?.summary || null);
          }
        } catch (summaryError) {
          // Failed to fetch payment summary
        }
      }

      setPayments(historyData.data?.payments || []);
      setPagination(historyData.data?.pagination || pagination);
    } catch (error) {
      const errorMessage =
        error instanceof Error
          ? error.message
          : "Failed to load payment history";
      setError(errorMessage);
      toast.error(errorMessage);
    } finally {
      setIsLoading(false);
    }
  };

  // Utility functions
  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const formatPropertyAddress = (
    address?: PaymentHistoryItem["propertyId"]["address"]
  ) => {
    if (!address) return "N/A";
    if (typeof address === "string") return address;
    const parts = [address.street, address.city, address.state, address.zipCode]
      .filter(Boolean)
      .map((part) => part?.toString().trim())
      .filter(Boolean);
    return parts.length ? parts.join(", ") : "N/A";
  };

  const buildPrintableInvoice = (
    payment: PaymentHistoryItem
  ): PrintableInvoice | null => {
    const invoice = payment.invoiceId;
    if (!invoice) {
      const fallbackInvoice = {
        invoiceNumber:
          payment.referenceNumber ||
          `PAY-${payment._id?.slice?.(-8)?.toUpperCase?.() || Date.now()}`,
        issueDate: payment.paidDate || payment.createdAt,
        dueDate: payment.paidDate || payment.createdAt,
        status: payment.status === "completed" ? "paid" : "issued",
        subtotal: payment.amount,
        totalAmount: payment.amount,
        amountPaid: payment.amount,
        balanceRemaining: 0,
        notes: payment.notes,
        tenantId: payment.tenantId,
        propertyId: payment.propertyId,
        lineItems: [
          {
            description:
              payment.description ||
              payment.type
                .replace("_", " ")
                .replace(/\b\w/g, (l) => l.toUpperCase()),
            amount: payment.amount,
            quantity: 1,
            unitPrice: payment.amount,
            total: payment.amount,
            type: payment.type,
          },
        ],
      } as Record<string, unknown>;

      return normalizeInvoiceForPrint(fallbackInvoice) as PrintableInvoice;
    }

    const normalizedInvoice = normalizeInvoiceForPrint(
      {
        ...invoice,
        status: invoice.status || payment.status,
        propertyId: invoice.propertyId || payment.propertyId,
        lineItems:
          Array.isArray(invoice.lineItems) && invoice.lineItems.length > 0
            ? invoice.lineItems
            : [
                {
                  description:
                    payment.description ||
                    payment.type
                      .replace("_", " ")
                      .replace(/\b\w/g, (l) => l.toUpperCase()),
                  amount: payment.amount,
                  quantity: 1,
                  unitPrice: payment.amount,
                  total: payment.amount,
                  type: payment.type,
                },
              ],
      },
      { fallbackStatus: invoice.status || payment.status }
    );

    return normalizedInvoice as PrintableInvoice;
  };

  const getStatusIcon = (status: PaymentStatus) => {
    switch (status) {
      case PaymentStatus.COMPLETED:
        return <CheckCircle className="h-4 w-4 text-green-500" />;
      case PaymentStatus.FAILED:
        return <XCircle className="h-4 w-4 text-red-500" />;
      case PaymentStatus.PENDING:
        return <Clock className="h-4 w-4 text-yellow-500" />;
      case PaymentStatus.PROCESSING:
        return <Clock className="h-4 w-4 text-blue-500" />;
      default:
        return <Clock className="h-4 w-4 text-gray-500" />;
    }
  };

  const getStatusBadge = (status: PaymentStatus) => {
    switch (status) {
      case PaymentStatus.COMPLETED:
        return (
          <Badge variant="default" className="bg-green-500">
            Paid
          </Badge>
        );
      case PaymentStatus.FAILED:
        return <Badge variant="destructive">Failed</Badge>;
      case PaymentStatus.PENDING:
        return <Badge variant="secondary">Pending</Badge>;
      case PaymentStatus.PROCESSING:
        return <Badge variant="secondary">Processing</Badge>;
      default:
        return <Badge variant="outline">{status}</Badge>;
    }
  };

  const handleFilterChange = (key: string, value: string) => {
    setFilters((prev) => ({ ...prev, [key]: value }));
    setPagination((prev) => ({ ...prev, page: 1 })); // Reset to first page
  };

  const handlePageChange = (newPage: number) => {
    setPagination((prev) => ({ ...prev, page: newPage }));
  };

  const downloadReceipt = async (payment: PaymentHistoryItem) => {
    try {
      setIsDownloadingReceipt(payment._id);

      if (payment.invoiceId) {
        const printable = buildPrintableInvoice(payment);
        if (!printable) {
          throw new Error("Receipt data is incomplete");
        }

        // Fetch company info from display settings
        const { getCompanyInfo } = await import("@/lib/utils/company-info");
        const companyInfo = await getCompanyInfo();

        // Normalize with company info
        const normalizedPrintable = normalizeInvoiceForPrint(printable, {
          companyInfo: companyInfo || undefined,
        }) as PrintableInvoice;

        await downloadInvoiceAsPDF(normalizedPrintable);
        toast.success("Receipt generated successfully");
        return;
      }

      throw new Error("Receipt is not available for this payment yet");
    } catch (error) {
      toast.error(
        error instanceof Error ? error.message : "Failed to download receipt"
      );
    } finally {
      setIsDownloadingReceipt(null);
    }
  };

  const viewPaymentDetails = (payment: PaymentHistoryItem) => {
    setSelectedPayment(payment);
    setShowPaymentDetails(true);
  };

  const exportPaymentHistory = async (format: "csv" | "pdf") => {
    try {
      const searchParams = new URLSearchParams({
        format,
        ...(filters.status !== "all" && { status: filters.status }),
        ...(filters.type !== "all" && { type: filters.type }),
        ...(filters.startDate && { startDate: filters.startDate }),
        ...(filters.endDate && { endDate: filters.endDate }),
        ...(filters.search && { search: filters.search }),
      });

      const response = await fetch(
        `/api/tenant/payments/export?${searchParams}`
      );
      if (!response.ok) {
        throw new Error("Failed to export payment history");
      }

      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `payment-history.${format}`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);

      toast.success(`Payment history exported as ${format.toUpperCase()}`);
    } catch (error) {
      toast.error("Failed to export payment history");
    }
  };

  // Loading state
  if (status === "loading") {
    return <PaymentHistoryPageSkeleton />;
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">
              Payment History
            </h1>
            <p className="text-muted-foreground">
              View your complete payment history and download receipts
            </p>
          </div>
        </div>
        <div className="flex items-center gap-2">
          <Button variant="outline" onClick={() => exportPaymentHistory("csv")}>
            <Download className="mr-2 h-4 w-4" />
            Export CSV
          </Button>
          <Button variant="outline" onClick={() => exportPaymentHistory("pdf")}>
            <Download className="mr-2 h-4 w-4" />
            Export PDF
          </Button>
        </div>
      </div>

      {/* Payment Summary Statistics */}
      {paymentSummary && (
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Paid</CardTitle>
              <DollarSign className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600">
                {formatCurrency(paymentSummary.totalPaid)}
              </div>
              <p className="text-xs text-muted-foreground">
                {paymentSummary.paymentsThisYear} payments this year
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                Average Payment
              </CardTitle>
              <TrendingUp className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {formatCurrency(paymentSummary.averagePaymentAmount)}
              </div>
              <p className="text-xs text-muted-foreground">
                Per payment average
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                On-Time Rate
              </CardTitle>
              <CheckCircle className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600">
                {Math.round(paymentSummary.onTimePaymentRate)}%
              </div>
              <p className="text-xs text-muted-foreground">
                Payments made on time
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">This Month</CardTitle>
              <Calendar className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {paymentSummary.paymentsThisMonth}
              </div>
              <p className="text-xs text-muted-foreground">
                Payments made this month
              </p>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Filters */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Filter className="h-5 w-5" />
            Filter Payments
          </CardTitle>
          <CardDescription>
            Filter your payment history by status, type, and date range
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-5">
            <div>
              <Label htmlFor="status">Status</Label>
              <Select
                value={filters.status}
                onValueChange={(value) => handleFilterChange("status", value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="All statuses" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Statuses</SelectItem>
                  <SelectItem value="paid">Paid</SelectItem>
                  <SelectItem value="pending">Pending</SelectItem>
                  <SelectItem value="overdue">Overdue</SelectItem>
                  <SelectItem value="failed">Failed</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <Label htmlFor="type">Type</Label>
              <Select
                value={filters.type}
                onValueChange={(value) => handleFilterChange("type", value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="All types" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="rent">Rent</SelectItem>
                  <SelectItem value="security_deposit">
                    Security Deposit
                  </SelectItem>
                  <SelectItem value="late_fee">Late Fee</SelectItem>
                  <SelectItem value="utility">Utility</SelectItem>
                  <SelectItem value="other">Other</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <Label htmlFor="startDate">Start Date</Label>
              <Input
                id="startDate"
                type="date"
                value={filters.startDate}
                onChange={(e) =>
                  handleFilterChange("startDate", e.target.value)
                }
              />
            </div>

            <div>
              <Label htmlFor="endDate">End Date</Label>
              <Input
                id="endDate"
                type="date"
                value={filters.endDate}
                onChange={(e) => handleFilterChange("endDate", e.target.value)}
              />
            </div>

            <div className="flex items-end">
              <Button
                onClick={() => {
                  setFilters({
                    status: "all",
                    type: "all",
                    search: "",
                    startDate: "",
                    endDate: "",
                  });
                }}
                variant="outline"
                className="w-full"
              >
                Clear Filters
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Payment History Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CreditCard className="h-5 w-5" />
            Payment History ({pagination.total})
          </CardTitle>
          <CardDescription>
            Complete history of all your payments
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <PaymentHistoryTableSkeleton />
          ) : error ? (
            <div className="text-center py-8">
              <AlertTriangle className="h-12 w-12 text-destructive mx-auto mb-4" />
              <h3 className="text-lg font-semibold mb-2">
                Unable to Load Payment History
              </h3>
              <p className="text-muted-foreground mb-4">{error}</p>
              <Button onClick={fetchPaymentHistory}>Try Again</Button>
            </div>
          ) : payments.length === 0 ? (
            <div className="text-center py-8">
              <Receipt className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-lg font-semibold mb-2">No Payment History</h3>
              <p className="text-muted-foreground">
                No payments found matching your criteria.
              </p>
            </div>
          ) : (
            <>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead className="w-[50px]">#</TableHead>
                    <TableHead>Date</TableHead>
                    <TableHead>Type</TableHead>
                    <TableHead>Property</TableHead>
                    <TableHead>Amount</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Due Date</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {payments.map((payment, index) => (
                    <TableRow key={payment._id}>
                      <TableCell className="font-medium">
                        {(pagination.page - 1) * pagination.limit + index + 1}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          {getStatusIcon(payment.status)}
                          <span>
                            {payment.paidDate
                              ? formatDate(payment.paidDate)
                              : formatDate(payment.createdAt)}
                          </span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <span className="font-medium">
                          {payment.type
                            .replace("_", " ")
                            .replace(/\b\w/g, (l) => l.toUpperCase())}
                        </span>
                      </TableCell>
                      <TableCell>
                        <div>
                          <p className="font-medium">
                            {payment.propertyId.name || "Property"}
                          </p>
                          <p className="text-xs text-muted-foreground">
                            {formatPropertyAddress(payment.propertyId.address)}
                          </p>
                        </div>
                      </TableCell>
                      <TableCell>
                        <span className="font-semibold">
                          {formatCurrency(payment.amount)}
                        </span>
                      </TableCell>
                      <TableCell>{getStatusBadge(payment.status)}</TableCell>
                      <TableCell>
                        <span
                          className={`${
                            new Date(payment.dueDate) < new Date() &&
                            payment.status === PaymentStatus.PENDING
                              ? "text-destructive font-medium"
                              : ""
                          }`}
                        >
                          {formatDate(payment.dueDate)}
                        </span>
                      </TableCell>
                      <TableCell className="text-right">
                        <div className="flex items-center justify-end gap-2">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => viewPaymentDetails(payment)}
                          >
                            <Eye className="h-4 w-4 mr-1" />
                            Details
                          </Button>
                          {payment.status === PaymentStatus.COMPLETED && (
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => downloadReceipt(payment)}
                              disabled={isDownloadingReceipt === payment._id}
                            >
                              {isDownloadingReceipt === payment._id ? (
                                <Loader2 className="h-4 w-4 mr-1 animate-spin" />
                              ) : (
                                <Download className="h-4 w-4 mr-1" />
                              )}
                              Receipt
                            </Button>
                          )}
                          {payment.stripePaymentIntentId && (
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() =>
                                window.open(
                                  `https://dashboard.stripe.com/payments/${payment.stripePaymentIntentId}`,
                                  "_blank"
                                )
                              }
                            >
                              <ExternalLink className="h-4 w-4 mr-1" />
                              Stripe
                            </Button>
                          )}
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>

              {/* Pagination */}
              {pagination.pages > 1 && (
                <div className="flex items-center justify-between mt-6">
                  <div className="text-sm text-muted-foreground">
                    Showing {(pagination.page - 1) * pagination.limit + 1} to{" "}
                    {Math.min(
                      pagination.page * pagination.limit,
                      pagination.total
                    )}{" "}
                    of {pagination.total} payments
                  </div>
                  <div className="flex items-center gap-2">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => handlePageChange(pagination.page - 1)}
                      disabled={pagination.page <= 1}
                    >
                      Previous
                    </Button>
                    <div className="flex items-center gap-1">
                      {Array.from(
                        { length: Math.min(5, pagination.pages) },
                        (_, i) => {
                          const pageNum = i + 1;
                          return (
                            <Button
                              key={pageNum}
                              variant={
                                pagination.page === pageNum
                                  ? "default"
                                  : "outline"
                              }
                              size="sm"
                              onClick={() => handlePageChange(pageNum)}
                            >
                              {pageNum}
                            </Button>
                          );
                        }
                      )}
                    </div>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => handlePageChange(pagination.page + 1)}
                      disabled={pagination.page >= pagination.pages}
                    >
                      Next
                    </Button>
                  </div>
                </div>
              )}
            </>
          )}
        </CardContent>
      </Card>

      {/* Payment Details Modal */}
      <Dialog open={showPaymentDetails} onOpenChange={setShowPaymentDetails}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Receipt className="h-5 w-5" />
              Payment Details
            </DialogTitle>
            <DialogDescription>
              Complete information about this payment transaction
            </DialogDescription>
          </DialogHeader>

          {selectedPayment && (
            <div className="space-y-6">
              {/* Payment Overview */}
              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <Label className="text-sm font-medium">Payment Amount</Label>
                  <p className="text-2xl font-bold">
                    {formatCurrency(selectedPayment.amount)}
                  </p>
                  {selectedPayment.lateFeeApplied &&
                    selectedPayment.lateFeeApplied > 0 && (
                      <p className="text-sm text-destructive">
                        + {formatCurrency(selectedPayment.lateFeeApplied)} late
                        fee
                      </p>
                    )}
                </div>
                <div>
                  <Label className="text-sm font-medium">Status</Label>
                  <div className="mt-1">
                    {getStatusBadge(selectedPayment.status)}
                  </div>
                </div>
              </div>

              {/* Payment Information */}
              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <Label className="text-sm font-medium">Payment Type</Label>
                  <p className="font-medium">
                    {selectedPayment.type
                      .replace("_", " ")
                      .replace(/\b\w/g, (l) => l.toUpperCase())}
                  </p>
                </div>
                <div>
                  <Label className="text-sm font-medium">Payment Method</Label>
                  <p className="font-medium">
                    {selectedPayment.paymentMethod
                      ? selectedPayment.paymentMethod
                          .replace("_", " ")
                          .replace(/\b\w/g, (l) => l.toUpperCase())
                      : "Not specified"}
                  </p>
                </div>
              </div>

              {/* Dates */}
              <div className="grid gap-4 md:grid-cols-3">
                <div>
                  <Label className="text-sm font-medium">Due Date</Label>
                  <p className="font-medium">
                    {formatDate(selectedPayment.dueDate)}
                  </p>
                </div>
                <div>
                  <Label className="text-sm font-medium">Paid Date</Label>
                  <p className="font-medium">
                    {selectedPayment.paidDate
                      ? formatDate(selectedPayment.paidDate)
                      : "Not paid"}
                  </p>
                </div>
                <div>
                  <Label className="text-sm font-medium">Created Date</Label>
                  <p className="font-medium">
                    {formatDate(selectedPayment.createdAt)}
                  </p>
                </div>
              </div>

              {/* Property Information */}
              <div>
                <Label className="text-sm font-medium">Property</Label>
                <div className="mt-1 p-3 bg-muted/50 rounded-lg">
                  <p className="font-medium">
                    {selectedPayment.propertyId.name || "Property"}
                  </p>
                  <p className="text-sm text-muted-foreground">
                    {formatPropertyAddress(selectedPayment.propertyId.address)}
                  </p>
                </div>
              </div>

              {/* Description and Notes */}
              {(selectedPayment.description || selectedPayment.notes) && (
                <div className="space-y-3">
                  {selectedPayment.description && (
                    <div>
                      <Label className="text-sm font-medium">Description</Label>
                      <p className="text-sm">{selectedPayment.description}</p>
                    </div>
                  )}
                  {selectedPayment.notes && (
                    <div>
                      <Label className="text-sm font-medium">Notes</Label>
                      <p className="text-sm">{selectedPayment.notes}</p>
                    </div>
                  )}
                </div>
              )}

              {/* Transaction Details */}
              {selectedPayment.stripePaymentIntentId && (
                <div>
                  <Label className="text-sm font-medium">Transaction ID</Label>
                  <div className="flex items-center gap-2 mt-1">
                    <code className="text-xs bg-muted px-2 py-1 rounded">
                      {selectedPayment.stripePaymentIntentId}
                    </code>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() =>
                        window.open(
                          `https://dashboard.stripe.com/payments/${selectedPayment.stripePaymentIntentId}`,
                          "_blank"
                        )
                      }
                    >
                      <ExternalLink className="h-3 w-3 mr-1" />
                      View in Stripe
                    </Button>
                  </div>
                </div>
              )}

              {/* Actions */}
              <div className="flex gap-3 pt-4 border-t">
                {selectedPayment.status === PaymentStatus.COMPLETED && (
                  <Button
                    onClick={() => downloadReceipt(selectedPayment)}
                    disabled={isDownloadingReceipt === selectedPayment._id}
                  >
                    {isDownloadingReceipt === selectedPayment._id ? (
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    ) : (
                      <Download className="mr-2 h-4 w-4" />
                    )}
                    Download Receipt
                  </Button>
                )}
                <Button
                  variant="outline"
                  onClick={() => setShowPaymentDetails(false)}
                >
                  Close
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}

// ============================================================================
// SKELETON LOADING COMPONENTS
// ============================================================================

function PaymentHistoryPageSkeleton() {
  return (
    <div className="space-y-6">
      {/* Header Skeleton */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4">
          <div className="h-9 w-32 bg-muted rounded animate-pulse" />
          <div>
            <div className="h-8 w-48 bg-muted rounded animate-pulse mb-2" />
            <div className="h-4 w-64 bg-muted rounded animate-pulse" />
          </div>
        </div>
      </div>

      {/* Filters Skeleton */}
      <Card>
        <CardHeader>
          <div className="h-6 w-32 bg-muted rounded animate-pulse mb-2" />
          <div className="h-4 w-64 bg-muted rounded animate-pulse" />
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-5">
            {[...Array(5)].map((_, i) => (
              <div key={i}>
                <div className="h-4 w-16 bg-muted rounded animate-pulse mb-2" />
                <div className="h-10 w-full bg-muted rounded animate-pulse" />
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Table Skeleton */}
      <Card>
        <CardHeader>
          <div className="h-6 w-40 bg-muted rounded animate-pulse mb-2" />
          <div className="h-4 w-48 bg-muted rounded animate-pulse" />
        </CardHeader>
        <CardContent>
          <PaymentHistoryTableSkeleton />
        </CardContent>
      </Card>
    </div>
  );
}

function PaymentHistoryTableSkeleton() {
  return (
    <div className="space-y-4">
      {[...Array(5)].map((_, i) => (
        <div key={i} className="flex items-center space-x-4 p-4 border rounded">
          <div className="h-4 w-8 bg-muted rounded animate-pulse" />
          <div className="h-4 w-20 bg-muted rounded animate-pulse" />
          <div className="h-4 w-16 bg-muted rounded animate-pulse" />
          <div className="h-4 w-32 bg-muted rounded animate-pulse" />
          <div className="h-4 w-20 bg-muted rounded animate-pulse" />
          <div className="h-4 w-16 bg-muted rounded animate-pulse" />
          <div className="h-4 w-20 bg-muted rounded animate-pulse" />
          <div className="h-8 w-20 bg-muted rounded animate-pulse ml-auto" />
        </div>
      ))}
    </div>
  );
}
