"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { toast } from "sonner";
import Link from "next/link";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  AlertTriangle,
  Search,
  MoreHorizontal,
  Edit,
  Eye,
  Calendar,
  DollarSign,
  CheckCircle,
  Clock,
  Send,
  Phone,
  Mail,
  FileText,
  Download,
  RefreshCw,
  CreditCard
} from "lucide-react";
import { PaymentType, PaymentStatus, PaymentMethod } from "@/types";

interface OverdueInvoice {
  _id: string;
  invoiceNumber: string;
  amount: number;
  balanceRemaining: number;
  dueDate: string;
  issueDate: string;
  status: string;
  description?: string;
  notes?: string;
  daysOverdue: number;
  tenantId: {
    _id: string;
    firstName: string;
    lastName: string;
    email: string;
    phone?: string;
  };
  propertyId: {
    _id: string;
    name: string;
    address: {
      street: string;
      city: string;
      state: string;
      zipCode: string;
    };
  };
  leaseId?: {
    _id: string;
    startDate: string;
    endDate: string;
  };
  createdAt: string;
  updatedAt: string;
}

export default function OverduePaymentsPage() {
  const { data: session } = useSession();
  const [overduePayments, setOverduePayments] = useState<OverdueInvoice[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [sortBy, setSortBy] = useState<string>("daysOverdue");
  const [filterBy, setFilterBy] = useState<string>("all");

  // Fetch overdue invoices
  useEffect(() => {
    const fetchOverduePayments = async () => {
      if (!session) return;

      try {
        setIsLoading(true);
        // Fetch invoices with outstanding balances
        const response = await fetch("/api/invoices?limit=100");

        if (response.ok) {
          const result = await response.json();
          const invoices = result.data?.invoices || [];

          // Filter and calculate days overdue for invoices with outstanding balances
          const overdueInvoices = invoices
            .map((invoice: any) => {
              const dueDate = invoice.dueDate ? new Date(invoice.dueDate) : null;
              if (!dueDate || Number.isNaN(dueDate.getTime())) {
                return null;
              }

              const now = new Date();
              const daysOverdue = Math.ceil(
                (now.getTime() - dueDate.getTime()) / (1000 * 60 * 60 * 24)
              );

              // Only include invoices that are actually overdue (past due date)
              // and have outstanding balance
              const hasBalance = (invoice.balanceRemaining || invoice.amount) > 0;
              const isOverdue = daysOverdue > 0;

              if (!hasBalance || !isOverdue) {
                return null;
              }

              return {
                ...invoice,
                daysOverdue,
                amount: invoice.amount || 0,
                balanceRemaining: invoice.balanceRemaining || invoice.amount || 0,
              } as OverdueInvoice;
            })
            .filter((invoice): invoice is OverdueInvoice => invoice !== null);

          setOverduePayments(overdueInvoices);
        } else {
          throw new Error("Failed to fetch overdue invoices");
        }
      } catch (error) {
        toast.error(
          error instanceof Error
            ? error.message
            : "Failed to load overdue payments. Please try again."
        );
      } finally {
        setIsLoading(false);
      }
    };

    fetchOverduePayments();
  }, [session]);

  const filteredPayments = overduePayments.filter((invoice) => {
    const matchesSearch =
      (invoice.invoiceNumber &&
        invoice.invoiceNumber
          .toLowerCase()
          .includes(searchTerm.toLowerCase())) ||
      (invoice.tenantId?.firstName &&
        invoice.tenantId.firstName
          .toLowerCase()
          .includes(searchTerm.toLowerCase())) ||
      (invoice.tenantId?.lastName &&
        invoice.tenantId.lastName
          .toLowerCase()
          .includes(searchTerm.toLowerCase())) ||
      (invoice.propertyId?.name &&
        invoice.propertyId.name
          .toLowerCase()
          .includes(searchTerm.toLowerCase()));

    const matchesFilter =
      filterBy === "all" ||
      (filterBy === "critical" && invoice.daysOverdue > 30) ||
      (filterBy === "severe" &&
        invoice.daysOverdue > 14 &&
        invoice.daysOverdue <= 30) ||
      (filterBy === "moderate" &&
        invoice.daysOverdue > 7 &&
        invoice.daysOverdue <= 14) ||
      (filterBy === "recent" && invoice.daysOverdue <= 7);

    return matchesSearch && matchesFilter;
  });

  // Sort invoices
  const sortedPayments = [...filteredPayments].sort((a, b) => {
    switch (sortBy) {
      case "daysOverdue":
        return b.daysOverdue - a.daysOverdue;
      case "amount":
        return (b.balanceRemaining || b.amount) - (a.balanceRemaining || a.amount);
      case "dueDate":
        return new Date(a.dueDate).getTime() - new Date(b.dueDate).getTime();
      case "tenant":
        return (a.tenantId?.lastName || "").localeCompare(
          b.tenantId?.lastName || ""
        );
      case "invoice":
        return (a.invoiceNumber || "").localeCompare(b.invoiceNumber || "");
      default:
        return b.daysOverdue - a.daysOverdue;
    }
  });

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const getOverdueSeverity = (daysOverdue: number) => {
    if (daysOverdue > 30) return { label: "Critical", color: "destructive" };
    if (daysOverdue > 14) return { label: "Severe", color: "destructive" };
    if (daysOverdue > 7) return { label: "Moderate", color: "secondary" };
    return { label: "Recent", color: "outline" };
  };

  const totalOverdueAmount = sortedPayments.reduce(
    (sum, payment) => sum + (payment.balanceRemaining || payment.amount),
    0
  );
  const criticalCount = sortedPayments.filter((p) => p.daysOverdue > 30).length;
  const severeCount = sortedPayments.filter(
    (p) => p.daysOverdue > 14 && p.daysOverdue <= 30
  ).length;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight flex items-center gap-2">
            <AlertTriangle className="h-8 w-8 text-red-600" />
            Overdue Payments
          </h1>
          <p className="text-muted-foreground">
            Manage and track overdue invoices and outstanding balances
          </p>
        </div>
        <div className="flex items-center space-x-2">
          <Button variant="outline">
            <RefreshCw className="mr-2 h-4 w-4" />
            Refresh
          </Button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-4 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Overdue</CardTitle>
            <DollarSign className="h-4 w-4 text-red-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">
              {formatCurrency(totalOverdueAmount)}
            </div>
            <p className="text-xs text-muted-foreground">
              {sortedPayments.length} payments
            </p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">
              Critical (30+ days)
            </CardTitle>
            <AlertTriangle className="h-4 w-4 text-red-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">
              {criticalCount}
            </div>
            <p className="text-xs text-muted-foreground">
              Immediate attention required
            </p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">
              Severe (15-30 days)
            </CardTitle>
            <Clock className="h-4 w-4 text-orange-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">
              {severeCount}
            </div>
            <p className="text-xs text-muted-foreground">Follow-up needed</p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Average Days</CardTitle>
            <Calendar className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {sortedPayments.length > 0
                ? Math.round(
                    sortedPayments.reduce((sum, p) => sum + p.daysOverdue, 0) /
                      sortedPayments.length
                  )
                : 0}
            </div>
            <p className="text-xs text-muted-foreground">Days overdue</p>
          </CardContent>
        </Card>
      </div>

      {/* Filters */}
      <Card>
        <CardHeader>
          <CardTitle>Filter Overdue Payments</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col gap-4 md:flex-row md:items-center">
            <div className="relative flex-1">
              <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
              <Input
                placeholder="Search by invoice #, tenant, or property..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
            <Select value={filterBy} onValueChange={setFilterBy}>
              <SelectTrigger className="w-full md:w-[180px]">
                <SelectValue placeholder="Filter by severity" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Overdue</SelectItem>
                <SelectItem value="critical">Critical (30+ days)</SelectItem>
                <SelectItem value="severe">Severe (15-30 days)</SelectItem>
                <SelectItem value="moderate">Moderate (8-14 days)</SelectItem>
                <SelectItem value="recent">Recent (1-7 days)</SelectItem>
              </SelectContent>
            </Select>
            <Select value={sortBy} onValueChange={setSortBy}>
              <SelectTrigger className="w-full md:w-[180px]">
                <SelectValue placeholder="Sort by" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="daysOverdue">Days Overdue</SelectItem>
                <SelectItem value="amount">Balance Remaining</SelectItem>
                <SelectItem value="dueDate">Due Date</SelectItem>
                <SelectItem value="tenant">Tenant Name</SelectItem>
                <SelectItem value="invoice">Invoice Number</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Overdue Payments Display */}
      {isLoading ? (
        <div className="text-center py-8">
          <p>Loading overdue payments...</p>
        </div>
      ) : sortedPayments.length === 0 ? (
        <Card>
          <CardContent className="text-center py-12">
            <CheckCircle className="h-16 w-16 text-green-600 mx-auto mb-4" />
            <h3 className="text-lg font-semibold mb-2">No Overdue Payments</h3>
            <p className="text-muted-foreground mb-4">
              Great! All payments are up to date.
            </p>
            <Link href="/dashboard/payments">
              <Button>View All Payments</Button>
            </Link>
          </CardContent>
        </Card>
      ) : (
        <Card>
          <CardHeader>
            <CardTitle>Overdue Payments ({sortedPayments.length})</CardTitle>
            <CardDescription>
              Payments that are past their due date and require attention
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Invoice #</TableHead>
                  <TableHead>Tenant</TableHead>
                  <TableHead>Property</TableHead>
                  <TableHead>Amount</TableHead>
                  <TableHead>Balance</TableHead>
                  <TableHead>Due Date</TableHead>
                  <TableHead>Days Overdue</TableHead>
                  <TableHead>Severity</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {sortedPayments.map((invoice) => {
                  const severity = getOverdueSeverity(invoice.daysOverdue);

                  return (
                    <TableRow key={invoice._id} className="hover:bg-muted/50">
                      <TableCell>
                        <div className="font-medium text-blue-600">
                          {invoice.invoiceNumber || "N/A"}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div>
                          <div className="font-medium">
                            {invoice.tenantId?.firstName || "N/A"}{" "}
                            {invoice.tenantId?.lastName || ""}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            {invoice.tenantId?.email || "N/A"}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div>
                          <div className="font-medium">
                            {invoice.propertyId?.name || "N/A"}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            {invoice.propertyId?.address?.city || "N/A"},{" "}
                            {invoice.propertyId?.address?.state || "N/A"}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="font-medium">
                          {formatCurrency(invoice.amount)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="font-medium text-red-600">
                          {formatCurrency(invoice.balanceRemaining || invoice.amount)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-1 text-sm">
                          <Calendar className="h-3 w-3 text-muted-foreground" />
                          <span>{formatDate(invoice.dueDate)}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="text-sm font-medium text-red-600">
                          {invoice.daysOverdue} days
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge
                          variant={
                            severity.color as
                              | "default"
                              | "secondary"
                              | "destructive"
                              | "outline"
                          }
                          className="text-xs"
                        >
                          {severity.label}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem asChild>
                              <Link href={`/dashboard/leases/invoices/${invoice._id}`}>
                                <Eye className="mr-2 h-4 w-4" />
                                View Invoice
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuItem asChild>
                              <Link
                                href={`/dashboard/leases/invoices/${invoice._id}/edit`}
                              >
                                <Edit className="mr-2 h-4 w-4" />
                                Edit Invoice
                              </Link>
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            <DropdownMenuItem>
                              <Send className="mr-2 h-4 w-4" />
                              Send Reminder
                            </DropdownMenuItem>
                            <DropdownMenuItem>
                              <Phone className="mr-2 h-4 w-4" />
                              Call Tenant
                            </DropdownMenuItem>
                            <DropdownMenuItem>
                              <Mail className="mr-2 h-4 w-4" />
                              Email Invoice
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            <DropdownMenuItem className="text-green-600">
                              <CreditCard className="mr-2 h-4 w-4" />
                              Record Payment
                            </DropdownMenuItem>
                            <DropdownMenuItem>
                              <FileText className="mr-2 h-4 w-4" />
                              Generate Notice
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
