"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { toast } from "sonner";
import Link from "next/link";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  CreditCard,
  Plus,
  Search,
  MoreHorizontal,
  Edit,
  Eye,
  Trash2,
  Calendar,
  DollarSign,
  AlertTriangle,
  CheckCircle,
  XCircle,
  Clock,
  RefreshCw,
  Send,
  Grid3X3,
  List,
  BarChart3,
} from "lucide-react";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { PaymentType, PaymentStatus, PaymentMethod } from "@/types";

// Mock data removed - using real API data

interface Payment {
  _id: string;
  amount: number;
  type: PaymentType;
  status: PaymentStatus;
  paymentMethod?: PaymentMethod;
  dueDate: string;
  paidDate?: string;
  description?: string;
  notes?: string;
  tenantId: {
    _id: string;
    firstName: string;
    lastName: string;
    email: string;
    phone?: string;
  };
  propertyId: {
    _id: string;
    name: string;
    address: {
      street: string;
      city: string;
      state: string;
      zipCode: string;
    };
  };
  leaseId?: {
    _id: string;
    startDate: string;
    endDate: string;
  };
  createdAt: string;
  updatedAt: string;
}

export default function PaymentsPage() {
  const { data: session } = useSession();
  const [payments, setPayments] = useState<Payment[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [typeFilter, setTypeFilter] = useState<string>("all");
  const [paymentMethodFilter, setPaymentMethodFilter] = useState<string>("all");
  const [viewMode, setViewMode] = useState<"table" | "card">("table");
  const [showDeleteDialog, setShowDeleteDialog] = useState(false);
  const [paymentToDelete, setPaymentToDelete] = useState<string | null>(null);
  const [isDeleting, setIsDeleting] = useState(false);
  const [showRefundDialog, setShowRefundDialog] = useState(false);
  const [paymentToRefund, setPaymentToRefund] = useState<string | null>(null);
  const [isRefunding, setIsRefunding] = useState(false);

  // Fetch payments function
  const fetchPayments = async () => {
    if (!session) return;

    try {
      setIsLoading(true);
      const response = await fetch("/api/payments");

      if (response.ok) {
        const result = await response.json();
        setPayments(result.data || []);
      } else {
        throw new Error("Failed to fetch payments");
      }
    } catch (error) {
      toast.error(
        error instanceof Error
          ? error.message
          : "Failed to load payments. Please try again."
      );
    } finally {
      setIsLoading(false);
    }
  };

  // Fetch payments on mount
  useEffect(() => {
    fetchPayments();
  }, [session]);

  const filteredPayments = payments.filter((payment) => {
    const matchesSearch =
      (payment.tenantId?.firstName &&
        payment.tenantId.firstName
          .toLowerCase()
          .includes(searchTerm.toLowerCase())) ||
      (payment.tenantId?.lastName &&
        payment.tenantId.lastName
          .toLowerCase()
          .includes(searchTerm.toLowerCase())) ||
      (payment.propertyId?.name &&
        payment.propertyId.name
          .toLowerCase()
          .includes(searchTerm.toLowerCase())) ||
      (payment.description &&
        payment.description.toLowerCase().includes(searchTerm.toLowerCase()));

    const matchesStatus =
      statusFilter === "all" || payment.status === statusFilter;
    const matchesType = typeFilter === "all" || payment.type === typeFilter;
    const matchesPaymentMethod =
      paymentMethodFilter === "all" ||
      payment.paymentMethod === paymentMethodFilter;

    return (
      matchesSearch && matchesStatus && matchesType && matchesPaymentMethod
    );
  });

  // Delete payment handler
  const handleDeletePayment = async () => {
    if (!paymentToDelete) return;

    try {
      setIsDeleting(true);
      const response = await fetch(`/api/payments/${paymentToDelete}`, {
        method: "DELETE",
      });

      if (response.ok) {
        setPayments(payments.filter((p) => p._id !== paymentToDelete));
        toast.success("Payment deleted successfully.");
      } else {
        const error = await response.json();
        throw new Error(error.error || "Failed to delete payment");
      }
    } catch (error) {
      toast.error(
        error instanceof Error
          ? error.message
          : "Failed to delete payment. Please try again."
      );
    } finally {
      setIsDeleting(false);
      setShowDeleteDialog(false);
      setPaymentToDelete(null);
    }
  };

  const confirmDelete = (paymentId: string) => {
    setPaymentToDelete(paymentId);
    setShowDeleteDialog(true);
  };

  // Refund handler
  const confirmRefund = (paymentId: string) => {
    setPaymentToRefund(paymentId);
    setShowRefundDialog(true);
  };

  const handleRefund = async () => {
    if (!paymentToRefund) return;

    try {
      setIsRefunding(true);

      const response = await fetch(`/api/payments/${paymentToRefund}`, {
        method: "PATCH",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          action: "refund",
        }),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "Failed to refund payment");
      }

      toast.success("Payment refunded successfully");
      fetchPayments();
      setShowRefundDialog(false);
      setPaymentToRefund(null);
    } catch (error) {
      toast.error(
        error instanceof Error
          ? error.message
          : "Failed to refund payment. Please try again."
      );
    } finally {
      setIsRefunding(false);
    }
  };

  const getStatusColor = (status: PaymentStatus) => {
    switch (status) {
      case PaymentStatus.PENDING:
        return "default";
      case PaymentStatus.PROCESSING:
        return "secondary";
      case PaymentStatus.COMPLETED:
        return "secondary";
      case PaymentStatus.FAILED:
        return "destructive";
      case PaymentStatus.REFUNDED:
        return "outline";
      default:
        return "outline";
    }
  };

  const getStatusIcon = (status: PaymentStatus) => {
    switch (status) {
      case PaymentStatus.PENDING:
        return Clock;
      case PaymentStatus.PROCESSING:
        return RefreshCw;
      case PaymentStatus.COMPLETED:
        return CheckCircle;
      case PaymentStatus.FAILED:
        return XCircle;
      case PaymentStatus.REFUNDED:
        return RefreshCw;
      default:
        return Clock;
    }
  };

  const getTypeColor = (type: PaymentType) => {
    switch (type) {
      case PaymentType.RENT:
        return "default";
      case PaymentType.SECURITY_DEPOSIT:
        return "secondary";
      case PaymentType.INVOICE:
        return "secondary";
      case PaymentType.PET_DEPOSIT:
        return "secondary";
      case PaymentType.LATE_FEE:
        return "destructive";
      case PaymentType.UTILITY:
        return "outline";
      case PaymentType.MAINTENANCE:
        return "outline";
      case PaymentType.OTHER:
        return "outline";
      default:
        return "outline";
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const getDaysOverdue = (dueDate: string, status: PaymentStatus) => {
    if (
      status === PaymentStatus.COMPLETED ||
      status === PaymentStatus.REFUNDED
    ) {
      return 0;
    }
    const now = new Date();
    const due = new Date(dueDate);
    const diffTime = now.getTime() - due.getTime();
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    return Math.max(0, diffDays);
  };

  const totalAmount = filteredPayments.reduce(
    (sum, payment) => sum + payment.amount,
    0
  );
  const completedAmount = filteredPayments
    .filter((p) => p.status === PaymentStatus.COMPLETED)
    .reduce((sum, payment) => sum + payment.amount, 0);
  const pendingAmount = filteredPayments
    .filter((p) => p.status === PaymentStatus.PENDING)
    .reduce((sum, payment) => sum + payment.amount, 0);
  const overduePayments = filteredPayments.filter(
    (p) =>
      (p.status === PaymentStatus.PENDING ||
        p.status === PaymentStatus.FAILED) &&
      getDaysOverdue(p.dueDate, p.status) > 0
  );

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Payments</h1>
          <p className="text-muted-foreground">
            Manage rent payments and financial transactions
          </p>
        </div>
        <div className="flex items-center space-x-2">
          <div className="flex items-center border rounded-lg p-1">
            <Button
              variant={viewMode === "table" ? "default" : "ghost"}
              size="sm"
              onClick={() => setViewMode("table")}
            >
              <List className="h-4 w-4" />
            </Button>
            <Button
              variant={viewMode === "card" ? "default" : "ghost"}
              size="sm"
              onClick={() => setViewMode("card")}
            >
              <Grid3X3 className="h-4 w-4" />
            </Button>
          </div>
          <Link href="/dashboard/payments/analytics">
            <Button variant="outline">
              <BarChart3 className="mr-2 h-4 w-4" />
              Analytics
            </Button>
          </Link>
          <Link href="/dashboard/payments/new">
            <Button>
              <Plus className="mr-2 h-4 w-4" />
              Add Payment
            </Button>
          </Link>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-4 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Amount</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(totalAmount)}
            </div>
            <p className="text-xs text-muted-foreground">
              {filteredPayments.length} payments
            </p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Collected</CardTitle>
            <CheckCircle className="h-4 w-4 text-green-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(completedAmount)}
            </div>
            <p className="text-xs text-muted-foreground">
              {
                filteredPayments.filter(
                  (p) => p.status === PaymentStatus.COMPLETED
                ).length
              }{" "}
              completed
            </p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Pending</CardTitle>
            <Clock className="h-4 w-4 text-orange-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {formatCurrency(pendingAmount)}
            </div>
            <p className="text-xs text-muted-foreground">
              {
                filteredPayments.filter(
                  (p) => p.status === PaymentStatus.PENDING
                ).length
              }{" "}
              pending
            </p>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Overdue</CardTitle>
            <AlertTriangle className="h-4 w-4 text-red-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{overduePayments.length}</div>
            <p className="text-xs text-muted-foreground">
              {formatCurrency(
                overduePayments.reduce((sum, p) => sum + p.amount, 0)
              )}{" "}
              total
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Filters */}
      <Card>
        <CardHeader>
          <CardTitle>Filter Payments</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-col gap-4 md:flex-row md:items-center">
            <div className="relative flex-1">
              <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
              <Input
                placeholder="Search payments..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-full md:w-[180px]">
                <SelectValue placeholder="Filter by status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Statuses</SelectItem>
                <SelectItem value={PaymentStatus.PENDING}>Pending</SelectItem>
                <SelectItem value={PaymentStatus.PROCESSING}>
                  Processing
                </SelectItem>
                <SelectItem value={PaymentStatus.COMPLETED}>
                  Completed
                </SelectItem>
                <SelectItem value={PaymentStatus.FAILED}>Failed</SelectItem>
                <SelectItem value={PaymentStatus.REFUNDED}>Refunded</SelectItem>
              </SelectContent>
            </Select>
            <Select value={typeFilter} onValueChange={setTypeFilter}>
              <SelectTrigger className="w-full md:w-[180px]">
                <SelectValue placeholder="Filter by type" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Types</SelectItem>
                <SelectItem value={PaymentType.RENT}>Rent</SelectItem>
                <SelectItem value={PaymentType.SECURITY_DEPOSIT}>
                  Security Deposit
                </SelectItem>
                <SelectItem value={PaymentType.INVOICE}>Invoice</SelectItem>
                <SelectItem value={PaymentType.LATE_FEE}>Late Fee</SelectItem>
                <SelectItem value={PaymentType.UTILITY}>Utility</SelectItem>
                <SelectItem value={PaymentType.MAINTENANCE}>
                  Maintenance
                </SelectItem>
                <SelectItem value={PaymentType.PET_DEPOSIT}>
                  Pet Deposit
                </SelectItem>
                <SelectItem value={PaymentType.OTHER}>Other</SelectItem>
              </SelectContent>
            </Select>
            <Select
              value={paymentMethodFilter}
              onValueChange={setPaymentMethodFilter}
            >
              <SelectTrigger className="w-full md:w-[180px]">
                <SelectValue placeholder="Filter by method" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Methods</SelectItem>
                <SelectItem value="credit_card">Credit Card</SelectItem>
                <SelectItem value="debit_card">Debit Card</SelectItem>
                <SelectItem value="bank_transfer">Bank Transfer</SelectItem>
                <SelectItem value="ach">ACH</SelectItem>
                <SelectItem value="check">Check</SelectItem>
                <SelectItem value="cash">Cash</SelectItem>
                <SelectItem value="money_order">Money Order</SelectItem>
                <SelectItem value="other">Other</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Payments Display */}
      {isLoading ? (
        <div className="text-center py-8">
          <p>Loading payments...</p>
        </div>
      ) : viewMode === "card" ? (
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
          {filteredPayments.map((payment) => {
            const StatusIcon = getStatusIcon(payment.status);
            const daysOverdue = getDaysOverdue(payment.dueDate, payment.status);

            return (
              <Card
                key={payment._id}
                className="hover:shadow-md transition-shadow"
              >
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <CreditCard className="h-5 w-5 text-muted-foreground" />
                      <Badge
                        variant={getStatusColor(payment.status) as any}
                        className="text-xs"
                      >
                        <StatusIcon className="h-3 w-3 mr-1" />
                        {payment.status.charAt(0).toUpperCase() +
                          payment.status.slice(1)}
                      </Badge>
                    </div>
                    <DropdownMenu>
                      <DropdownMenuTrigger asChild>
                        <Button variant="ghost" className="h-8 w-8 p-0">
                          <MoreHorizontal className="h-4 w-4" />
                        </Button>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent align="end">
                        <DropdownMenuLabel>Actions</DropdownMenuLabel>
                        <DropdownMenuItem asChild>
                          <Link href={`/dashboard/payments/${payment._id}`}>
                            <Eye className="mr-2 h-4 w-4" />
                            View Details
                          </Link>
                        </DropdownMenuItem>
                        {payment.status !== PaymentStatus.COMPLETED &&
                         payment.status !== PaymentStatus.REFUNDED && (
                          <DropdownMenuItem asChild>
                            <Link
                              href={`/dashboard/payments/${payment._id}/edit`}
                            >
                              <Edit className="mr-2 h-4 w-4" />
                              Edit Payment
                            </Link>
                          </DropdownMenuItem>
                        )}
                        {/* Only allow delete for PENDING payments that haven't been processed */}
                        {payment.status === PaymentStatus.PENDING && (
                          <>
                            <DropdownMenuSeparator />
                            <DropdownMenuItem
                              className="text-red-600"
                              onClick={() => confirmDelete(payment._id)}
                            >
                              <Trash2 className="mr-2 h-4 w-4" />
                              Delete Payment
                            </DropdownMenuItem>
                          </>
                        )}
                      </DropdownMenuContent>
                    </DropdownMenu>
                  </div>
                  <div className="text-2xl font-bold">
                    {formatCurrency(payment.amount)}
                  </div>
                  <Badge
                    variant={getTypeColor(payment.type) as any}
                    className="w-fit text-xs capitalize"
                  >
                    {payment.type.replace("_", " ")}
                  </Badge>
                </CardHeader>
                <CardContent className="space-y-3">
                  <div>
                    <p className="text-sm font-medium">
                      {payment.tenantId?.firstName || "N/A"}{" "}
                      {payment.tenantId?.lastName || ""}
                    </p>
                    <p className="text-xs text-muted-foreground">
                      {payment.tenantId?.email || "N/A"}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm font-medium">
                      {payment.propertyId?.name || "N/A"}
                    </p>
                    <p className="text-xs text-muted-foreground">
                      {payment.propertyId?.address?.city || "N/A"},{" "}
                      {payment.propertyId?.address?.state || "N/A"}
                    </p>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span className="text-muted-foreground">Due:</span>
                    <span>{formatDate(payment.dueDate)}</span>
                  </div>
                  {payment.paidDate && (
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Paid:</span>
                      <span>{formatDate(payment.paidDate)}</span>
                    </div>
                  )}
                  {daysOverdue > 0 && (
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Overdue:</span>
                      <span className="text-red-600 font-medium">
                        {daysOverdue} days
                      </span>
                    </div>
                  )}
                  {payment.paymentMethod && (
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Method:</span>
                      <span className="capitalize">
                        {payment.paymentMethod.replace("_", " ")}
                      </span>
                    </div>
                  )}
                </CardContent>
              </Card>
            );
          })}
        </div>
      ) : (
        <Card>
          <CardHeader>
            <CardTitle>Payments ({filteredPayments.length})</CardTitle>
            <CardDescription>
              A list of all payment transactions and their current status
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Tenant</TableHead>
                  <TableHead>Property</TableHead>
                  <TableHead>Type</TableHead>
                  <TableHead>Amount</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Due Date</TableHead>
                  <TableHead>Paid Date</TableHead>
                  <TableHead>Overdue</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredPayments.map((payment) => {
                  const StatusIcon = getStatusIcon(payment.status);
                  const daysOverdue = getDaysOverdue(
                    payment.dueDate,
                    payment.status
                  );

                  return (
                    <TableRow key={payment._id}>
                      <TableCell>
                        <div>
                          <div className="font-medium">
                            {payment.tenantId?.firstName || "N/A"}{" "}
                            {payment.tenantId?.lastName || ""}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            {payment.tenantId?.email || "N/A"}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div>
                          <div className="font-medium">
                            {payment.propertyId?.name || "N/A"}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            {payment.propertyId?.address?.city || "N/A"},{" "}
                            {payment.propertyId?.address?.state || "N/A"}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge
                          variant={getTypeColor(payment.type) as any}
                          className="capitalize"
                        >
                          {payment.type.replace("_", " ")}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="font-medium">
                          {formatCurrency(payment.amount)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge
                          variant={getStatusColor(payment.status) as any}
                          className="flex items-center gap-1 w-fit"
                        >
                          <StatusIcon className="h-3 w-3" />
                          {payment.status.charAt(0).toUpperCase() +
                            payment.status.slice(1)}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-1 text-sm">
                          <Calendar className="h-3 w-3 text-muted-foreground" />
                          <span>{formatDate(payment.dueDate)}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        {payment.paidDate ? (
                          <div className="flex items-center space-x-1 text-sm text-green-600">
                            <CheckCircle className="h-3 w-3" />
                            <span>{formatDate(payment.paidDate)}</span>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">
                            Not paid
                          </span>
                        )}
                      </TableCell>
                      <TableCell>
                        {daysOverdue > 0 ? (
                          <div className="text-sm text-red-600 font-medium">
                            {daysOverdue} days
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem asChild>
                              <Link href={`/dashboard/payments/${payment._id}`}>
                                <Eye className="mr-2 h-4 w-4" />
                                View Details
                              </Link>
                            </DropdownMenuItem>
                            {payment.status !== PaymentStatus.COMPLETED &&
                             payment.status !== PaymentStatus.REFUNDED && (
                              <DropdownMenuItem asChild>
                                <Link
                                  href={`/dashboard/payments/${payment._id}/edit`}
                                >
                                  <Edit className="mr-2 h-4 w-4" />
                                  Edit Payment
                                </Link>
                              </DropdownMenuItem>
                            )}
                            {payment.status === PaymentStatus.PENDING && (
                              <>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem>
                                  <Send className="mr-2 h-4 w-4" />
                                  Send Payment Link
                                </DropdownMenuItem>
                                <DropdownMenuItem className="text-green-600">
                                  <CheckCircle className="mr-2 h-4 w-4" />
                                  Mark as Paid
                                </DropdownMenuItem>
                              </>
                            )}
                            {payment.status === PaymentStatus.COMPLETED && (
                              <>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem
                                  className="text-orange-600"
                                  onClick={() => confirmRefund(payment._id)}
                                >
                                  <RefreshCw className="mr-2 h-4 w-4" />
                                  Refund Payment
                                </DropdownMenuItem>
                              </>
                            )}
                            {/* Only allow delete for PENDING payments that haven't been processed */}
                            {payment.status === PaymentStatus.PENDING && (
                              <>
                                <DropdownMenuSeparator />
                                <DropdownMenuItem
                                  className="text-red-600"
                                  onClick={() => confirmDelete(payment._id)}
                                >
                                  <Trash2 className="mr-2 h-4 w-4" />
                                  Delete Payment
                                </DropdownMenuItem>
                              </>
                            )}
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          </CardContent>
        </Card>
      )}

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={showDeleteDialog} onOpenChange={setShowDeleteDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Are you sure?</AlertDialogTitle>
            <AlertDialogDescription>
              This action cannot be undone. This will permanently delete the
              payment record.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeletePayment}
              disabled={isDeleting}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
            >
              {isDeleting ? "Deleting..." : "Delete"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      {/* Refund Confirmation Dialog */}
      <AlertDialog open={showRefundDialog} onOpenChange={setShowRefundDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Refund Payment</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to refund this payment? This action cannot be undone.
              The payment status will be changed to REFUNDED.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleRefund}
              disabled={isRefunding}
              className="bg-orange-600 text-white hover:bg-orange-700"
            >
              {isRefunding ? "Refunding..." : "Refund"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
