"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  CreditCard,
  Building2,
  Calendar,
  DollarSign,
  AlertTriangle,
  CheckCircle,
  Clock,
  Receipt,
  ArrowLeft,
  Download,
} from "lucide-react";
import Link from "next/link";
import { UserRole, PaymentStatus, PaymentType } from "@/types";
import { StripePayment } from "@/components/payments/stripe-payment";
import { downloadInvoiceAsPDF, PrintableInvoice } from "@/lib/invoice-print";
import { normalizeInvoiceForPrint } from "@/lib/invoice/invoice-shared";

// ============================================================================
// INTERFACES
// ============================================================================

interface InvoiceLineItemLite {
  description: string;
  amount?: number;
  type?: string;
  quantity?: number;
  unitPrice?: number;
}

interface InvoiceSummaryLite {
  _id: string;
  invoiceNumber: string;
  issueDate: string;
  dueDate: string;
  status: string;
  subtotal?: number;
  taxAmount?: number;
  totalAmount: number;
  amountPaid: number;
  balanceRemaining: number;
  notes?: string;
  lineItems?: InvoiceLineItemLite[];
  propertyId?: {
    name?: string;
    address?:
      | {
          street?: string;
          city?: string;
          state?: string;
          zipCode?: string;
          [key: string]: any;
        }
      | string;
  };
  tenantId?: {
    firstName?: string;
    lastName?: string;
    email?: string;
  };
}

interface TenantPayment {
  _id: string;
  amount: number;
  type: PaymentType;
  status: PaymentStatus;
  paymentMethod?: string;
  dueDate: string;
  paidDate?: string;
  description?: string;
  notes?: string;
  lateFee?: number;
  invoiceId?: InvoiceSummaryLite;
  propertyId: {
    _id?: string;
    name?: string;
    address?:
      | {
          street?: string;
          city?: string;
          state?: string;
          zipCode?: string;
          [key: string]: any;
        }
      | string;
  };
  leaseId?: {
    _id: string;
    terms?: {
      rentAmount?: number;
      lateFee?: number;
    };
  };
  stripePaymentIntentId?: string;
  createdAt?: string;
}

interface TenantLease {
  _id: string;
  propertyId: {
    _id: string;
    name: string;
    address: {
      street: string;
      city: string;
      state: string;
      zipCode: string;
    };
  };
  terms: {
    rentAmount: number;
    securityDeposit: number;
    lateFee: number;
  };
  startDate: string;
  endDate: string;
  status: string;
}

interface PaymentSummary {
  totalPaid?: number;
  totalPending?: number;
  totalOverdue?: number;
  totalOutstanding?: number;
  paymentsThisMonth?: number;
  paymentsThisYear?: number;
  averagePaymentAmount?: number;
  onTimePaymentRate?: number;
  nextPaymentDue?: {
    amount: number;
    dueDate: string;
    daysUntilDue: number;
  } | null;
  statistics?: {
    totalPayments?: number;
    paidPayments?: number;
    pendingPayments?: number;
    overduePayments?: number;
  };
}

// ============================================================================
// HELPER FUNCTIONS
// ============================================================================

const formatCurrency = (amount: number = 0): string => {
  return new Intl.NumberFormat("en-US", {
    style: "currency",
    currency: "USD",
  }).format(amount || 0);
};

const formatDate = (dateInput: string | Date): string => {
  const date = dateInput instanceof Date ? dateInput : new Date(dateInput);
  if (Number.isNaN(date.getTime())) return "N/A";
  return date.toLocaleDateString("en-US", {
    year: "numeric",
    month: "short",
    day: "numeric",
  });
};

const isOverdue = (dueDate: string | Date): boolean => {
  const due = dueDate instanceof Date ? dueDate : new Date(dueDate);
  if (Number.isNaN(due.getTime())) return false;
  const now = new Date();
  return due < now;
};

const formatAddress = (
  address?: TenantPayment["propertyId"]["address"]
): string => {
  if (!address) return "N/A";
  if (typeof address === "string") return address;
  const parts = [address.street, address.city, address.state, address.zipCode]
    .filter(Boolean)
    .map((part) => part?.toString().trim())
    .filter(Boolean);
  return parts.length ? parts.join(", ") : "N/A";
};

// ============================================================================
// MAIN COMPONENT
// ============================================================================

export default function TenantPayRentPage() {
  const router = useRouter();
  const { data: session, status } = useSession();

  // State management
  const [pendingPayments, setPendingPayments] = useState<TenantPayment[]>([]);
  const [currentLease, setCurrentLease] = useState<TenantLease | null>(null);
  const [paymentSummary, setPaymentSummary] = useState<PaymentSummary | null>(
    null
  );
  const [recentPayments, setRecentPayments] = useState<TenantPayment[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [selectedPayment, setSelectedPayment] = useState<TenantPayment | null>(
    null
  );
  const [showPaymentDialog, setShowPaymentDialog] = useState(false);

  // Authentication check
  useEffect(() => {
    if (status === "loading") return;

    if (!session) {
      router.push("/auth/signin");
      return;
    }

    if (session.user?.role !== UserRole.TENANT) {
      toast.error("Access denied. This page is for tenants only.");
      router.push("/dashboard");
      return;
    }
  }, [session, status, router]);

  // Fetch tenant payment data
  useEffect(() => {
    if (session?.user?.role === UserRole.TENANT) {
      fetchTenantPaymentData();
    }
  }, [session]);

  const fetchTenantPaymentData = async () => {
    try {
      setIsLoading(true);
      setError(null);

      const [pendingRes, summaryRes, recentRes, dashboardRes] =
        await Promise.all([
          fetch("/api/tenant/payments?status=pending,overdue&limit=20"),
          fetch("/api/tenant/payments/summary"),
          fetch("/api/tenant/payments?status=completed&limit=5"),
          fetch("/api/tenant/dashboard"),
        ]);

      if (!pendingRes.ok) {
        throw new Error("Failed to fetch pending payments");
      }
      if (!summaryRes.ok) {
        throw new Error("Failed to fetch payment summary");
      }
      if (!recentRes.ok) {
        throw new Error("Failed to fetch recent payments");
      }
      if (!dashboardRes.ok) {
        throw new Error("Failed to fetch tenant dashboard data");
      }

      const [pendingData, summaryData, recentData, dashboardData] =
        await Promise.all([
          pendingRes.json(),
          summaryRes.json(),
          recentRes.json(),
          dashboardRes.json(),
        ]);

      const pendingList: TenantPayment[] = pendingData.data?.payments || [];
      const summary: PaymentSummary | null = summaryData?.data || null;
      const recentList: TenantPayment[] = recentData.data?.payments || [];

      setPendingPayments(pendingList);
      setPaymentSummary(summary);
      setRecentPayments(recentList);
      setCurrentLease(dashboardData.data?.currentLease || null);
    } catch (error) {
      const errorMessage =
        error instanceof Error ? error.message : "Failed to load payment data";
      setError(errorMessage);
      toast.error(errorMessage);
    } finally {
      setIsLoading(false);
    }
  };

  // Utility functions
  const buildPrintableInvoice = (
    payment: TenantPayment
  ): PrintableInvoice | null => {
    const invoice = payment.invoiceId;
    if (!invoice) {
      const fallbackInvoice = {
        invoiceNumber:
          payment.referenceNumber ||
          `PAY-${payment._id?.slice?.(-8)?.toUpperCase?.() || Date.now()}`,
        issueDate: payment.paidDate || payment.createdAt,
        dueDate: payment.paidDate || payment.createdAt,
        status: payment.status === "completed" ? "paid" : "issued",
        subtotal: payment.amount,
        totalAmount: payment.amount,
        amountPaid: payment.amount,
        balanceRemaining: 0,
        notes: payment.notes,
        tenantId: payment.tenantId,
        propertyId: payment.propertyId,
        lineItems: [
          {
            description:
              payment.description ||
              payment.type
                .replace(/_/g, " ")
                .replace(/\b\w/g, (l) => l.toUpperCase()),
            amount: payment.amount,
            quantity: 1,
            unitPrice: payment.amount,
            total: payment.amount,
            type: payment.type,
          },
        ],
      } as Record<string, unknown>;

      return normalizeInvoiceForPrint(fallbackInvoice) as PrintableInvoice;
    }

    const normalizedInvoice = normalizeInvoiceForPrint(
      {
        ...invoice,
        propertyId: invoice.propertyId || payment.propertyId,
        lineItems:
          Array.isArray(invoice.lineItems) && invoice.lineItems.length > 0
            ? invoice.lineItems
            : [
                {
                  description:
                    payment.description ||
                    payment.type
                      .replace(/_/g, " ")
                      .replace(/\b\w/g, (l) => l.toUpperCase()),
                  amount: payment.amount,
                  quantity: 1,
                  unitPrice: payment.amount,
                  total: payment.amount,
                  type: payment.type,
                },
              ],
      },
      { fallbackStatus: invoice.status }
    );

    return normalizedInvoice as PrintableInvoice;
  };

  const handleDownloadInvoice = async (payment: TenantPayment) => {
    try {
      const printable = buildPrintableInvoice(payment);
      if (!printable) {
        throw new Error("Invoice details are not available yet");
      }

      // Fetch company info from display settings
      const { getCompanyInfo } = await import("@/lib/utils/company-info");
      const companyInfo = await getCompanyInfo();

      // Normalize with company info
      const normalizedPrintable = normalizeInvoiceForPrint(printable, {
        companyInfo: companyInfo || undefined,
      }) as PrintableInvoice;

      await downloadInvoiceAsPDF(normalizedPrintable);
      toast.success("Invoice downloaded successfully");
    } catch (error) {
      toast.error(
        error instanceof Error ? error.message : "Failed to download invoice"
      );
    }
  };

  const handleViewInvoice = (payment: TenantPayment) => {
    if (payment.invoiceId?._id) {
      router.push(`/dashboard/leases/invoices/${payment.invoiceId._id}`);
    } else {
      toast.error("Invoice details are not available yet");
    }
  };

  const handleDownloadReceipt = (payment: TenantPayment) => {
    if (payment.invoiceId) {
      void handleDownloadInvoice(payment);
    } else {
      toast.error("Receipt generation is not available for this payment");
    }
  };

  const getPaymentStatusBadge = (payment: TenantPayment) => {
    const isPaymentOverdue = isOverdue(payment.dueDate);

    if (isPaymentOverdue) {
      return <Badge variant="destructive">Overdue</Badge>;
    }

    if (payment.status === PaymentStatus.PENDING) {
      return <Badge variant="secondary">Pending</Badge>;
    }

    if (payment.status === PaymentStatus.COMPLETED) {
      return (
        <Badge variant="default" className="bg-green-500">
          Paid
        </Badge>
      );
    }

    if (payment.status === PaymentStatus.FAILED) {
      return <Badge variant="destructive">Failed</Badge>;
    }

    return <Badge variant="outline">{payment.status}</Badge>;
  };

  // Loading state
  if (status === "loading" || isLoading) {
    return <PayRentPageSkeleton />;
  }

  // Error state
  if (error) {
    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">Pay Rent</h1>
            <p className="text-muted-foreground">
              Manage your rent payments and view payment history
            </p>
          </div>
        </div>

        <Card>
          <CardContent className="flex items-center justify-center py-12">
            <div className="text-center">
              <AlertTriangle className="h-12 w-12 text-destructive mx-auto mb-4" />
              <h3 className="text-lg font-semibold mb-2">
                Unable to Load Payment Data
              </h3>
              <p className="text-muted-foreground mb-4">{error}</p>
              <Button onClick={fetchTenantPaymentData}>Try Again</Button>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 className="text-2xl font-bold tracking-tight sm:text-3xl">
            Pay Rent
          </h1>
          <p className="text-sm text-muted-foreground sm:text-base">
            Manage your rent payments and view payment history
          </p>
        </div>
        <div className="flex flex-col gap-2 sm:flex-row sm:items-center">
          <Link href="/dashboard/payments/history" className="w-full sm:w-auto">
            <Button variant="outline" className="w-full sm:w-auto">
              <Receipt className="mr-2 h-4 w-4" />
              Payment History
            </Button>
          </Link>
        </div>
      </div>

      {/* Payment Summary Cards */}
      {paymentSummary && (
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Owed</CardTitle>
              <DollarSign className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-destructive">
                {formatCurrency(paymentSummary.totalOutstanding || 0)}
              </div>
              <p className="text-xs text-muted-foreground">
                {paymentSummary.statistics?.overduePayments || 0} overdue
                payments
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                Payments Made
              </CardTitle>
              <CheckCircle className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600">
                {paymentSummary.statistics?.paidPayments || 0}
              </div>
              <p className="text-xs text-muted-foreground">
                {paymentSummary.paymentsThisYear} this year
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                Next Payment
              </CardTitle>
              <Calendar className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {paymentSummary.nextPaymentDue?.dueDate
                  ? formatDate(paymentSummary.nextPaymentDue.dueDate)
                  : "N/A"}
              </div>
              <p className="text-xs text-muted-foreground">
                {paymentSummary.nextPaymentDue
                  ? `Amount ${formatCurrency(
                      paymentSummary.nextPaymentDue.amount
                    )}`
                  : "Due date"}
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                Current Rent
              </CardTitle>
              <Building2 className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {currentLease
                  ? formatCurrency(currentLease.terms.rentAmount)
                  : "N/A"}
              </div>
              <p className="text-xs text-muted-foreground">Monthly rent</p>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Current Lease Information */}
      {currentLease && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Building2 className="h-5 w-5" />
              Current Lease
            </CardTitle>
            <CardDescription>Your active lease information</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-2">
              <div>
                <h4 className="font-semibold mb-2">Property Details</h4>
                <p className="font-medium">{currentLease.propertyId.name}</p>
                <p className="text-sm text-muted-foreground">
                  {currentLease.propertyId.address.street}
                  <br />
                  {currentLease.propertyId.address.city},{" "}
                  {currentLease.propertyId.address.state}{" "}
                  {currentLease.propertyId.address.zipCode}
                </p>
              </div>
              <div>
                <h4 className="font-semibold mb-2">Lease Terms</h4>
                <div className="space-y-1 text-sm">
                  <div className="flex justify-between">
                    <span>Monthly Rent:</span>
                    <span className="font-medium">
                      {formatCurrency(currentLease.terms.rentAmount)}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span>Security Deposit:</span>
                    <span className="font-medium">
                      {formatCurrency(currentLease.terms.securityDeposit)}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span>Late Fee:</span>
                    <span className="font-medium">
                      {formatCurrency(currentLease.terms.lateFee)}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span>Lease Period:</span>
                    <span className="font-medium">
                      {formatDate(currentLease.startDate)} -{" "}
                      {formatDate(currentLease.endDate)}
                    </span>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Pending Payments */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CreditCard className="h-5 w-5" />
            Pending Payments ({pendingPayments.length})
          </CardTitle>
          <CardDescription>
            Payments that require your attention
          </CardDescription>
        </CardHeader>
        <CardContent>
          {pendingPayments.length === 0 ? (
            <div className="text-center py-8">
              <CheckCircle className="h-12 w-12 text-green-500 mx-auto mb-4" />
              <h3 className="text-lg font-semibold mb-2">All Caught Up!</h3>
              <p className="text-muted-foreground">
                You have no pending payments at this time.
              </p>
            </div>
          ) : (
            <div className="space-y-4">
              {pendingPayments.map((payment) => (
                <PaymentCard
                  key={payment._id}
                  payment={payment}
                  onPayNow={() => {
                    setSelectedPayment(payment);
                    setShowPaymentDialog(true);
                  }}
                />
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Payment Processing Dialog */}
      <PaymentDialog
        payment={selectedPayment}
        isOpen={showPaymentDialog}
        onClose={() => {
          setShowPaymentDialog(false);
          setSelectedPayment(null);
        }}
        onSuccess={() => {
          setShowPaymentDialog(false);
          setSelectedPayment(null);
          fetchTenantPaymentData();
        }}
        onError={(error) => {
          toast.error(error);
        }}
      />
    </div>
  );
}

// ============================================================================
// PAYMENT CARD COMPONENT
// ============================================================================

interface PaymentCardProps {
  payment: TenantPayment;
  onPayNow: () => void;
}

function PaymentCard({ payment, onPayNow }: PaymentCardProps) {
  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const isOverdue = (dueDate: string) => {
    return new Date(dueDate) < new Date();
  };

  const getPaymentStatusBadge = (payment: TenantPayment) => {
    const isPaymentOverdue = isOverdue(payment.dueDate);

    if (isPaymentOverdue) {
      return <Badge variant="destructive">Overdue</Badge>;
    }

    if (payment.status === PaymentStatus.PENDING) {
      return <Badge variant="secondary">Pending</Badge>;
    }

    if (payment.status === PaymentStatus.COMPLETED) {
      return (
        <Badge variant="default" className="bg-green-500">
          Paid
        </Badge>
      );
    }

    if (payment.status === PaymentStatus.FAILED) {
      return <Badge variant="destructive">Failed</Badge>;
    }

    return <Badge variant="outline">{payment.status}</Badge>;
  };

  const totalAmount = payment.amount + (payment.lateFee || 0);
  const paymentOverdue = isOverdue(payment.dueDate);

  return (
    <Card
      className={`transition-all hover:shadow-md ${
        paymentOverdue ? "border-destructive" : ""
      }`}
    >
      <CardContent className="p-4 sm:p-6">
        <div className="flex flex-col gap-4 sm:flex-row sm:items-start sm:justify-between">
          <div className="flex-1">
            <div className="flex flex-col gap-2 sm:flex-row sm:items-center sm:gap-3 mb-3">
              <div className="flex items-center gap-2">
                <CreditCard className="h-4 w-4 text-muted-foreground" />
                <span className="font-semibold text-sm sm:text-base">
                  {payment.type
                    .replace("_", " ")
                    .replace(/\b\w/g, (l) => l.toUpperCase())}
                </span>
              </div>
              <div className="flex items-center gap-2 flex-wrap">
                {getPaymentStatusBadge(payment)}
                {paymentOverdue && (
                  <Badge variant="destructive" className="animate-pulse">
                    <AlertTriangle className="h-3 w-3 mr-1" />
                    Overdue
                  </Badge>
                )}
              </div>
            </div>

            <div className="grid gap-3 sm:grid-cols-2">
              <div>
                <p className="text-sm text-muted-foreground">Property</p>
                <p className="font-medium">{payment.propertyId.name}</p>
                <p className="text-xs text-muted-foreground">
                  {payment.propertyId.address.street}
                </p>
              </div>

              <div>
                <p className="text-sm text-muted-foreground">Due Date</p>
                <p
                  className={`font-medium ${
                    paymentOverdue ? "text-destructive" : ""
                  }`}
                >
                  {formatDate(payment.dueDate)}
                </p>
                {paymentOverdue && (
                  <p className="text-xs text-destructive">
                    {Math.ceil(
                      (new Date().getTime() -
                        new Date(payment.dueDate).getTime()) /
                        (1000 * 60 * 60 * 24)
                    )}{" "}
                    days overdue
                  </p>
                )}
              </div>
            </div>

            {payment.description && (
              <div className="mt-2">
                <p className="text-sm text-muted-foreground">Description</p>
                <p className="text-sm">{payment.description}</p>
              </div>
            )}
          </div>

          <div className="w-full sm:w-auto sm:text-right sm:ml-6">
            <div className="mb-3 sm:mb-2">
              <p className="text-sm text-muted-foreground">Amount Due</p>
              <p className="text-xl sm:text-2xl font-bold">
                {formatCurrency(payment.amount)}
              </p>
              {payment.lateFee && payment.lateFee > 0 && (
                <p className="text-sm text-destructive">
                  + {formatCurrency(payment.lateFee)} late fee
                </p>
              )}
            </div>

            <div className="space-y-2">
              <Button
                onClick={onPayNow}
                className="w-full sm:w-auto sm:min-w-[140px]"
                variant={paymentOverdue ? "destructive" : "default"}
                size="sm"
              >
                <CreditCard className="mr-2 h-4 w-4" />
                <span className="hidden sm:inline">Pay </span>
                {formatCurrency(totalAmount)}
              </Button>

              {payment.lateFee && payment.lateFee > 0 && (
                <p className="text-xs text-muted-foreground text-center sm:text-right">
                  Total: {formatCurrency(totalAmount)}
                </p>
              )}
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}

// ============================================================================
// PAYMENT DIALOG COMPONENT
// ============================================================================

interface PaymentDialogProps {
  payment: TenantPayment | null;
  isOpen: boolean;
  onClose: () => void;
  onSuccess: () => void;
  onError: (error: string) => void;
}

function PaymentDialog({
  payment,
  isOpen,
  onClose,
  onSuccess,
  onError,
}: PaymentDialogProps) {
  const [isProcessing, setIsProcessing] = useState(false);

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const handlePaymentSuccess = (paymentIntentId: string) => {
    setIsProcessing(false);
    onSuccess();
  };

  const handlePaymentError = (error: string) => {
    setIsProcessing(false);
    onError(error);
  };

  if (!payment) return null;

  const totalAmount = payment.amount + (payment.lateFee || 0);

  return (
    <AlertDialog open={isOpen} onOpenChange={onClose}>
      <AlertDialogContent className="max-w-2xl">
        <AlertDialogHeader>
          <AlertDialogTitle className="flex items-center gap-2">
            <CreditCard className="h-5 w-5" />
            Process Payment
          </AlertDialogTitle>
          <AlertDialogDescription>
            Complete your payment securely using Stripe
          </AlertDialogDescription>
        </AlertDialogHeader>

        <div className="space-y-4">
          {/* Payment Summary */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Payment Summary</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                <div className="flex justify-between">
                  <span>Payment Type:</span>
                  <span className="font-medium">
                    {payment.type
                      .replace("_", " ")
                      .replace(/\b\w/g, (l) => l.toUpperCase())}
                  </span>
                </div>
                <div className="flex justify-between">
                  <span>Property:</span>
                  <span className="font-medium">{payment.propertyId.name}</span>
                </div>
                <div className="flex justify-between">
                  <span>Base Amount:</span>
                  <span className="font-medium">
                    {formatCurrency(payment.amount)}
                  </span>
                </div>
                {payment.lateFee && payment.lateFee > 0 && (
                  <div className="flex justify-between text-destructive">
                    <span>Late Fee:</span>
                    <span className="font-medium">
                      {formatCurrency(payment.lateFee)}
                    </span>
                  </div>
                )}
                <div className="border-t pt-2">
                  <div className="flex justify-between text-lg font-bold">
                    <span>Total Amount:</span>
                    <span>{formatCurrency(totalAmount)}</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Stripe Payment Form */}
          <div className="min-h-[200px]">
            <StripePayment
              paymentId={payment._id}
              amount={totalAmount}
              description={`${payment.type
                .replace("_", " ")
                .replace(/\b\w/g, (l) => l.toUpperCase())} payment for ${
                payment.propertyId.name
              }`}
              onSuccess={handlePaymentSuccess}
              onError={handlePaymentError}
            />
          </div>
        </div>

        <AlertDialogFooter>
          <AlertDialogCancel onClick={onClose} disabled={isProcessing}>
            Cancel
          </AlertDialogCancel>
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  );
}

// ============================================================================
// SKELETON LOADING COMPONENT
// ============================================================================

function PayRentPageSkeleton() {
  return (
    <div className="space-y-6">
      {/* Header Skeleton */}
      <div className="flex items-center justify-between">
        <div>
          <Skeleton className="h-8 w-32 mb-2" />
          <Skeleton className="h-4 w-64" />
        </div>
        <Skeleton className="h-10 w-32" />
      </div>

      {/* Summary Cards Skeleton */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {[...Array(4)].map((_, i) => (
          <Card key={i}>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <Skeleton className="h-4 w-20" />
              <Skeleton className="h-4 w-4" />
            </CardHeader>
            <CardContent>
              <Skeleton className="h-8 w-24 mb-1" />
              <Skeleton className="h-3 w-16" />
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Lease Information Skeleton */}
      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-32 mb-2" />
          <Skeleton className="h-4 w-48" />
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2">
            <div>
              <Skeleton className="h-5 w-24 mb-2" />
              <Skeleton className="h-4 w-32 mb-1" />
              <Skeleton className="h-3 w-40" />
            </div>
            <div>
              <Skeleton className="h-5 w-20 mb-2" />
              <div className="space-y-2">
                {[...Array(4)].map((_, i) => (
                  <div key={i} className="flex justify-between">
                    <Skeleton className="h-3 w-20" />
                    <Skeleton className="h-3 w-16" />
                  </div>
                ))}
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Pending Payments Skeleton */}
      <Card>
        <CardHeader>
          <Skeleton className="h-6 w-40 mb-2" />
          <Skeleton className="h-4 w-56" />
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {[...Array(2)].map((_, i) => (
              <Card key={i}>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <Skeleton className="h-4 w-4" />
                        <Skeleton className="h-4 w-20" />
                        <Skeleton className="h-5 w-16" />
                      </div>
                      <div className="grid gap-2 md:grid-cols-2">
                        <div>
                          <Skeleton className="h-3 w-12 mb-1" />
                          <Skeleton className="h-4 w-24 mb-1" />
                          <Skeleton className="h-3 w-32" />
                        </div>
                        <div>
                          <Skeleton className="h-3 w-16 mb-1" />
                          <Skeleton className="h-4 w-20" />
                        </div>
                      </div>
                    </div>
                    <div className="text-right ml-6">
                      <Skeleton className="h-3 w-16 mb-1" />
                      <Skeleton className="h-8 w-20 mb-2" />
                      <Skeleton className="h-10 w-24" />
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
