"use client";

import Link from "next/link";
import { toast } from "sonner";
import { useSession } from "next-auth/react";
import { useState, useEffect, useCallback } from "react";
import { ErrorAlert } from "@/components/ui/error-alert";
import TenantCard from "@/components/tenants/TenantCard";
import TenantStats from "@/components/tenants/TenantStats";
import { useSearchParams, useRouter } from "next/navigation";
import TenantFilters from "@/components/tenants/TenantFilters";
import BulkStatusDialog from "@/components/tenants/BulkStatusDialog";
import TenantStatusBadge from "@/components/tenants/TenantStatusBadge";
import TenantDeleteDialog from "@/components/tenants/TenantDeleteDialog";
import TenantStatusDialog from "@/components/tenants/TenantStatusDialog";
import type { TenantRecord, TenantStatus } from "@/components/tenants/types";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Checkbox } from "@/components/ui/checkbox";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import {
  Plus,
  MoreHorizontal,
  Edit,
  Eye,
  Trash2,
  Phone,
  Mail,
  Calendar,
  List,
  LayoutGrid,
  RefreshCw,
} from "lucide-react";
import { formatCurrency, formatDate } from "@/lib/utils/formatting";

const currencyDisplayOptions = {
  minimumFractionDigits: 0,
  maximumFractionDigits: 0,
} as const;

const formatTenantDate = (value?: string) =>
  value ? formatDate(value, { format: "medium" }) : "-";

const formatIncome = (amount?: number) =>
  typeof amount === "number"
    ? formatCurrency(amount, undefined, currencyDisplayOptions)
    : null;
type Tenant = TenantRecord;

export default function TenantsPage() {
  const searchParams = useSearchParams();
  const { data: session } = useSession();
  const router = useRouter();

  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [tenants, setTenants] = useState<Tenant[]>([]);
  const [allTenants, setAllTenants] = useState<Tenant[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isStatsLoading, setIsStatsLoading] = useState(true);
  const [viewMode, setViewMode] = useState<"table" | "cards">("table");

  const [statusDialogOpen, setStatusDialogOpen] = useState(false);
  const [selectedTenantForStatus, setSelectedTenantForStatus] =
    useState<Tenant | null>(null);

  const [selectedTenants, setSelectedTenants] = useState<string[]>([]);
  const [bulkStatusDialogOpen, setBulkStatusDialogOpen] = useState(false);

  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [selectedTenantForDelete, setSelectedTenantForDelete] =
    useState<Tenant | null>(null);

  const [currentPage, setCurrentPage] = useState(
    parseInt(searchParams.get("page") || "1")
  );
  const [itemsPerPage, setItemsPerPage] = useState(
    parseInt(searchParams.get("limit") || "10")
  );
  const [totalTenants, setTotalTenants] = useState(0);
  const [totalPages, setTotalPages] = useState(1);

  const fetchAllTenants = useCallback(async () => {
    try {
      setIsStatsLoading(true);

      const params = new URLSearchParams();
      params.set("limit", "1000");

      const response = await fetch(`/api/tenants?${params.toString()}`);
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: Failed to fetch all tenants`);
      }

      const data = await response.json();
      setAllTenants(data.data || []);
    } catch (error) {
      setAllTenants([]);
    } finally {
      setIsStatsLoading(false);
    }
  }, []);

  const [error, setError] = useState<string | null>(null);

  const clearError = () => setError(null);

  const handleStatusChange = (tenant: Tenant) => {
    setSelectedTenantForStatus(tenant);
    setStatusDialogOpen(true);
  };

  const handleStatusUpdate = (newStatus: TenantStatus) => {
    setTenants((prev) =>
      prev.map((tenant) =>
        tenant._id === selectedTenantForStatus?._id
          ? { ...tenant, tenantStatus: newStatus }
          : tenant
      )
    );

    setAllTenants((prev) =>
      prev.map((tenant) =>
        tenant._id === selectedTenantForStatus?._id
          ? { ...tenant, tenantStatus: newStatus }
          : tenant
      )
    );

    setSelectedTenantForStatus(null);
  };

  const handleTenantSelect = (tenantId: string, selected: boolean) => {
    setSelectedTenants((prev) =>
      selected ? [...prev, tenantId] : prev.filter((id) => id !== tenantId)
    );
  };

  const handleSelectAll = (selected: boolean) => {
    setSelectedTenants(selected ? tenants.map((t) => t._id) : []);
  };

  const handleBulkStatusUpdate = (
    tenantIds: string[],
    newStatus: TenantStatus
  ) => {
    setTenants((prev) =>
      prev.map((tenant) =>
        tenantIds.includes(tenant._id)
          ? { ...tenant, tenantStatus: newStatus }
          : tenant
      )
    );

    setAllTenants((prev) =>
      prev.map((tenant) =>
        tenantIds.includes(tenant._id)
          ? { ...tenant, tenantStatus: newStatus }
          : tenant
      )
    );

    setSelectedTenants([]);
  };

  const handleDeleteTenant = (tenant: Tenant) => {
    setSelectedTenantForDelete(tenant);
    setDeleteDialogOpen(true);
  };

  const handleTenantDeleted = (tenantId: string) => {
    setTenants((prev) => prev.filter((tenant) => tenant._id !== tenantId));
    setAllTenants((prev) => prev.filter((tenant) => tenant._id !== tenantId));
    setSelectedTenantForDelete(null);
    setSelectedTenants((prev) => prev.filter((id) => id !== tenantId));
  };

  const fetchTenants = useCallback(async () => {
    try {
      setIsLoading(true);
      clearError();

      const params = new URLSearchParams();
      params.set("page", currentPage.toString());
      params.set("limit", itemsPerPage.toString());
      if (searchTerm) params.set("search", searchTerm);
      if (statusFilter !== "all") params.set("status", statusFilter);

      console.log("🔍 Fetching tenants with params:", {
        page: currentPage,
        limit: itemsPerPage,
        search: searchTerm,
        status: statusFilter,
        url: `/api/tenants?${params.toString()}`,
      });

      const response = await fetch(`/api/tenants?${params.toString()}`);
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: Failed to fetch tenants`);
      }

      const data = await response.json();
      console.log("✅ Received tenants:", data.data?.length, "tenants");
      setTenants(data.data || []);

      if (data.pagination) {
        setTotalTenants(data.pagination.total);
        setTotalPages(data.pagination.totalPages);
      }
    } catch (err) {
      const message =
        err instanceof Error ? err.message : "Failed to fetch tenants";
      setError(message);
      toast.error(message);
    } finally {
      setIsLoading(false);
    }
  }, [currentPage, itemsPerPage, searchTerm, statusFilter]);

  const handleRetry = () => {
    clearError();
    void fetchTenants();
    void fetchAllTenants();
  };

  useEffect(() => {
    void fetchTenants();
  }, [fetchTenants]);

  useEffect(() => {
    void fetchAllTenants();
  }, [fetchAllTenants]);

  const handlePageChange = (page: number) => {
    setCurrentPage(page);
  };

  const handleItemsPerPageChange = (value: string) => {
    const newLimit = parseInt(value);
    setItemsPerPage(newLimit);
    setCurrentPage(1);
  };

  const handleSearchChange = (value: string) => {
    setSearchTerm(value);
    setCurrentPage(1);
  };

  const handleStatusFilterChange = (value: string) => {
    console.log("🎯 Status filter changed to:", value);
    setStatusFilter(value);
    setCurrentPage(1);
  };

  const generatePageNumbers = () => {
    const pages = [];
    const maxVisiblePages = 5;

    if (totalPages <= maxVisiblePages) {
      for (let i = 1; i <= totalPages; i++) {
        pages.push(i);
      }
    } else {
      const startPage = Math.max(1, currentPage - 2);
      const endPage = Math.min(totalPages, currentPage + 2);

      for (let i = startPage; i <= endPage; i++) {
        pages.push(i);
      }
    }

    return pages;
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Tenants</h1>
          <p className="text-muted-foreground">
            Manage tenant applications and profiles
          </p>
        </div>
        <Link href="/dashboard/tenants/new">
          <Button>
            <Plus className="mr-2 h-4 w-4" />
            Add Tenant
          </Button>
        </Link>
      </div>

      {error && (
        <div className="mb-6">
          <ErrorAlert
            title="Error Loading Tenants"
            message={error}
            onRetry={handleRetry}
            onDismiss={clearError}
          />
        </div>
      )}

      <TenantStats tenants={allTenants} isLoading={isStatsLoading} />

      <TenantFilters
        searchTerm={searchTerm}
        onSearchChange={handleSearchChange}
        statusFilter={statusFilter}
        onStatusFilterChange={handleStatusFilterChange}
      />

      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Tenants ({totalTenants})</CardTitle>
              <CardDescription>
                Showing {(currentPage - 1) * itemsPerPage + 1} to{" "}
                {Math.min(currentPage * itemsPerPage, totalTenants)} of{" "}
                {totalTenants} tenants
              </CardDescription>
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant={viewMode === "table" ? "default" : "outline"}
                size="sm"
                onClick={() => setViewMode("table")}
              >
                <List className="h-4 w-4 mr-2" />
                Table
              </Button>
              <Button
                variant={viewMode === "cards" ? "default" : "outline"}
                size="sm"
                onClick={() => setViewMode("cards")}
              >
                <LayoutGrid className="h-4 w-4 mr-2" />
                Cards
              </Button>
            </div>
          </div>

          {selectedTenants.length > 0 && (
            <div className="px-6 py-3 bg-muted/50 border-b">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <span className="text-sm font-medium">
                    {selectedTenants.length} tenant(s) selected
                  </span>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setSelectedTenants([])}
                  >
                    Clear Selection
                  </Button>
                </div>
                <div className="flex items-center gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setBulkStatusDialogOpen(true)}
                  >
                    <RefreshCw className="h-4 w-4 mr-2" />
                    Change Status
                  </Button>
                </div>
              </div>
            </div>
          )}
        </CardHeader>
        <CardContent>
          {isLoading ? (
            viewMode === "cards" ? (
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {[...Array(6)].map((_, i) => (
                  <div key={i} className="space-y-4">
                    <div className="flex items-center space-x-3">
                      <div className="h-12 w-12 bg-muted rounded-full animate-pulse" />
                      <div className="flex-1 space-y-2">
                        <div className="h-4 bg-muted rounded animate-pulse" />
                        <div className="h-3 bg-muted rounded w-3/4 animate-pulse" />
                      </div>
                    </div>
                    <div className="space-y-2">
                      <div className="h-3 bg-muted rounded animate-pulse" />
                      <div className="h-3 bg-muted rounded w-2/3 animate-pulse" />
                      <div className="h-3 bg-muted rounded w-1/2 animate-pulse" />
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="space-y-4">
                {[...Array(5)].map((_, i) => (
                  <div
                    key={i}
                    className="flex items-center space-x-4 p-4 border rounded-lg"
                  >
                    <div className="h-4 w-4 bg-muted rounded animate-pulse" />
                    <div className="h-10 w-10 bg-muted rounded-full animate-pulse" />
                    <div className="flex-1 space-y-2">
                      <div className="h-4 bg-muted rounded animate-pulse" />
                      <div className="h-3 bg-muted rounded w-3/4 animate-pulse" />
                    </div>
                    <div className="h-6 w-16 bg-muted rounded animate-pulse" />
                    <div className="h-8 w-8 bg-muted rounded animate-pulse" />
                  </div>
                ))}
              </div>
            )
          ) : tenants.length === 0 ? (
            <div className="text-center py-8">
              <h3 className="text-lg font-semibold mb-2">No tenants found</h3>
              <p className="text-muted-foreground mb-4">
                No tenants have been created yet.
              </p>
              <Link href="/dashboard/tenants/new">
                <Button>
                  <Plus className="h-4 w-4 mr-2" />
                  Add First Tenant
                </Button>
              </Link>
            </div>
          ) : viewMode === "cards" ? (
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {tenants.map((tenant) => (
                <TenantCard
                  key={tenant._id}
                  tenant={tenant}
                  onEdit={(id) => router.push(`/dashboard/tenants/${id}/edit`)}
                  onDelete={handleDeleteTenant}
                />
              ))}
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-12">
                    <Checkbox
                      checked={
                        selectedTenants.length === tenants.length &&
                        tenants.length > 0
                      }
                      onCheckedChange={handleSelectAll}
                      aria-label="Select all tenants"
                    />
                  </TableHead>
                  <TableHead>Tenant</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Contact</TableHead>
                  <TableHead>Employment</TableHead>
                  <TableHead>Credit Score</TableHead>
                  <TableHead>Application Date</TableHead>
                  <TableHead>Move In/Out</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {tenants.map((tenant) => {
                  return (
                    <TableRow key={tenant._id}>
                      <TableCell>
                        <Checkbox
                          checked={selectedTenants.includes(tenant._id)}
                          onCheckedChange={(checked) =>
                            handleTenantSelect(tenant._id, checked as boolean)
                          }
                          aria-label={`Select ${tenant.firstName} ${tenant.lastName}`}
                        />
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-3">
                          <Avatar className="h-10 w-10">
                            <AvatarImage
                              src={tenant.avatar}
                              alt={`${tenant.firstName} ${tenant.lastName}`}
                            />
                            <AvatarFallback>
                              {tenant.firstName.charAt(0)}
                              {tenant.lastName.charAt(0)}
                            </AvatarFallback>
                          </Avatar>
                          <div>
                            <div className="font-medium">
                              {tenant.firstName} {tenant.lastName}
                            </div>
                            <div className="text-sm text-muted-foreground">
                              {tenant.email}
                            </div>
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        <TenantStatusBadge
                          status={tenant.tenantStatus}
                          showIcon={true}
                          size="default"
                        />
                      </TableCell>
                      <TableCell>
                        <div className="space-y-1">
                          <div className="flex items-center text-sm">
                            <Phone className="h-3 w-3 mr-1" />
                            {tenant.phone}
                          </div>
                          <div className="flex items-center text-sm text-muted-foreground">
                            <Mail className="h-3 w-3 mr-1" />
                            {tenant.email}
                          </div>
                        </div>
                      </TableCell>
                      <TableCell>
                        {tenant.employmentInfo ? (
                          <div className="space-y-1">
                            <div className="text-sm font-medium">
                              {tenant.employmentInfo.employer}
                            </div>
                            <div className="text-sm text-muted-foreground">
                              {tenant.employmentInfo.position}
                            </div>
                            <div className="text-sm text-muted-foreground">
                              {formatIncome(tenant.employmentInfo.income) ??
                                "-"}
                            </div>
                          </div>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {tenant.creditScore !== undefined &&
                        tenant.creditScore !== null ? (
                          <span className="font-medium">
                            {tenant.creditScore}
                          </span>
                        ) : (
                          <span className="text-muted-foreground">-</span>
                        )}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center text-sm">
                          <Calendar className="h-3 w-3 mr-1" />
                          {formatTenantDate(tenant.applicationDate)}
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="space-y-1">
                          {tenant.moveInDate && (
                            <div className="text-sm">
                              In: {formatTenantDate(tenant.moveInDate)}
                            </div>
                          )}
                          {tenant.moveOutDate && (
                            <div className="text-sm text-muted-foreground">
                              Out: {formatTenantDate(tenant.moveOutDate)}
                            </div>
                          )}
                          {!tenant.moveInDate && !tenant.moveOutDate && (
                            <span className="text-muted-foreground">-</span>
                          )}
                        </div>
                      </TableCell>
                      <TableCell className="text-right">
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" className="h-8 w-8 p-0">
                              <span className="sr-only">Open menu</span>
                              <MoreHorizontal className="h-4 w-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Actions</DropdownMenuLabel>
                            <DropdownMenuItem
                              onClick={() =>
                                router.push(`/dashboard/tenants/${tenant._id}`)
                              }
                            >
                              <Eye className="mr-2 h-4 w-4" />
                              View Profile
                            </DropdownMenuItem>
                            <DropdownMenuItem
                              onClick={() =>
                                router.push(
                                  `/dashboard/tenants/${tenant._id}/edit`
                                )
                              }
                            >
                              <Edit className="mr-2 h-4 w-4" />
                              Edit Tenant
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            <DropdownMenuItem
                              onClick={() => handleStatusChange(tenant)}
                            >
                              <RefreshCw className="mr-2 h-4 w-4" />
                              Change Status
                            </DropdownMenuItem>
                            <DropdownMenuSeparator />
                            <DropdownMenuItem
                              className="text-red-600"
                              onClick={() => handleDeleteTenant(tenant)}
                            >
                              <Trash2 className="mr-2 h-4 w-4" />
                              Delete Tenant
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          )}
        </CardContent>

        {totalPages > 1 && (
          <div className="px-6 pb-6">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
              <div className="text-sm text-muted-foreground">
                Showing {(currentPage - 1) * itemsPerPage + 1} to{" "}
                {Math.min(currentPage * itemsPerPage, totalTenants)} of{" "}
                {totalTenants} tenants
              </div>
              <div className="flex items-center space-x-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handlePageChange(currentPage - 1)}
                  disabled={currentPage <= 1}
                >
                  Previous
                </Button>
                {generatePageNumbers().map((pageNum) => (
                  <Button
                    key={pageNum}
                    variant={currentPage === pageNum ? "default" : "outline"}
                    size="sm"
                    onClick={() => handlePageChange(pageNum)}
                    className="min-w-[40px]"
                  >
                    {pageNum}
                  </Button>
                ))}

                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handlePageChange(currentPage + 1)}
                  disabled={currentPage >= totalPages}
                >
                  Next
                </Button>
              </div>
              <div className="flex items-center space-x-2">
                <span className="text-sm text-muted-foreground">
                  Items per page:
                </span>
                <Select
                  value={itemsPerPage.toString()}
                  onValueChange={handleItemsPerPageChange}
                >
                  <SelectTrigger className="w-20">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="10">10</SelectItem>
                    <SelectItem value="25">25</SelectItem>
                    <SelectItem value="50">50</SelectItem>
                    <SelectItem value="100">100</SelectItem>
                    <SelectItem value="1000">All</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </div>
        )}
      </Card>

      {selectedTenantForStatus && (
        <TenantStatusDialog
          isOpen={statusDialogOpen}
          onClose={() => {
            setStatusDialogOpen(false);
            setSelectedTenantForStatus(null);
          }}
          tenant={selectedTenantForStatus}
          onStatusChange={handleStatusUpdate}
          userRole={session?.user?.role || ""}
        />
      )}

      <BulkStatusDialog
        isOpen={bulkStatusDialogOpen}
        onClose={() => setBulkStatusDialogOpen(false)}
        selectedTenants={selectedTenants.map((id) => {
          const tenant = tenants.find((t) => t._id === id);
          return tenant
            ? {
                _id: tenant._id,
                firstName: tenant.firstName,
                lastName: tenant.lastName,
                tenantStatus: tenant.tenantStatus,
              }
            : { _id: id, firstName: "", lastName: "" };
        })}
        onBulkStatusChange={handleBulkStatusUpdate}
        userRole={session?.user?.role || ""}
      />

      <TenantDeleteDialog
        isOpen={deleteDialogOpen}
        onClose={() => setDeleteDialogOpen(false)}
        tenant={
          selectedTenantForDelete || { _id: "", firstName: "", lastName: "" }
        }
        onDelete={handleTenantDeleted}
        userRole={session?.user?.role || ""}
      />
    </div>
  );
}
