import React from "react";
import { cn } from "@/lib/utils";
import { LucideIcon } from "lucide-react";
import { formatCurrency as formatCurrencyValue } from "@/lib/formatters";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";

export interface AnalyticsCardProps {
  title: string;
  value: string | number;
  description?: string;
  icon?: LucideIcon;
  iconColor?: "primary" | "success" | "warning" | "error" | "info";
  trend?: {
    value: string;
    isPositive: boolean;
    icon?: LucideIcon;
  };
  variant?: "default" | "financial" | "metric";
  className?: string;
  children?: React.ReactNode;
}

export interface FinancialCardProps {
  label: string;
  amount: number;
  variant: "success" | "warning" | "error" | "info";
  className?: string;
}

export interface MetricCardProps {
  title: string;
  value: string | number;
  subtitle?: string;
  icon?: LucideIcon;
  iconColor?: "primary" | "success" | "warning" | "error" | "info";
  className?: string;
}

const iconColorClasses = {
  primary: "text-primary bg-primary/10",
  success: "text-success bg-success/10",
  warning: "text-warning bg-warning/10",
  error: "text-error bg-error/10",
  info: "text-info bg-info/10",
};

const financialVariantClasses = {
  success: {
    background: "bg-gradient-to-r from-success/15 to-success/8",
    border: "border-success/15",
    text: "text-success",
  },
  warning: {
    background: "bg-gradient-to-r from-warning/15 to-warning/8",
    border: "border-warning/15",
    text: "text-warning",
  },
  error: {
    background: "bg-gradient-to-r from-error/15 to-error/8",
    border: "border-error/15",
    text: "text-error",
  },
  info: {
    background: "bg-gradient-to-r from-info/15 to-info/8",
    border: "border-info/15",
    text: "text-info",
  },
};

export function AnalyticsCard({
  title,
  value,
  description,
  icon: Icon,
  iconColor = "primary",
  trend,
  variant = "default",
  className,
  children,
}: AnalyticsCardProps) {
  const TrendIcon = trend?.icon;

  return (
    <Card
      className={cn("hover:shadow-md transition-shadow py-4 gap-2", className)}
    >
      <CardHeader className="flex flex-row items-center justify-between space-y-0">
        <CardTitle className="text-sm font-medium">{title}</CardTitle>
        {Icon && (
          <div className={cn("p-2 rounded-lg", iconColorClasses[iconColor])}>
            <Icon className="h-4 w-4" />
          </div>
        )}
      </CardHeader>
      <CardContent>
        <div className="text-2xl font-bold">{value}</div>
        {description && (
          <p className="text-xs text-muted-foreground mt-1">{description}</p>
        )}
        {trend && (
          <div className="flex items-center text-xs mt-2">
            {TrendIcon && (
              <TrendIcon
                className={cn(
                  "h-3 w-3 mr-1",
                  trend.isPositive ? "text-green-600" : "text-red-600"
                )}
              />
            )}
            <span
              className={cn(
                "flex items-center",
                trend.isPositive ? "text-green-600" : "text-red-600"
              )}
            >
              {trend.value}
            </span>
          </div>
        )}
        {children}
      </CardContent>
    </Card>
  );
}

export function FinancialCard({
  label,
  amount,
  variant,
  className,
}: FinancialCardProps) {
  const variantClasses = financialVariantClasses[variant];

  return (
    <div
      className={cn(
        "flex flex-col sm:flex-row sm:justify-between sm:items-center gap-2 p-3 lg:p-4 rounded-xl border",
        variantClasses.background,
        variantClasses.border,
        className
      )}
    >
      <span
        className={cn(
          "font-semibold text-sm lg:text-base",
          variantClasses.text
        )}
      >
        {label}
      </span>
      <span
        className={cn("font-bold text-base lg:text-lg", variantClasses.text)}
      >
        {formatCurrencyValue(amount)}
      </span>
    </div>
  );
}

export function MetricCard({
  title,
  value,
  subtitle,
  icon: Icon,
  iconColor = "primary",
  className,
}: MetricCardProps) {
  return (
    <Card className={cn("hover:shadow-md transition-shadow", className)}>
      <CardHeader>
        <CardTitle className="flex items-center gap-3 text-lg">
          {Icon && (
            <div className={cn("p-2 rounded-lg", iconColorClasses[iconColor])}>
              <Icon className="h-5 w-5" />
            </div>
          )}
          {title}
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="text-2xl font-bold">{value}</div>
        {subtitle && (
          <p className="text-xs text-muted-foreground">{subtitle}</p>
        )}
      </CardContent>
    </Card>
  );
}

export function AnalyticsCardGrid({
  children,
  className,
}: {
  children: React.ReactNode;
  className?: string;
}) {
  return (
    <div
      className={cn(
        "grid grid-4 gap-4 grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-5",
        className
      )}
    >
      {children}
    </div>
  );
}

export function FinancialCardGrid({
  children,
  className,
}: {
  children: React.ReactNode;
  className?: string;
}) {
  return (
    <div className={cn("grid grid-cols-1 gap-3 lg:gap-4", className)}>
      {children}
    </div>
  );
}
