"use client";

import React from "react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Input } from "@/components/ui/input";
import { Separator } from "@/components/ui/separator";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Settings,
  Clock,
  Calendar,
  Globe,
  Eye,
  Mail,
  Send,
} from "lucide-react";
import { toast } from "sonner";

interface CalendarSettingsProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  settings: {
    weekends: boolean;
    businessHours: {
      startTime: string;
      endTime: string;
    };
    defaultEventDuration: string;
    slotDuration: string;
    snapDuration: string;
    timezone: string;
    firstDay: number;
  };
  onSettingsChange: (settings: any) => void;
}

const timezones = [
  { value: "local", label: "Local Time" },
  { value: "UTC", label: "UTC" },
  { value: "America/New_York", label: "Eastern Time" },
  { value: "America/Chicago", label: "Central Time" },
  { value: "America/Denver", label: "Mountain Time" },
  { value: "America/Los_Angeles", label: "Pacific Time" },
  { value: "Europe/London", label: "London" },
  { value: "Europe/Paris", label: "Paris" },
  { value: "Asia/Tokyo", label: "Tokyo" },
  { value: "Asia/Shanghai", label: "Shanghai" },
  { value: "Australia/Sydney", label: "Sydney" },
];

const daysOfWeek = [
  { value: 0, label: "Sunday" },
  { value: 1, label: "Monday" },
  { value: 2, label: "Tuesday" },
  { value: 3, label: "Wednesday" },
  { value: 4, label: "Thursday" },
  { value: 5, label: "Friday" },
  { value: 6, label: "Saturday" },
];

const timeSlots = [
  { value: "00:15", label: "15 minutes" },
  { value: "00:30", label: "30 minutes" },
  { value: "01:00", label: "1 hour" },
  { value: "02:00", label: "2 hours" },
];

export function CalendarSettings({
  open,
  onOpenChange,
  settings,
  onSettingsChange,
}: CalendarSettingsProps) {
  const [emailTesting, setEmailTesting] = React.useState(false);

  const updateSetting = (key: string, value: any) => {
    if (key.includes(".")) {
      const [parent, child] = key.split(".");
      onSettingsChange({
        ...settings,
        [parent]: {
          ...settings[parent as keyof typeof settings],
          [child]: value,
        },
      });
    } else {
      onSettingsChange({
        ...settings,
        [key]: value,
      });
    }
  };

  const handleTestEmail = async () => {
    setEmailTesting(true);
    try {
      const response = await fetch("/api/test/email", {
        method: "POST",
      });

      const data = await response.json();

      if (response.ok && data.success) {
        toast.success("Test email sent successfully! Check your inbox.");
      } else {
        toast.error(data.message || "Failed to send test email");
      }
    } catch (error) {
      console.error("Email test error:", error);
      toast.error("Failed to send test email");
    } finally {
      setEmailTesting(false);
    }
  };

  const handleSave = () => {
    // Save settings to localStorage or API
    localStorage.setItem("calendarSettings", JSON.stringify(settings));
    onOpenChange(false);
  };

  const handleReset = () => {
    const defaultSettings = {
      weekends: true,
      businessHours: {
        startTime: "09:00",
        endTime: "17:00",
      },
      defaultEventDuration: "01:00",
      slotDuration: "00:30",
      snapDuration: "00:15",
      timezone: "local",
      firstDay: 0,
    };
    onSettingsChange(defaultSettings);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Settings className="h-5 w-5" />
            Calendar Settings
          </DialogTitle>
          <DialogDescription>
            Customize your calendar view and behavior preferences.
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-6">
          {/* Display Settings */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <Eye className="h-4 w-4" />
                Display Settings
              </CardTitle>
              <CardDescription>
                Configure how your calendar appears and behaves.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="weekends">Show Weekends</Label>
                  <p className="text-sm text-muted-foreground">
                    Display Saturday and Sunday in calendar views
                  </p>
                </div>
                <Switch
                  id="weekends"
                  checked={settings.weekends}
                  onCheckedChange={(checked) =>
                    updateSetting("weekends", checked)
                  }
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="firstDay">Week Starts On</Label>
                <Select
                  value={settings.firstDay.toString()}
                  onValueChange={(value) =>
                    updateSetting("firstDay", parseInt(value))
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {daysOfWeek.map((day) => (
                      <SelectItem key={day.value} value={day.value.toString()}>
                        {day.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="timezone">Timezone</Label>
                <Select
                  value={settings.timezone}
                  onValueChange={(value) => updateSetting("timezone", value)}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {timezones.map((tz) => (
                      <SelectItem key={tz.value} value={tz.value}>
                        {tz.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>

          {/* Time Settings */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-lg">
                <Clock className="h-4 w-4" />
                Time Settings
              </CardTitle>
              <CardDescription>
                Configure time slots and business hours.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="startTime">Business Hours Start</Label>
                  <Input
                    id="startTime"
                    type="time"
                    value={settings.businessHours.startTime}
                    onChange={(e) =>
                      updateSetting("businessHours.startTime", e.target.value)
                    }
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="endTime">Business Hours End</Label>
                  <Input
                    id="endTime"
                    type="time"
                    value={settings.businessHours.endTime}
                    onChange={(e) =>
                      updateSetting("businessHours.endTime", e.target.value)
                    }
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="slotDuration">Time Slot Duration</Label>
                <Select
                  value={settings.slotDuration}
                  onValueChange={(value) =>
                    updateSetting("slotDuration", value)
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {timeSlots.map((slot) => (
                      <SelectItem key={slot.value} value={slot.value}>
                        {slot.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="snapDuration">Snap Duration</Label>
                <Select
                  value={settings.snapDuration}
                  onValueChange={(value) =>
                    updateSetting("snapDuration", value)
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {timeSlots.map((slot) => (
                      <SelectItem key={slot.value} value={slot.value}>
                        {slot.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="defaultDuration">Default Event Duration</Label>
                <Select
                  value={settings.defaultEventDuration}
                  onValueChange={(value) =>
                    updateSetting("defaultEventDuration", value)
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {timeSlots.map((slot) => (
                      <SelectItem key={slot.value} value={slot.value}>
                        {slot.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>

          {/* Email Service Test */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Mail className="h-5 w-5" />
                Email Service
              </CardTitle>
              <CardDescription>
                Test email service configuration and send test emails
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium">Email Service Status</p>
                  <p className="text-xs text-muted-foreground">
                    Test if email invitations and notifications are working
                  </p>
                </div>
                <Button
                  onClick={handleTestEmail}
                  disabled={emailTesting}
                  variant="outline"
                  size="sm"
                >
                  <Send className="h-4 w-4 mr-2" />
                  {emailTesting ? "Sending..." : "Send Test Email"}
                </Button>
              </div>
            </CardContent>
          </Card>

          <Separator />

          {/* Action Buttons */}
          <div className="flex justify-between">
            <Button variant="outline" onClick={handleReset}>
              Reset to Defaults
            </Button>
            <div className="flex gap-2">
              <Button variant="outline" onClick={() => onOpenChange(false)}>
                Cancel
              </Button>
              <Button onClick={handleSave}>Save Settings</Button>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
