"use client";

import React from "react";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Clock,
  MapPin,
  Users,
  Calendar,
  Edit,
  Trash2,
  Copy,
  Mail,
  MoreVertical,
  User,
  Building2,
  AlertTriangle,
  CheckCircle,
  XCircle,
} from "lucide-react";
import { format } from "date-fns";
import { IEvent, EventType, EventStatus, EventPriority } from "@/types";

interface EventDetailsDialogProps {
  event: IEvent;
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onEdit?: (event: IEvent) => void;
  onDelete?: (eventId: string) => void;
  onDuplicate?: (event: IEvent) => void;
  onSendReminder?: (event: IEvent) => void;
}

const eventTypeColors = {
  [EventType.LEASE_RENEWAL]: "#3b82f6",
  [EventType.PROPERTY_INSPECTION]: "#10b981",
  [EventType.MAINTENANCE_APPOINTMENT]: "#f59e0b",
  [EventType.PROPERTY_SHOWING]: "#8b5cf6",
  [EventType.TENANT_MEETING]: "#6366f1",
  [EventType.RENT_COLLECTION]: "#059669",
  [EventType.MOVE_IN]: "#06b6d4",
  [EventType.MOVE_OUT]: "#ef4444",
  [EventType.GENERAL]: "#6b7280",
};

const eventStatusIcons = {
  [EventStatus.SCHEDULED]: Clock,
  [EventStatus.CONFIRMED]: CheckCircle,
  [EventStatus.IN_PROGRESS]: AlertTriangle,
  [EventStatus.COMPLETED]: CheckCircle,
  [EventStatus.CANCELLED]: XCircle,
  [EventStatus.RESCHEDULED]: Clock,
};

const getStatusColor = (status: EventStatus) => {
  switch (status) {
    case EventStatus.SCHEDULED:
      return "bg-blue-100 text-blue-800 border-blue-200";
    case EventStatus.CONFIRMED:
      return "bg-green-100 text-green-800 border-green-200";
    case EventStatus.IN_PROGRESS:
      return "bg-yellow-100 text-yellow-800 border-yellow-200";
    case EventStatus.COMPLETED:
      return "bg-emerald-100 text-emerald-800 border-emerald-200";
    case EventStatus.CANCELLED:
      return "bg-red-100 text-red-800 border-red-200";
    case EventStatus.RESCHEDULED:
      return "bg-orange-100 text-orange-800 border-orange-200";
    default:
      return "bg-gray-100 text-gray-800 border-gray-200";
  }
};

const getPriorityColor = (priority: EventPriority) => {
  switch (priority) {
    case EventPriority.HIGH:
      return "bg-red-100 text-red-800 border-red-200";
    case EventPriority.MEDIUM:
      return "bg-yellow-100 text-yellow-800 border-yellow-200";
    case EventPriority.LOW:
      return "bg-green-100 text-green-800 border-green-200";
    default:
      return "bg-gray-100 text-gray-800 border-gray-200";
  }
};

export function EventDetailsDialog({
  event,
  open,
  onOpenChange,
  onEdit,
  onDelete,
  onDuplicate,
  onSendReminder,
}: EventDetailsDialogProps) {
  const [showDeleteDialog, setShowDeleteDialog] = React.useState(false);
  const StatusIcon = eventStatusIcons[event.status];

  const formatEventTime = (event: IEvent) => {
    if (event.allDay) return "All day";
    const start = format(new Date(event.startDate), "h:mm a");
    const end = event.endDate ? format(new Date(event.endDate), "h:mm a") : "";
    return end ? `${start} - ${end}` : start;
  };

  const getEventTypeLabel = (type: EventType) => {
    return type.replace(/_/g, " ").replace(/\b\w/g, (l) => l.toUpperCase());
  };

  const handleDelete = () => {
    if (onDelete) {
      onDelete(event._id);
    }
    setShowDeleteDialog(false);
    onOpenChange(false);
  };

  return (
    <>
      <Dialog open={open} onOpenChange={onOpenChange}>
        <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <div className="flex items-start justify-between">
              <div className="flex-1">
                <DialogTitle className="text-xl font-semibold pr-8">
                  {event.title}
                </DialogTitle>
                <DialogDescription className="mt-1">
                  <div className="flex items-center gap-2">
                    <div
                      className="w-3 h-3 rounded-full"
                      style={{ backgroundColor: eventTypeColors[event.type] }}
                    />
                    {getEventTypeLabel(event.type)}
                  </div>
                </DialogDescription>
              </div>

              {/* Only show action menu if user has edit/delete permissions */}
              {(onEdit || onDelete || onDuplicate || onSendReminder) && (
                <DropdownMenu>
                  <DropdownMenuTrigger asChild>
                    <Button variant="ghost" size="sm">
                      <MoreVertical className="h-4 w-4" />
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="end">
                    {onEdit && (
                      <DropdownMenuItem onClick={() => onEdit(event)}>
                        <Edit className="h-4 w-4 mr-2" />
                        Edit Event
                      </DropdownMenuItem>
                    )}
                    {onDuplicate && (
                      <DropdownMenuItem onClick={() => onDuplicate(event)}>
                        <Copy className="h-4 w-4 mr-2" />
                        Duplicate Event
                      </DropdownMenuItem>
                    )}
                    {onSendReminder && (
                      <DropdownMenuItem onClick={() => onSendReminder(event)}>
                        <Mail className="h-4 w-4 mr-2" />
                        Send Reminder
                      </DropdownMenuItem>
                    )}
                    {onDelete && (
                      <>
                        <DropdownMenuSeparator />
                        <DropdownMenuItem
                          onClick={() => setShowDeleteDialog(true)}
                          className="text-red-600 focus:text-red-600"
                        >
                          <Trash2 className="h-4 w-4 mr-2" />
                          Delete Event
                        </DropdownMenuItem>
                      </>
                    )}
                  </DropdownMenuContent>
                </DropdownMenu>
              )}
            </div>
          </DialogHeader>

          <div className="space-y-6">
            {/* Status and Priority */}
            <div className="flex items-center gap-4">
              <div className="flex items-center gap-2">
                <StatusIcon className="h-4 w-4" />
                <Badge className={getStatusColor(event.status)}>
                  {event.status.replace(/_/g, " ")}
                </Badge>
              </div>
              <Badge className={getPriorityColor(event.priority)}>
                {event.priority} Priority
              </Badge>
            </div>

            {/* Date and Time */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <div className="flex items-center gap-2 text-sm font-medium">
                  <Calendar className="h-4 w-4" />
                  Date
                </div>
                <p className="text-sm text-muted-foreground">
                  {format(new Date(event.startDate), "EEEE, MMMM d, yyyy")}
                </p>
              </div>
              <div className="space-y-2">
                <div className="flex items-center gap-2 text-sm font-medium">
                  <Clock className="h-4 w-4" />
                  Time
                </div>
                <p className="text-sm text-muted-foreground">
                  {formatEventTime(event)}
                </p>
              </div>
            </div>

            {/* Location */}
            {event.location && (
              <div className="space-y-2">
                <div className="flex items-center gap-2 text-sm font-medium">
                  <MapPin className="h-4 w-4" />
                  Location
                </div>
                <div className="text-sm text-muted-foreground">
                  {event.location.type === "physical" ? (
                    <div>
                      <p>{event.location.address}</p>
                      {event.unitNumber && (
                        <p className="text-xs text-muted-foreground">
                          Unit: {event.unitNumber}
                        </p>
                      )}
                    </div>
                  ) : (
                    <div>
                      <p className="font-medium">Online Meeting</p>
                      {event.location.platform && (
                        <p className="text-xs">
                          Platform: {event.location.platform.replace("_", " ")}
                        </p>
                      )}
                      {event.location.meetingLink && (
                        <a
                          href={event.location.meetingLink}
                          target="_blank"
                          rel="noopener noreferrer"
                          className="text-blue-600 hover:text-blue-800 underline break-all"
                        >
                          {event.location.meetingLink}
                        </a>
                      )}
                      {event.location.meetingId && (
                        <p className="text-xs">
                          Meeting ID: {event.location.meetingId}
                        </p>
                      )}
                      {event.location.passcode && (
                        <p className="text-xs">
                          Passcode: {event.location.passcode}
                        </p>
                      )}
                    </div>
                  )}
                </div>
              </div>
            )}

            {/* Property Information */}
            {event.propertyId && (
              <div className="space-y-2">
                <div className="flex items-center gap-2 text-sm font-medium">
                  <Building2 className="h-4 w-4" />
                  Property
                </div>
                <p className="text-sm text-muted-foreground">
                  Property ID: {event.propertyId}
                  {event.unitNumber && ` - Unit ${event.unitNumber}`}
                </p>
              </div>
            )}

            {/* Description */}
            {event.description && (
              <div className="space-y-2">
                <div className="text-sm font-medium">Description</div>
                <p className="text-sm text-muted-foreground whitespace-pre-wrap">
                  {event.description}
                </p>
              </div>
            )}

            {/* Attendees */}
            {event.attendees && event.attendees.length > 0 && (
              <div className="space-y-2">
                <div className="flex items-center gap-2 text-sm font-medium">
                  <Users className="h-4 w-4" />
                  Attendees ({event.attendees.length})
                </div>
                <div className="space-y-2">
                  {event.attendees.map((attendee, index) => (
                    <div
                      key={index}
                      className="flex items-center justify-between p-2 rounded-lg border"
                    >
                      <div className="flex items-center gap-2">
                        <User className="h-4 w-4 text-muted-foreground" />
                        <div>
                          <p className="text-sm font-medium">{attendee.name}</p>
                          <p className="text-xs text-muted-foreground">
                            {attendee.email}
                          </p>
                        </div>
                      </div>
                      <Badge variant="outline" className="text-xs">
                        {attendee.status}
                      </Badge>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Notes */}
            {event.notes && (
              <div className="space-y-2">
                <div className="text-sm font-medium">Notes</div>
                <p className="text-sm text-muted-foreground whitespace-pre-wrap">
                  {event.notes}
                </p>
              </div>
            )}

            {/* Metadata */}
            <Separator />
            <div className="text-xs text-muted-foreground">
              <p>Created: {format(new Date(event.createdAt), "PPp")}</p>
              {event.updatedAt && (
                <p>Updated: {format(new Date(event.updatedAt), "PPp")}</p>
              )}
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={showDeleteDialog} onOpenChange={setShowDeleteDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Event</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete "{event.title}"? This action
              cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDelete}
              className="bg-red-600 hover:bg-red-700"
            >
              Delete Event
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  );
}
