"use client";

import { z } from "zod";
import { useForm } from "react-hook-form";
import { useState, useEffect } from "react";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { zodResolver } from "@hookform/resolvers/zod";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Alert, AlertDescription } from "@/components/ui/alert";
import {
  AlertTriangle,
  Zap,
  Phone,
  Clock,
  Upload,
  X,
  CheckCircle,
} from "lucide-react";
import { toast } from "sonner";

// Emergency-specific validation schema
const emergencyRequestSchema = z.object({
  title: z.string().min(1, "Title is required").max(200, "Title too long"),
  description: z
    .string()
    .min(10, "Description must be at least 10 characters")
    .max(2000, "Description too long"),
  emergencyType: z.enum([
    "water_leak",
    "electrical_hazard",
    "gas_leak",
    "security_breach",
    "fire_hazard",
    "structural_damage",
    "hvac_failure",
    "other",
  ]),
  category: z.enum([
    "Emergency",
    "Plumbing",
    "Electrical",
    "HVAC",
    "Security",
    "Other",
  ]),
  propertyId: z.string().min(1, "Property is required"),
  tenantId: z.string().min(1, "Tenant is required"),
  contactPhone: z.string().optional(),
  immediateAction: z.string().optional(),
  safetyRisk: z.enum(["low", "medium", "high", "critical"]),
  images: z.array(z.string()).optional(),
});

type EmergencyRequestFormData = z.infer<typeof emergencyRequestSchema>;

interface Property {
  id: string;
  name: string;
  address: string;
}

interface Tenant {
  id: string;
  name: string;
  email: string;
  phone: string;
  propertyName?: string;
}

interface EmergencyRequestFormProps {
  onSubmit: (data: EmergencyRequestFormData) => Promise<void>;
  isLoading?: boolean;
  properties?: Property[];
  tenants?: Tenant[];
  initialData?: Partial<EmergencyRequestFormData>;
}

const emergencyTypes = [
  { value: "water_leak", label: "Water Leak", icon: "💧", severity: "high" },
  {
    value: "electrical_hazard",
    label: "Electrical Hazard",
    icon: "⚡",
    severity: "critical",
  },
  { value: "gas_leak", label: "Gas Leak", icon: "🔥", severity: "critical" },
  {
    value: "security_breach",
    label: "Security Breach",
    icon: "🔒",
    severity: "high",
  },
  {
    value: "fire_hazard",
    label: "Fire Hazard",
    icon: "🔥",
    severity: "critical",
  },
  {
    value: "structural_damage",
    label: "Structural Damage",
    icon: "🏗️",
    severity: "high",
  },
  {
    value: "hvac_failure",
    label: "HVAC Failure",
    icon: "❄️",
    severity: "medium",
  },
  { value: "other", label: "Other Emergency", icon: "⚠️", severity: "medium" },
];

const safetyRiskLevels = [
  {
    value: "low",
    label: "Low Risk",
    color: "text-green-600",
    bg: "bg-green-50",
  },
  {
    value: "medium",
    label: "Medium Risk",
    color: "text-yellow-600",
    bg: "bg-yellow-50",
  },
  {
    value: "high",
    label: "High Risk",
    color: "text-orange-600",
    bg: "bg-orange-50",
  },
  {
    value: "critical",
    label: "Critical Risk",
    color: "text-red-600",
    bg: "bg-red-50",
  },
];

export function EmergencyRequestForm({
  onSubmit,
  isLoading = false,
  properties = [],
  tenants = [],
  initialData,
}: EmergencyRequestFormProps) {
  const [uploadedImages, setUploadedImages] = useState<string[]>([]);
  const [uploadingImages, setUploadingImages] = useState(false);

  const form = useForm<EmergencyRequestFormData>({
    resolver: zodResolver(emergencyRequestSchema),
    defaultValues: {
      title: "",
      description: "",
      emergencyType: "other",
      category: "Emergency",
      propertyId: "",
      tenantId: "",
      contactPhone: "",
      immediateAction: "",
      safetyRisk: "medium",
      images: [],
      ...initialData,
    },
  });

  const selectedEmergencyType = form.watch("emergencyType");

  // Auto-set category based on emergency type
  useEffect(() => {
    const typeToCategory: Record<string, string> = {
      water_leak: "Plumbing",
      electrical_hazard: "Electrical",
      gas_leak: "Plumbing",
      security_breach: "Security",
      fire_hazard: "Emergency",
      structural_damage: "Emergency",
      hvac_failure: "HVAC",
      other: "Emergency",
    };

    if (selectedEmergencyType && typeToCategory[selectedEmergencyType]) {
      form.setValue("category", typeToCategory[selectedEmergencyType] as any);
    }
  }, [selectedEmergencyType, form]);

  const handleImageUpload = async (files: FileList) => {
    if (files.length === 0) return;

    setUploadingImages(true);
    const newImages: string[] = [];

    try {
      for (const file of Array.from(files)) {
        if (file.size > 5 * 1024 * 1024) {
          toast.error(`File ${file.name} is too large. Maximum size is 5MB.`);
          continue;
        }

        if (!file.type.startsWith("image/")) {
          toast.error(`File ${file.name} is not an image.`);
          continue;
        }

        // Create FormData for upload
        const formData = new FormData();
        formData.append("file", file);
        formData.append("upload_preset", "property_images");

        // Upload to R2
        const response = await fetch("/api/upload/image", {
          method: "POST",
          body: formData,
        });

        if (response.ok) {
          const result = await response.json();
          newImages.push(result.secure_url);
        } else {
          toast.error(`Failed to upload ${file.name}`);
        }
      }

      const updatedImages = [...uploadedImages, ...newImages];
      setUploadedImages(updatedImages);
      form.setValue("images", updatedImages);

      if (newImages.length > 0) {
        toast.success(`Uploaded ${newImages.length} image(s) successfully`);
      }
    } catch (error) {
      toast.error("Failed to upload images");
    } finally {
      setUploadingImages(false);
    }
  };

  const removeImage = (index: number) => {
    const updatedImages = uploadedImages.filter((_, i) => i !== index);
    setUploadedImages(updatedImages);
    form.setValue("images", updatedImages);
  };

  const handleFormSubmit = async (data: EmergencyRequestFormData) => {
    try {
      await onSubmit({
        ...data,
        images: uploadedImages,
      });
    } catch (error) {
      // Error handling is done in the parent component
    }
  };

  const selectedType = emergencyTypes.find(
    (type) => type.value === selectedEmergencyType
  );

  return (
    <div className="max-w-4xl mx-auto space-y-6">
      {/* Emergency Alert */}
      <Alert className="border-red-200 bg-red-50">
        <AlertTriangle className="h-4 w-4 text-red-600" />
        <AlertDescription className="text-red-800">
          <strong>Emergency Request:</strong> This form is for urgent
          maintenance issues that require immediate attention. For
          life-threatening emergencies, call 911 first.
        </AlertDescription>
      </Alert>

      <Form {...form}>
        <form
          onSubmit={form.handleSubmit(handleFormSubmit)}
          className="space-y-6"
        >
          {/* Emergency Type and Safety Risk */}
          <Card className="border-red-200">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-red-700">
                <Zap className="h-5 w-5" />
                Emergency Classification
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="emergencyType"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Emergency Type *</FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select emergency type" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {emergencyTypes.map((type) => (
                            <SelectItem key={type.value} value={type.value}>
                              <div className="flex items-center gap-2">
                                <span>{type.icon}</span>
                                <span>{type.label}</span>
                                <span
                                  className={`text-xs px-2 py-1 rounded ${
                                    type.severity === "critical"
                                      ? "bg-red-100 text-red-700"
                                      : type.severity === "high"
                                      ? "bg-orange-100 text-orange-700"
                                      : "bg-yellow-100 text-yellow-700"
                                  }`}
                                >
                                  {type.severity}
                                </span>
                              </div>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="safetyRisk"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Safety Risk Level *</FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select risk level" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {safetyRiskLevels.map((risk) => (
                            <SelectItem key={risk.value} value={risk.value}>
                              <div
                                className={`flex items-center gap-2 px-2 py-1 rounded ${risk.bg}`}
                              >
                                <span className={risk.color}>●</span>
                                <span className={risk.color}>{risk.label}</span>
                              </div>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              {selectedType && (
                <div
                  className={`p-3 rounded-lg ${
                    selectedType.severity === "critical"
                      ? "bg-red-50 border border-red-200"
                      : selectedType.severity === "high"
                      ? "bg-orange-50 border border-orange-200"
                      : "bg-yellow-50 border border-yellow-200"
                  }`}
                >
                  <div className="flex items-center gap-2">
                    <span className="text-lg">{selectedType.icon}</span>
                    <span className="font-medium">{selectedType.label}</span>
                    <span
                      className={`text-xs px-2 py-1 rounded ${
                        selectedType.severity === "critical"
                          ? "bg-red-100 text-red-700"
                          : selectedType.severity === "high"
                          ? "bg-orange-100 text-orange-700"
                          : "bg-yellow-100 text-yellow-700"
                      }`}
                    >
                      {selectedType.severity.toUpperCase()} PRIORITY
                    </span>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>

          {/* Request Details */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5" />
                Request Details
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <FormField
                control={form.control}
                name="title"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Emergency Title *</FormLabel>
                    <FormControl>
                      <Input
                        placeholder="Brief description of the emergency"
                        {...field}
                        className="border-red-200 focus:border-red-400"
                      />
                    </FormControl>
                    <FormDescription>
                      Provide a clear, concise title for the emergency
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Detailed Description *</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="Describe the emergency situation in detail..."
                        className="min-h-[120px] border-red-200 focus:border-red-400"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      Include what happened, when it started, and any immediate
                      dangers
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="immediateAction"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Immediate Action Taken</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="Describe any immediate actions you've taken..."
                        className="min-h-[80px]"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      What steps have you already taken to address the
                      emergency?
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>

          {/* Property and Contact Information */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Phone className="h-5 w-5" />
                Property & Contact Information
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 md:grid-cols-2">
                <FormField
                  control={form.control}
                  name="propertyId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Property *</FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select property" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {properties.map((property) => (
                            <SelectItem key={property.id} value={property.id}>
                              <div>
                                <div className="font-medium">
                                  {property.name}
                                </div>
                                <div className="text-sm text-muted-foreground">
                                  {property.address}
                                </div>
                              </div>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="tenantId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Tenant *</FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select tenant" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {tenants.map((tenant) => (
                            <SelectItem key={tenant.id} value={tenant.id}>
                              <div>
                                <div className="font-medium">{tenant.name}</div>
                                <div className="text-sm text-muted-foreground">
                                  {tenant.email} • {tenant.phone}
                                </div>
                              </div>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="contactPhone"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Emergency Contact Phone</FormLabel>
                    <FormControl>
                      <Input
                        placeholder="Alternative phone number for immediate contact"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      Provide an alternative phone number if different from
                      tenant's primary number
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>

          {/* Image Upload */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Upload className="h-5 w-5" />
                Emergency Photos
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="border-2 border-dashed border-red-200 rounded-lg p-6 text-center">
                <input
                  type="file"
                  multiple
                  accept="image/*"
                  onChange={(e) =>
                    e.target.files && handleImageUpload(e.target.files)
                  }
                  className="hidden"
                  id="emergency-images"
                  disabled={uploadingImages}
                />
                <label
                  htmlFor="emergency-images"
                  className="cursor-pointer flex flex-col items-center gap-2"
                >
                  <Upload className="h-8 w-8 text-red-400" />
                  <div className="text-sm">
                    <span className="font-medium text-red-600">
                      Click to upload
                    </span>{" "}
                    emergency photos
                  </div>
                  <div className="text-xs text-muted-foreground">
                    PNG, JPG up to 5MB each
                  </div>
                </label>
              </div>

              {uploadingImages && (
                <div className="text-center text-sm text-muted-foreground">
                  Uploading images...
                </div>
              )}

              {uploadedImages.length > 0 && (
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                  {uploadedImages.map((image, index) => (
                    <div key={index} className="relative group">
                      <img
                        src={image}
                        alt={`Emergency photo ${index + 1}`}
                        className="w-full h-24 object-cover rounded-lg border"
                      />
                      <button
                        type="button"
                        onClick={() => removeImage(index)}
                        className="absolute -top-2 -right-2 bg-red-500 text-white rounded-full p-1 opacity-0 group-hover:opacity-100 transition-opacity"
                      >
                        <X className="h-3 w-3" />
                      </button>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Card className="border-red-200 bg-red-50">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <Clock className="h-5 w-5 text-red-600" />
                  <div>
                    <div className="font-medium text-red-700">
                      Emergency Response
                    </div>
                    <div className="text-sm text-red-600">
                      This request will be prioritized for immediate attention
                    </div>
                  </div>
                </div>
                <Button
                  type="submit"
                  disabled={isLoading}
                  className="bg-red-600 hover:bg-red-700 text-white px-8"
                >
                  {isLoading ? (
                    <>
                      <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2" />
                      Submitting...
                    </>
                  ) : (
                    <>
                      <CheckCircle className="mr-2 h-4 w-4" />
                      Submit Emergency Request
                    </>
                  )}
                </Button>
              </div>
            </CardContent>
          </Card>
        </form>
      </Form>
    </div>
  );
}
