"use client";

import { useState, useEffect } from "react";
import { format } from "date-fns";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { toast } from "sonner";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { ImageUpload } from "@/components/ui/image-upload";
import { FormDateTimePicker } from "@/components/ui/date-time-picker";
import {
  Wrench,
  AlertTriangle,
  Building2,
  User,
  Image as ImageIcon,
} from "lucide-react";
import { MaintenancePriority } from "@/types";

// Form validation schema
const maintenanceRequestFormSchema = z.object({
  title: z.string().min(1, "Title is required").max(100, "Title too long"),
  description: z
    .string()
    .min(10, "Description must be at least 10 characters")
    .max(1000, "Description too long"),
  category: z.string().min(1, "Category is required"),
  priority: z.nativeEnum(MaintenancePriority),
  propertyId: z.string().min(1, "Property is required"),
  unitId: z.string().optional(),
  tenantId: z.string().min(1, "Tenant is required"),
  assignedTo: z.string().optional(),
  estimatedCost: z.number().min(0, "Cost cannot be negative").optional(),
  scheduledDate: z
    .string()
    .optional()
    .refine((date) => {
      if (!date) return true;
      return !isNaN(Date.parse(date));
    }, "Invalid date format"),
  images: z.array(z.string()).optional(),
});

type MaintenanceRequestFormData = z.infer<typeof maintenanceRequestFormSchema>;

interface MaintenanceRequestFormProps {
  onSubmit: (data: MaintenanceRequestFormData) => void;
  isLoading?: boolean;
  initialData?: Partial<MaintenanceRequestFormData>;
  isTenantView?: boolean; // New prop to indicate if this is being used by a tenant
  properties?: Array<{
    id: string;
    name: string;
    address: string;
    isMultiUnit?: boolean;
    units?: Array<{
      _id: string;
      unitNumber: string;
      unitType: string;
      status: string;
    }>;
  }>;
  tenants?: Array<{
    id: string;
    name: string;
    email: string;
    phone?: string;
    avatar?: string;
    unitNumber?: string;
    unitType?: string;
    leaseStatus?: string;
    propertyName?: string;
  }>;
  technicians?: Array<{
    id: string;
    name: string;
    email: string;
    specialties?: string[];
  }>;
}

const maintenanceCategories = [
  "Plumbing",
  "Electrical",
  "HVAC",
  "Appliances",
  "Flooring",
  "Painting",
  "Roofing",
  "Windows",
  "Doors",
  "Landscaping",
  "Cleaning",
  "Pest Control",
  "Security",
  "General Repair",
  "Emergency",
  "Other",
];

export function MaintenanceRequestForm({
  onSubmit,
  isLoading = false,
  initialData,
  isTenantView = false,
  properties = [],
  tenants = [],
  technicians = [],
}: MaintenanceRequestFormProps) {
  const [uploadedImages, setUploadedImages] = useState<
    { url: string; publicId: string }[]
  >((initialData?.images || []).map((url) => ({ url, publicId: "" })));

  const [availableUnits, setAvailableUnits] = useState<
    Array<{
      _id: string;
      unitNumber: string;
      unitType: string;
      status: string;
    }>
  >([]);

  const [filteredTenants, setFilteredTenants] = useState<
    Array<{
      id: string;
      name: string;
      email: string;
      phone?: string;
      avatar?: string;
      unitNumber?: string;
      unitType?: string;
      leaseStatus?: string;
      propertyName?: string;
    }>
  >([]);

  const [loadingTenants, setLoadingTenants] = useState(false);

  const form = useForm<MaintenanceRequestFormData>({
    resolver: zodResolver(maintenanceRequestFormSchema),
    defaultValues: {
      title: initialData?.title || "",
      description: initialData?.description || "",
      category: initialData?.category || "",
      priority: initialData?.priority || MaintenancePriority.MEDIUM,
      propertyId: initialData?.propertyId || "",
      unitId: initialData?.unitId || "",
      tenantId: initialData?.tenantId || "",
      assignedTo: initialData?.assignedTo || "",
      estimatedCost: initialData?.estimatedCost || undefined,
      scheduledDate: initialData?.scheduledDate || "",
      images: initialData?.images || [],
    },
  });

  const watchedPriority = form.watch("priority");
  const watchedCategory = form.watch("category");
  const watchedPropertyId = form.watch("propertyId");
  const watchedUnitId = form.watch("unitId");

  // Function to fetch tenants for a specific property
  const fetchPropertyTenants = async (propertyId: string, unitId?: string) => {
    try {
      setLoadingTenants(true);
      const url = new URL(
        `/api/properties/${propertyId}/tenants`,
        window.location.origin
      );
      if (unitId) {
        url.searchParams.set("unitId", unitId);
      }
      url.searchParams.set("status", "active");

      const response = await fetch(url.toString());

      if (response.ok) {
        const data = await response.json();
        // Handle both possible response structures
        const apiTenants = data.data?.tenants || data.tenants || [];

        // Map API response to expected format
        const mappedTenants = apiTenants
          .filter((tenant: any) => tenant && (tenant.id || tenant._id)) // Filter out invalid entries
          .map((tenant: any) => ({
            id: tenant.id || tenant._id,
            name:
              `${tenant.firstName || ""} ${tenant.lastName || ""}`.trim() ||
              "Unknown Tenant",
            email: tenant.email || "",
            phone: tenant.phone || "",
            avatar: tenant.avatar,
            unitNumber: tenant.unit?.unitNumber,
            unitType: tenant.unit?.type,
            leaseStatus: tenant.lease?.status,
            propertyName: undefined, // Not needed since we're filtering by property
          }));

        setFilteredTenants(mappedTenants);

        // Show success message if tenants found
        if (mappedTenants.length > 0) {
          toast.success(
            `Found ${mappedTenants.length} tenant(s) for the selected property`
          );
        } else {
          // If no property-specific tenants found, fall back to all tenants
          setFilteredTenants(tenants);
          toast.info(
            "No active tenants found for the selected property. Showing all tenants."
          );
        }
      } else {
        console.error("Failed to fetch property tenants:", response.status);
        // Fall back to showing all tenants
        setFilteredTenants(tenants);
        toast.error(
          "Failed to load tenants for the selected property. Showing all tenants."
        );
      }
    } catch (error) {
      console.error("Error fetching property tenants:", error);
      // Fall back to showing all tenants
      setFilteredTenants(tenants);
      toast.error("Error loading tenants. Showing all available tenants.");
    } finally {
      setLoadingTenants(false);
    }
  };

  // Initialize filtered tenants - always show all tenants initially
  useEffect(() => {
    // Always show all tenants when tenants list changes
    // Property-specific filtering will override this when a property is selected
    setFilteredTenants(tenants);
  }, [tenants]);

  // Update available units and filter tenants when property changes
  useEffect(() => {
    if (watchedPropertyId) {
      const selectedProperty = properties.find(
        (p) => p.id === watchedPropertyId
      );

      // Handle units for multi-unit properties
      if (selectedProperty?.isMultiUnit && selectedProperty.units) {
        setAvailableUnits(selectedProperty.units);
      } else {
        setAvailableUnits([]);
        form.setValue("unitId", ""); // Clear unit selection for single-unit properties
      }

      // Fetch tenants for the selected property
      fetchPropertyTenants(watchedPropertyId);

      // Clear tenant selection when property changes
      form.setValue("tenantId", "");
    } else {
      setAvailableUnits([]);
      setFilteredTenants(tenants); // Show all tenants when no property is selected
      form.setValue("unitId", "");
      form.setValue("tenantId", "");
    }
  }, [watchedPropertyId, properties, form]);

  // Update tenant filtering when unit changes
  useEffect(() => {
    if (watchedPropertyId && watchedUnitId) {
      // Fetch tenants for the specific unit
      fetchPropertyTenants(watchedPropertyId, watchedUnitId);
      form.setValue("tenantId", "");
    } else if (watchedPropertyId) {
      // Fetch all tenants for the property
      fetchPropertyTenants(watchedPropertyId);
    }
  }, [watchedUnitId, watchedPropertyId]);

  const getPriorityDescription = (priority: MaintenancePriority) => {
    switch (priority) {
      case MaintenancePriority.EMERGENCY:
        return "Immediate attention required - safety hazard or major system failure";
      case MaintenancePriority.HIGH:
        return "Urgent - affects habitability or causes significant inconvenience";
      case MaintenancePriority.MEDIUM:
        return "Important - should be addressed within a week";
      case MaintenancePriority.LOW:
        return "Non-urgent - can be scheduled for routine maintenance";
      default:
        return "";
    }
  };

  const handleFormSubmit = (data: MaintenanceRequestFormData) => {
    try {
      // Enhanced validation with better error messages
      if (!data.title?.trim()) {
        toast.error("Please provide a title for the maintenance request");
        form.setFocus("title");
        return;
      }

      if (!data.description?.trim()) {
        toast.error("Please describe the maintenance issue in detail");
        form.setFocus("description");
        return;
      }

      if (!data.propertyId) {
        toast.error("Please select a property for this maintenance request");
        form.setFocus("propertyId");
        return;
      }

      if (!data.tenantId) {
        toast.error("Please select a tenant for this maintenance request");
        form.setFocus("tenantId");
        return;
      }

      if (!data.category) {
        toast.error("Please select a maintenance category");
        form.setFocus("category");
        return;
      }

      // Check if property has units but no unit is selected
      const selectedProperty = properties.find((p) => p.id === data.propertyId);
      if (
        selectedProperty?.isMultiUnit &&
        selectedProperty.units?.length > 0 &&
        !data.unitId
      ) {
        toast.error("Please select a unit for this multi-unit property");
        form.setFocus("unitId");
        return;
      }

      const formattedData = {
        ...data,
        images: uploadedImages.map((img) => img.url),
        // Convert scheduledDate to ISO string if provided
        scheduledDate: data.scheduledDate
          ? new Date(data.scheduledDate).toISOString()
          : undefined,
      };
      onSubmit(formattedData);
    } catch (error) {
      console.error("Form submission error:", error);
      toast.error(
        "Failed to submit form. Please check all fields and try again."
      );
    }
  };

  return (
    <div className="w-full space-y-8">
      <Form {...form}>
        <form
          onSubmit={form.handleSubmit(handleFormSubmit)}
          className="space-y-8"
        >
          {/* Request Details */}
          <Card className="border-0 shadow-lg bg-white/70 dark:bg-gray-900/70 backdrop-blur-sm">
            <CardHeader className="pb-6">
              <CardTitle className="flex items-center gap-3 text-xl">
                <div className="p-2 rounded-lg bg-gradient-to-br from-blue-500 to-indigo-600 text-white">
                  <Wrench className="h-5 w-5" />
                </div>
                Request Details
              </CardTitle>
              <CardDescription className="text-base">
                Describe the maintenance issue that needs attention
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <FormField
                control={form.control}
                name="title"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                      Title
                    </FormLabel>
                    <FormControl>
                      <Input
                        placeholder="Brief description of the issue"
                        className="h-11 border-gray-200 dark:border-gray-700 focus:border-blue-500 focus:ring-blue-500/20 transition-all duration-200"
                        {...field}
                      />
                    </FormControl>
                    <FormDescription className="text-sm text-gray-500">
                      Provide a clear, concise title for the maintenance request
                    </FormDescription>
                    <FormMessage className="text-red-500 text-sm" />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <FormField
                  control={form.control}
                  name="category"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                        Category
                      </FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        value={field.value || undefined}
                      >
                        <FormControl>
                          <SelectTrigger className="h-11 border-gray-200 dark:border-gray-700 focus:border-blue-500 focus:ring-blue-500/20 transition-all duration-200">
                            <SelectValue placeholder="Select category" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {maintenanceCategories.map((category) => (
                            <SelectItem key={category} value={category}>
                              {category}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage className="text-red-500 text-sm" />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="priority"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                        Priority
                      </FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        value={field.value || undefined}
                      >
                        <FormControl>
                          <SelectTrigger className="h-11 border-gray-200 dark:border-gray-700 focus:border-blue-500 focus:ring-blue-500/20 transition-all duration-200">
                            <SelectValue placeholder="Select priority" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value={MaintenancePriority.EMERGENCY}>
                            <div className="flex items-center">
                              <AlertTriangle className="h-4 w-4 mr-2 text-red-500" />
                              Emergency
                            </div>
                          </SelectItem>
                          <SelectItem value={MaintenancePriority.HIGH}>
                            <div className="flex items-center">
                              <div className="h-2 w-2 rounded-full bg-orange-500 mr-2" />
                              High
                            </div>
                          </SelectItem>
                          <SelectItem value={MaintenancePriority.MEDIUM}>
                            <div className="flex items-center">
                              <div className="h-2 w-2 rounded-full bg-yellow-500 mr-2" />
                              Medium
                            </div>
                          </SelectItem>
                          <SelectItem value={MaintenancePriority.LOW}>
                            <div className="flex items-center">
                              <div className="h-2 w-2 rounded-full bg-green-500 mr-2" />
                              Low
                            </div>
                          </SelectItem>
                        </SelectContent>
                      </Select>
                      <FormDescription className="text-sm text-gray-500">
                        {getPriorityDescription(watchedPriority)}
                      </FormDescription>
                      <FormMessage className="text-red-500 text-sm" />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                      Description
                    </FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="Provide detailed information about the maintenance issue..."
                        className="resize-none border-gray-200 dark:border-gray-700 focus:border-blue-500 focus:ring-blue-500/20 transition-all duration-200"
                        rows={4}
                        {...field}
                      />
                    </FormControl>
                    <FormDescription className="text-sm text-gray-500">
                      Include as much detail as possible to help technicians
                      understand the issue
                    </FormDescription>
                    <FormMessage className="text-red-500 text-sm" />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>

          {/* Property and Tenant Information - Hidden for tenants */}
          {!isTenantView && (
            <Card className="border-0 shadow-lg bg-white/70 dark:bg-gray-900/70 backdrop-blur-sm">
              <CardHeader className="pb-6">
                <CardTitle className="flex items-center gap-3 text-xl">
                  <div className="p-2 rounded-lg bg-gradient-to-br from-green-500 to-emerald-600 text-white">
                    <Building2 className="h-5 w-5" />
                  </div>
                  Property & Tenant
                </CardTitle>
                <CardDescription className="text-base">
                  Select the property and tenant for this maintenance request
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div
                  className={`grid gap-6 ${
                    availableUnits.length > 0
                      ? "grid-cols-1 md:grid-cols-3"
                      : "grid-cols-1 md:grid-cols-2"
                  }`}
                >
                  <FormField
                    control={form.control}
                    name="propertyId"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                          Property
                        </FormLabel>
                        <Select
                          onValueChange={field.onChange}
                          value={field.value || undefined}
                        >
                          <FormControl>
                            <SelectTrigger className="h-11 border-gray-200 dark:border-gray-700 focus:border-blue-500 focus:ring-blue-500/20 transition-all duration-200">
                              <SelectValue placeholder="Select property" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {properties.map((property) => (
                              <SelectItem key={property.id} value={property.id}>
                                <div>
                                  <div className="font-medium">
                                    {property.name}
                                  </div>
                                  <div className="text-sm text-muted-foreground">
                                    {property.address}
                                  </div>
                                  {property.isMultiUnit && (
                                    <div className="text-xs text-blue-600 font-medium">
                                      Multi-unit property
                                    </div>
                                  )}
                                </div>
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage className="text-red-500 text-sm" />
                      </FormItem>
                    )}
                  />

                  {/* Unit Selection - Only show for multi-unit properties */}
                  {availableUnits.length > 0 && (
                    <FormField
                      control={form.control}
                      name="unitId"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Unit</FormLabel>
                          <Select
                            onValueChange={field.onChange}
                            value={field.value || undefined}
                          >
                            <FormControl>
                              <SelectTrigger>
                                <SelectValue placeholder="Select unit" />
                              </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                              {availableUnits.map((unit) => (
                                <SelectItem key={unit._id} value={unit._id}>
                                  <div>
                                    <div className="font-medium">
                                      Unit {unit.unitNumber}
                                    </div>
                                    <div className="text-sm text-muted-foreground">
                                      {unit.unitType} • {unit.status}
                                    </div>
                                  </div>
                                </SelectItem>
                              ))}
                            </SelectContent>
                          </Select>
                          <FormDescription>
                            Select the specific unit for this maintenance
                            request
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />
                  )}

                  <FormField
                    control={form.control}
                    name="tenantId"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className="text-sm font-semibold text-gray-700 dark:text-gray-300">
                          Tenant
                        </FormLabel>
                        <Select
                          onValueChange={field.onChange}
                          value={field.value || undefined}
                        >
                          <FormControl>
                            <SelectTrigger className="h-11 border-gray-200 dark:border-gray-700 focus:border-blue-500 focus:ring-blue-500/20 transition-all duration-200">
                              <SelectValue placeholder="Select tenant" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {loadingTenants ? (
                              <div className="p-2 text-sm text-muted-foreground flex items-center gap-2">
                                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-blue-600" />
                                Loading tenants...
                              </div>
                            ) : filteredTenants.length > 0 ? (
                              filteredTenants.map((tenant) => (
                                <SelectItem key={tenant.id} value={tenant.id}>
                                  <div>
                                    <div className="font-medium">
                                      {tenant.name}
                                    </div>
                                    <div className="text-sm text-muted-foreground">
                                      {tenant.email}
                                    </div>
                                    {tenant.unitNumber && (
                                      <div className="text-xs text-blue-600 font-medium">
                                        Unit {tenant.unitNumber}
                                        {tenant.unitType &&
                                          ` (${tenant.unitType})`}
                                      </div>
                                    )}
                                    {tenant.leaseStatus && (
                                      <div className="text-xs text-green-600 font-medium capitalize">
                                        {tenant.leaseStatus} Lease
                                      </div>
                                    )}
                                  </div>
                                </SelectItem>
                              ))
                            ) : (
                              <div className="p-2 text-sm text-muted-foreground">
                                {watchedPropertyId
                                  ? "No tenants found for selected property"
                                  : tenants.length > 0
                                  ? "No tenants available"
                                  : "Loading tenants..."}
                              </div>
                            )}
                          </SelectContent>
                        </Select>
                        <FormMessage className="text-red-500 text-sm" />
                      </FormItem>
                    )}
                  />
                </div>
              </CardContent>
            </Card>
          )}

          {/* Assignment and Scheduling - Hidden for tenants */}
          {!isTenantView && (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <User className="h-5 w-5" />
                  Assignment & Scheduling
                </CardTitle>
                <CardDescription>
                  Assign technician and schedule the work (optional)
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <FormField
                    control={form.control}
                    name="assignedTo"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Assign to Technician (Optional)</FormLabel>
                        <Select
                          onValueChange={(value) =>
                            field.onChange(value === "UNASSIGNED" ? "" : value)
                          }
                          value={field.value || "UNASSIGNED"}
                        >
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue placeholder="Select technician" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            <SelectItem value="UNASSIGNED">
                              Unassigned
                            </SelectItem>
                            {technicians.map((tech) => (
                              <SelectItem key={tech.id} value={tech.id}>
                                <div>
                                  <div className="font-medium">{tech.name}</div>
                                  <div className="text-sm text-muted-foreground">
                                    {tech.email}
                                  </div>
                                  {tech.specialties && (
                                    <div className="text-xs text-muted-foreground">
                                      {tech.specialties.join(", ")}
                                    </div>
                                  )}
                                </div>
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="estimatedCost"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Estimated Cost (Optional)</FormLabel>
                        <FormControl>
                          <Input
                            type="number"
                            step="0.01"
                            placeholder="0.00"
                            {...field}
                            onChange={(e) =>
                              field.onChange(
                                parseFloat(e.target.value) || undefined
                              )
                            }
                          />
                        </FormControl>
                        <FormDescription>Estimated cost in USD</FormDescription>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="scheduledDate"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Scheduled Date (Optional)</FormLabel>
                        <FormControl>
                          <FormDateTimePicker
                            value={
                              field.value ? new Date(field.value) : undefined
                            }
                            onChange={(date) => {
                              if (date) {
                                const formattedValue = format(
                                  date,
                                  "yyyy-MM-dd'T'HH:mm"
                                );
                                field.onChange(formattedValue);
                              } else {
                                field.onChange("");
                              }
                            }}
                            placeholder="Select date & time"
                          />
                        </FormControl>
                        <FormDescription>
                          When to perform the work
                        </FormDescription>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </CardContent>
            </Card>
          )}

          {/* Image Upload */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <ImageIcon className="h-5 w-5" />
                Photos
              </CardTitle>
              <CardDescription>
                Upload photos to help illustrate the maintenance issue
              </CardDescription>
            </CardHeader>
            <CardContent>
              <ImageUpload
                onImagesUploaded={(newImages) => {
                  const updatedImages = [...uploadedImages, ...newImages];
                  setUploadedImages(updatedImages);
                  form.setValue(
                    "images",
                    updatedImages.map((img) => img.url)
                  );
                }}
                onImagesRemoved={(removedImages) => {
                  const updatedImages = uploadedImages.filter(
                    (img) =>
                      !removedImages.some(
                        (removed) => removed.publicId === img.publicId
                      )
                  );
                  setUploadedImages(updatedImages);
                  form.setValue(
                    "images",
                    updatedImages.map((img) => img.url)
                  );
                }}
                existingImages={uploadedImages}
                maxFiles={10}
                folder="PropertyPro/maintenance"
                quality="auto"
              />
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Card className="border-0 shadow-lg bg-white/70 dark:bg-gray-900/70 backdrop-blur-sm">
            <CardContent className="pt-6">
              <div className="flex justify-end space-x-4">
                <Button
                  type="button"
                  variant="outline"
                  className="h-11 px-6 border-gray-300 hover:bg-gray-50 hover:border-gray-400 transition-all duration-200"
                  onClick={(e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    window.history.back();
                  }}
                >
                  Cancel
                </Button>
                <Button
                  type="submit"
                  disabled={isLoading}
                  className="h-11 px-8 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white font-semibold shadow-lg hover:shadow-xl transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {isLoading
                    ? "Submitting..."
                    : initialData
                    ? "Update Request"
                    : "Submit Request"}
                </Button>
              </div>
            </CardContent>
          </Card>
        </form>
      </Form>
    </div>
  );
}
