"use client";

import { useState } from "react";
import { format } from "date-fns";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { toast } from "sonner";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  CreditCard,
  DollarSign,
  Calendar,
  User,
  Building2,
  FileText,
  Banknote,
} from "lucide-react";
import { PaymentType, PaymentMethod } from "@/types";
import { paymentCreateSchema } from "@/lib/validations";
import { FormDatePicker } from "@/components/ui/date-picker";

// Form validation schema with string date for form handling
const paymentFormSchema = z.object({
  tenantId: z.string().min(1, "Tenant is required"),
  propertyId: z.string().min(1, "Property is required"),
  leaseId: z.string().optional(),
  amount: z
    .number()
    .min(0.01, "Amount must be at least $0.01")
    .max(100000, "Amount too high"),
  type: z.nativeEnum(PaymentType),
  paymentMethod: z.nativeEnum(PaymentMethod).optional(),
  dueDate: z.string().min(1, "Due date is required"),
  description: z.string().max(500, "Description too long").optional(),
  notes: z.string().max(1000, "Notes too long").optional(),
});

type PaymentFormData = z.infer<typeof paymentFormSchema>;

interface PaymentFormProps {
  onSubmit: (data: PaymentFormData) => void;
  onCancel?: () => void;
  isLoading?: boolean;
  initialData?: Partial<PaymentFormData>;
  tenants?: Array<{ id: string; name: string; email: string }>;
  properties?: Array<{ id: string; name: string; address: string }>;
  leases?: Array<{ id: string; propertyName: string; tenantName: string }>;
}

export function PaymentForm({
  onSubmit,
  onCancel,
  isLoading = false,
  initialData,
  tenants = [],
  properties = [],
  leases = [],
}: PaymentFormProps) {
  const [selectedTenant, setSelectedTenant] = useState<string>(
    initialData?.tenantId || ""
  );
  const [selectedProperty, setSelectedProperty] = useState<string>(
    initialData?.propertyId || ""
  );

  const form = useForm<PaymentFormData>({
    resolver: zodResolver(paymentFormSchema),
    defaultValues: {
      tenantId: initialData?.tenantId || "",
      propertyId: initialData?.propertyId || "",
      leaseId: initialData?.leaseId || "",
      amount: initialData?.amount || 0,
      type: initialData?.type || PaymentType.RENT,
      paymentMethod: initialData?.paymentMethod || undefined,
      dueDate: initialData?.dueDate || "",
      description: initialData?.description || "",
      notes: initialData?.notes || "",
    },
  });

  const watchedType = form.watch("type");

  // Filter leases based on selected tenant and property
  const filteredLeases = leases.filter((lease) => {
    if (!selectedTenant && !selectedProperty) return true;
    // This would need to be implemented based on actual lease data structure
    return true;
  });

  const getPaymentTypeDescription = (type: PaymentType) => {
    switch (type) {
      case PaymentType.RENT:
        return "Monthly rent payment";
      case PaymentType.SECURITY_DEPOSIT:
        return "Security deposit for lease";
      case PaymentType.LATE_FEE:
        return "Late payment fee";
      case PaymentType.INVOICE:
        return "Invoice payment";
      case PaymentType.PET_DEPOSIT:
        return "Pet deposit fee";
      case PaymentType.UTILITY:
        return "Utility bill payment";
      case PaymentType.MAINTENANCE:
        return "Maintenance or repair cost";
      case PaymentType.OTHER:
        return "Other payment type";
      default:
        return "";
    }
  };

  const getPaymentMethodLabel = (method: PaymentMethod) => {
    switch (method) {
      case PaymentMethod.CREDIT_CARD:
        return "Credit Card";
      case PaymentMethod.DEBIT_CARD:
        return "Debit Card";
      case PaymentMethod.BANK_TRANSFER:
        return "Bank Transfer";
      case PaymentMethod.ACH:
        return "ACH Transfer";
      case PaymentMethod.CHECK:
        return "Check";
      case PaymentMethod.CASH:
        return "Cash";
      case PaymentMethod.MONEY_ORDER:
        return "Money Order";
      case PaymentMethod.OTHER:
        return "Other";
      default:
        return method;
    }
  };

  return (
    <div className="max-w-2xl mx-auto space-y-6">
      {/* Header */}
      <div className="text-center">
        <h1 className="text-3xl font-bold">
          {initialData ? "Edit Payment" : "Create Payment"}
        </h1>
        <p className="text-muted-foreground mt-2">
          {initialData
            ? "Update payment details"
            : "Create a new payment record"}
        </p>
      </div>

      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
          {/* Payment Details */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <CreditCard className="h-5 w-5" />
                Payment Details
              </CardTitle>
              <CardDescription>
                Basic payment information and amount
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="type"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Payment Type</FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select payment type" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value={PaymentType.RENT}>Rent</SelectItem>
                          <SelectItem value={PaymentType.SECURITY_DEPOSIT}>
                            Security Deposit
                          </SelectItem>
                          <SelectItem value={PaymentType.INVOICE}>
                            Invoice
                          </SelectItem>
                          <SelectItem value={PaymentType.LATE_FEE}>
                            Late Fee
                          </SelectItem>
                          <SelectItem value={PaymentType.PET_DEPOSIT}>
                            Pet Deposit
                          </SelectItem>
                          <SelectItem value={PaymentType.UTILITY}>
                            Utility
                          </SelectItem>
                          <SelectItem value={PaymentType.MAINTENANCE}>
                            Maintenance
                          </SelectItem>
                          <SelectItem value={PaymentType.OTHER}>
                            Other
                          </SelectItem>
                        </SelectContent>
                      </Select>
                      <FormDescription>
                        {getPaymentTypeDescription(watchedType)}
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="amount"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Amount</FormLabel>
                      <FormControl>
                        <div className="relative">
                          <DollarSign className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                          <Input
                            type="number"
                            step="0.01"
                            placeholder="0.00"
                            className="pl-10"
                            {...field}
                            onChange={(e) =>
                              field.onChange(parseFloat(e.target.value) || 0)
                            }
                          />
                        </div>
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="paymentMethod"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Payment Method (Optional)</FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select payment method" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value={PaymentMethod.CREDIT_CARD}>
                            {getPaymentMethodLabel(PaymentMethod.CREDIT_CARD)}
                          </SelectItem>
                          <SelectItem value={PaymentMethod.DEBIT_CARD}>
                            {getPaymentMethodLabel(PaymentMethod.DEBIT_CARD)}
                          </SelectItem>
                          <SelectItem value={PaymentMethod.BANK_TRANSFER}>
                            {getPaymentMethodLabel(PaymentMethod.BANK_TRANSFER)}
                          </SelectItem>
                          <SelectItem value={PaymentMethod.ACH}>
                            {getPaymentMethodLabel(PaymentMethod.ACH)}
                          </SelectItem>
                          <SelectItem value={PaymentMethod.CHECK}>
                            {getPaymentMethodLabel(PaymentMethod.CHECK)}
                          </SelectItem>
                          <SelectItem value={PaymentMethod.CASH}>
                            {getPaymentMethodLabel(PaymentMethod.CASH)}
                          </SelectItem>
                          <SelectItem value={PaymentMethod.MONEY_ORDER}>
                            {getPaymentMethodLabel(PaymentMethod.MONEY_ORDER)}
                          </SelectItem>
                          <SelectItem value={PaymentMethod.OTHER}>
                            {getPaymentMethodLabel(PaymentMethod.OTHER)}
                          </SelectItem>
                        </SelectContent>
                      </Select>
                      <FormDescription>
                        How this payment will be processed
                      </FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="dueDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Due Date</FormLabel>
                      <FormControl>
                        <FormDatePicker
                          value={
                            field.value
                              ? new Date(`${field.value}T00:00:00`)
                              : undefined
                          }
                          onChange={(date) => {
                            if (date) {
                              const localDate = new Date(
                                date.getFullYear(),
                                date.getMonth(),
                                date.getDate()
                              );
                              field.onChange(format(localDate, "yyyy-MM-dd"));
                            } else {
                              field.onChange("");
                            }
                          }}
                          placeholder="Select due date"
                          disabled={(date) =>
                            date <
                            new Date(
                              new Date().setDate(new Date().getDate() - 1)
                            )
                          }
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Description (Optional)</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="Additional payment details..."
                        className="resize-none"
                        rows={3}
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      Optional description for this payment
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Internal Notes (Optional)</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="Internal notes for staff use..."
                        className="resize-none"
                        rows={3}
                        {...field}
                      />
                    </FormControl>
                    <FormDescription>
                      Private notes visible only to staff members
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>

          {/* Tenant and Property Selection */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <User className="h-5 w-5" />
                Tenant & Property
              </CardTitle>
              <CardDescription>
                Select the tenant and property for this payment
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="tenantId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Tenant</FormLabel>
                      <Select
                        onValueChange={(value) => {
                          field.onChange(value);
                          setSelectedTenant(value);
                        }}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select tenant" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {tenants.map((tenant) => (
                            <SelectItem key={tenant.id} value={tenant.id}>
                              <div>
                                <div className="font-medium">{tenant.name}</div>
                                <div className="text-sm text-muted-foreground">
                                  {tenant.email}
                                </div>
                              </div>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="propertyId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Property</FormLabel>
                      <Select
                        onValueChange={(value) => {
                          field.onChange(value);
                          setSelectedProperty(value);
                        }}
                        defaultValue={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select property" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {properties.map((property) => (
                            <SelectItem key={property.id} value={property.id}>
                              <div>
                                <div className="font-medium">
                                  {property.name}
                                </div>
                                <div className="text-sm text-muted-foreground">
                                  {property.address}
                                </div>
                              </div>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="leaseId"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Associated Lease (Optional)</FormLabel>
                    <Select
                      onValueChange={field.onChange}
                      defaultValue={field.value}
                    >
                      <FormControl>
                        <SelectTrigger>
                          <SelectValue placeholder="Select lease (optional)" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {filteredLeases.map((lease) => (
                          <SelectItem key={lease.id} value={lease.id}>
                            <div>
                              <div className="font-medium">
                                {lease.propertyName}
                              </div>
                              <div className="text-sm text-muted-foreground">
                                {lease.tenantName}
                              </div>
                            </div>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormDescription>
                      Link this payment to a specific lease agreement
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>

          {/* Submit Button */}
          <div className="flex justify-end space-x-4">
            <Button type="button" variant="outline" onClick={onCancel}>
              Cancel
            </Button>
            <Button type="submit" disabled={isLoading}>
              {isLoading
                ? "Saving..."
                : initialData
                ? "Update Payment"
                : "Create Payment"}
            </Button>
          </div>
        </form>
      </Form>
    </div>
  );
}
