/**
 * PropertyPro - Simplified Lease Creation Form
 * Streamlined lease creation focusing on core fields only
 */

"use client";

import React, { useState, useEffect } from "react";
import { format } from "date-fns";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Separator } from "@/components/ui/separator";
import {
  Home,
  Calendar,
  DollarSign,
  AlertTriangle,
  CheckCircle,
  Loader2,
} from "lucide-react";
import { toast } from "sonner";
import { PropertyStatus } from "@/types";
import { FormDatePicker } from "@/components/ui/date-picker";
import {
  LeaseResponse,
  leaseService,
} from "@/lib/services/lease.service";

interface SimplifiedLeaseData {
  // Core Information
  propertyId: string;
  unitId: string;
  tenantId: string;

  // Dates
  startDate: string;
  endDate: string;

  // Financial Terms
  rentAmount: number;
  securityDeposit: number;
  rentDueDay: number;

  // Late Fee Configuration
  lateFeeAmount: number;
  lateFeeGracePeriodDays: number;
  lateFeeType: "fixed" | "percentage";

  // Auto-generation Settings
  autoGenerateInvoices: boolean;
  autoEmailInvoices: boolean;
}

interface Property {
  _id: string;
  name: string;
  address: {
    street: string;
    city: string;
    state: string;
    zipCode: string;
    country: string;
  };
  units: Array<{
    _id: string;
    unitNumber: string;
    unitType: string;
    bedrooms: number;
    bathrooms: number;
    squareFootage: number;
    rentAmount: number;
    securityDeposit: number;
    status: string;
  }>;
}

interface Tenant {
  _id: string;
  firstName: string;
  lastName: string;
  email: string;
}

interface SimplifiedLeaseCreationProps {
  mode?: "create" | "edit";
  leaseId?: string;
  initialLease?: LeaseResponse;
  onSuccess?: (leaseId?: string) => void;
}

const createInitialLeaseState = (): SimplifiedLeaseData => ({
  propertyId: "",
  unitId: "",
  tenantId: "",
  startDate: "",
  endDate: "",
  rentAmount: 0,
  securityDeposit: 0,
  rentDueDay: 1,
  lateFeeAmount: 50,
  lateFeeGracePeriodDays: 5,
  lateFeeType: "fixed",
  autoGenerateInvoices: true,
  autoEmailInvoices: false,
});

export default function SimplifiedLeaseCreation({
  mode = "create",
  leaseId,
  initialLease,
  onSuccess,
}: SimplifiedLeaseCreationProps) {
  const [leaseData, setLeaseData] = useState<SimplifiedLeaseData>(
    createInitialLeaseState
  );
  const [originalLeaseData, setOriginalLeaseData] = useState<SimplifiedLeaseData>(
    createInitialLeaseState
  );

  const [properties, setProperties] = useState<Property[]>([]);
  const [tenants, setTenants] = useState<Tenant[]>([]);
  const [loadingProperties, setLoadingProperties] = useState(false);
  const [submitting, setSubmitting] = useState(false);
  const [initializingLease, setInitializingLease] = useState(
    mode === "edit"
  );
  const [leaseError, setLeaseError] = useState<string | null>(null);

  const isEditMode = mode === "edit";
  const submitLabel = isEditMode ? "Save Changes" : "Create Lease";
  const submitLoadingLabel = isEditMode
    ? "Saving Changes..."
    : "Creating Lease...";
  const resetLabel = isEditMode ? "Reset Changes" : "Reset Form";

  // Fetch properties and tenants on component mount
  useEffect(() => {
    fetchProperties();
    fetchTenants();
  }, []);

  const fetchProperties = async () => {
    try {
      setLoadingProperties(true);
      const response = await fetch("/api/properties?limit=100");
      const data = await response.json();

      if (response.ok && data.success) {
        // The API returns properties directly in data.data
        const properties = Array.isArray(data.data) ? data.data : [];

        // Ensure all properties have a units array (even if empty)
        const normalizedProperties = properties.map((property: any) => ({
          ...property,
          units: Array.isArray(property.units) ? property.units : [],
        }));

        setProperties(normalizedProperties);
      } else {
        toast.error(data.error || "Failed to load properties");
        setProperties([]);
      }
    } catch (error) {
      toast.error("Failed to load properties");
      setProperties([]);
    } finally {
      setLoadingProperties(false);
    }
  };

  const fetchTenants = async () => {
    try {
      const response = await fetch("/api/users?role=tenant&limit=100");
      const data = await response.json();
      if (data.success) {
        setTenants(data.data.users || []);
      }
    } catch (error) {
      toast.error("Failed to load tenants");
    }
  };

  const mapLeaseToFormData = (lease: LeaseResponse): SimplifiedLeaseData => {
    const propertyId =
      typeof lease.propertyId === "string"
        ? lease.propertyId
        : lease.propertyId?._id || "";
    const tenantId =
      typeof lease.tenantId === "string"
        ? lease.tenantId
        : lease.tenantId?._id || "";
    const unitId =
      typeof lease.unitId === "string"
        ? lease.unitId
        : lease.unit?._id || "";

    const paymentConfig = lease.terms?.paymentConfig;
    const lateFeeConfig = paymentConfig?.lateFeeConfig;

    return {
      propertyId: propertyId || "",
      unitId: unitId || "",
      tenantId: tenantId || "",
      startDate: lease.startDate ? lease.startDate.slice(0, 10) : "",
      endDate: lease.endDate ? lease.endDate.slice(0, 10) : "",
      rentAmount: lease.terms?.rentAmount ?? 0,
      securityDeposit: lease.terms?.securityDeposit ?? 0,
      rentDueDay: paymentConfig?.rentDueDay ?? 1,
      lateFeeAmount:
        lateFeeConfig?.feeAmount ?? lease.terms?.lateFee ?? 0,
      lateFeeGracePeriodDays: lateFeeConfig?.gracePeriodDays ?? 0,
      lateFeeType:
        lateFeeConfig?.feeType === "percentage" ? "percentage" : "fixed",
      autoGenerateInvoices: paymentConfig?.autoGenerateInvoices ?? true,
      autoEmailInvoices: paymentConfig?.autoEmailInvoices ?? false,
    };
  };

  const hydrateLeaseData = (lease: LeaseResponse) => {
    const mapped = mapLeaseToFormData(lease);
    setLeaseData(mapped);
    setOriginalLeaseData({ ...mapped });
  };

  const loadLeaseDetails = async (id: string) => {
    try {
      setInitializingLease(true);
      const lease = await leaseService.getLeaseById(id);
      hydrateLeaseData(lease);
      setLeaseError(null);
    } catch (error) {
      console.error("Failed to load lease", error);
      const message =
        error instanceof Error ? error.message : "Failed to load lease";
      setLeaseError(message);
      toast.error("Failed to load lease details");
    } finally {
      setInitializingLease(false);
    }
  };

  useEffect(() => {
    if (mode !== "edit") {
      setOriginalLeaseData(createInitialLeaseState());
      return;
    }

    if (initialLease) {
      hydrateLeaseData(initialLease);
      setInitializingLease(false);
      setLeaseError(null);
      return;
    }

    if (leaseId) {
      void loadLeaseDetails(leaseId);
    } else {
      setInitializingLease(false);
    }
  }, [mode, leaseId, initialLease]);

  const handleInputChange = (field: keyof SimplifiedLeaseData, value: any) => {
    setLeaseData((prev) => ({
      ...prev,
      [field]: value,
    }));
  };

  const getSelectedProperty = () => {
    return properties.find((p) => p._id === leaseData.propertyId);
  };

  const getAvailableUnits = () => {
    const property = getSelectedProperty();

    if (!property || !Array.isArray(property.units)) {
      return [];
    }

    const selectedUnitId = leaseData.unitId;

    const availableUnits = property.units.filter((unit) => {
      if (!unit || typeof unit !== "object") {
        return false;
      }

      if (!unit.unitNumber) {
        return false;
      }

      if (mode === "edit" && unit._id === selectedUnitId) {
        return true;
      }

      const status =
        typeof unit.status === "string" ? unit.status.toLowerCase() : "";
      return status === PropertyStatus.AVAILABLE;
    });

    if (
      mode === "edit" &&
      selectedUnitId &&
      !availableUnits.some((unit) => unit._id === selectedUnitId)
    ) {
      const selectedUnit = property.units.find(
        (unit) => unit?._id === selectedUnitId
      );
      if (selectedUnit) {
        return [selectedUnit, ...availableUnits];
      }
    }

    return availableUnits;
  };

  const getSelectedUnit = () => {
    const availableUnits = getAvailableUnits();
    return availableUnits.find((unit) => unit?._id === leaseData?.unitId);
  };

  const handleUnitChange = (unitId: string) => {
    handleInputChange("unitId", unitId);

    // Auto-fill rent amount and security deposit when unit is selected
    const selectedUnit = getAvailableUnits().find(
      (unit) => unit._id === unitId
    );
    if (selectedUnit) {
      if (selectedUnit.rentAmount) {
        handleInputChange("rentAmount", selectedUnit.rentAmount);
      }
      if (selectedUnit.securityDeposit) {
        handleInputChange("securityDeposit", selectedUnit.securityDeposit);
      }
    }
  };

  const validateForm = (): string[] => {
    const errors: string[] = [];

    if (!leaseData.propertyId) errors.push("Property is required");
    if (!leaseData.unitId) errors.push("Unit is required");
    if (!leaseData.tenantId) errors.push("Tenant is required");
    if (!leaseData.startDate) errors.push("Start date is required");
    if (!leaseData.endDate) errors.push("End date is required");
    if (leaseData.rentAmount <= 0)
      errors.push("Rent amount must be greater than 0");
    if (leaseData.securityDeposit < 0)
      errors.push("Security deposit cannot be negative");
    if (leaseData.lateFeeAmount < 0)
      errors.push("Late fee amount cannot be negative");
    if (leaseData.lateFeeGracePeriodDays < 0)
      errors.push("Grace period cannot be negative");

    // Date validation
    if (leaseData.startDate && leaseData.endDate) {
      const start = new Date(leaseData.startDate);
      const end = new Date(leaseData.endDate);
      if (end <= start) {
        errors.push("End date must be after start date");
      }
    }

    return errors;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    const errors = validateForm();
    if (errors.length > 0) {
      toast.error(`Please fix the following errors: ${errors.join(", ")}`);
      return;
    }

    if (mode === "edit" && !(leaseId || initialLease?._id)) {
      toast.error("Missing lease identifier for update");
      return;
    }

    setSubmitting(true);

    const basePayload = {
      propertyId: leaseData.propertyId,
      unitId: leaseData.unitId,
      tenantId: leaseData.tenantId,
      startDate: leaseData.startDate,
      endDate: leaseData.endDate,
      terms: {
        rentAmount: leaseData.rentAmount,
        securityDeposit: leaseData.securityDeposit,
        lateFee: leaseData.lateFeeAmount,
        utilities: [],
        restrictions: [],
        paymentConfig: {
          rentDueDay: leaseData.rentDueDay,
          lateFeeConfig: {
            enabled: leaseData.lateFeeAmount > 0,
            gracePeriodDays: leaseData.lateFeeGracePeriodDays,
            feeType: leaseData.lateFeeType,
            feeAmount: leaseData.lateFeeAmount,
            compoundDaily: false,
            notificationDays: [3, 7, 14],
          },
          autoGenerateInvoices: leaseData.autoGenerateInvoices,
          autoEmailInvoices: leaseData.autoEmailInvoices,
          autoCreatePayments: true,
          prorationEnabled: true,
          advancePaymentMonths: 0,
        },
      },
    };

    const targetLeaseId = leaseId ?? initialLease?._id;
    const endpoint =
      mode === "edit" && targetLeaseId
        ? `/api/leases/${targetLeaseId}`
        : "/api/leases";
    const method = mode === "edit" ? "PUT" : "POST";
    const payload =
      mode === "edit"
        ? basePayload
        : {
            ...basePayload,
            status: "active" as const,
          };

    try {
      const response = await fetch(endpoint, {
        method,
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(payload),
      });

      const result = await response.json();

      if (!response.ok || !result.success) {
        throw new Error(result.error || result.message || "Failed to save lease");
      }

      const resultLease = result.data?.lease ?? result.data ?? null;
      const resultLeaseId =
        resultLease?._id ??
        targetLeaseId ??
        (typeof result.data?.id === "string" ? result.data.id : undefined);

      if (mode === "edit") {
        toast.success("Lease updated successfully!");

        setLeaseError(null);

        if (resultLease) {
          hydrateLeaseData(resultLease);
        } else if (targetLeaseId) {
          void loadLeaseDetails(targetLeaseId);
        }

        if (onSuccess) {
          onSuccess(resultLeaseId);
        } else if (resultLeaseId) {
          window.location.href = `/dashboard/leases/${resultLeaseId}`;
        }
        return;
      }

      toast.success("Lease created successfully!");

      if (result.data?.invoiceGeneration) {
        const { invoicesGenerated, errors: invoiceErrors } =
          result.data.invoiceGeneration;
        if (invoicesGenerated > 0) {
          toast.success(`${invoicesGenerated} invoices generated automatically`, {
            description:
              "Invoices are now available in the invoice management system",
            duration: 6000,
          });
        }
        if (Array.isArray(invoiceErrors) && invoiceErrors.length > 0) {
          toast.warning(`Invoice generation warnings: ${invoiceErrors.join(", ")}`);
        }
      }

      setLeaseData(createInitialLeaseState());
      setOriginalLeaseData(createInitialLeaseState());

      const navigateAfterCreate = () => {
        if (onSuccess) {
          onSuccess(resultLeaseId);
        } else if (resultLeaseId) {
          window.location.href = `/dashboard/leases/${resultLeaseId}`;
        } else {
          window.location.href = "/dashboard/leases";
        }
      };

      setTimeout(navigateAfterCreate, 2000);
    } catch (error) {
      const message =
        error instanceof Error ? error.message : "Failed to save lease";
      toast.error(
        mode === "edit" ? "Failed to update lease" : "Failed to create lease",
        {
          description: message,
          duration: 5000,
        }
      );
    } finally {
      setSubmitting(false);
    }
  };

  if (isEditMode && initializingLease) {
    return (
      <div className="flex items-center justify-center py-16">
        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  return (
    <div className="max-w-4xl mx-auto space-y-6">
      <form onSubmit={handleSubmit} className="space-y-6">
        {leaseError && (
          <Alert variant="destructive">
            <AlertDescription className="flex flex-col gap-3 md:flex-row md:items-center md:justify-between">
              <span>{leaseError}</span>
              {isEditMode && (leaseId || initialLease?._id) && (
                <Button
                  type="button"
                  variant="outline"
                  size="sm"
                  onClick={() =>
                    void loadLeaseDetails(leaseId ?? initialLease?._id!)
                  }
                >
                  Retry loading lease
                </Button>
              )}
            </AlertDescription>
          </Alert>
        )}

        {/* Property & Tenant Selection */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Home className="h-5 w-5" />
              Property & Tenant
            </CardTitle>
            <CardDescription>
              Select the property, unit, and tenant for this lease
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label htmlFor="property">
                  Property ({properties.length} available)
                </Label>
                <Select
                  value={leaseData.propertyId}
                  onValueChange={(value) => {
                    handleInputChange("propertyId", value);
                    handleInputChange("unitId", ""); // Reset unit when property changes
                  }}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select property" />
                  </SelectTrigger>
                  <SelectContent>
                    {properties.map((property) => (
                      <SelectItem key={property._id} value={property._id}>
                        {property.name} - {property.address?.street},{" "}
                        {property.address?.city}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label htmlFor="unit">
                  Unit{" "}
                  {leaseData.propertyId &&
                    `(${getAvailableUnits().length} available)`}
                </Label>
                <Select
                  value={leaseData.unitId}
                  onValueChange={handleUnitChange}
                  disabled={!leaseData.propertyId || loadingProperties}
                >
                  <SelectTrigger>
                    <SelectValue
                      placeholder={
                        !leaseData.propertyId
                          ? "Select property first"
                          : loadingProperties
                          ? "Loading units..."
                          : getAvailableUnits().length === 0
                          ? "No available units"
                          : "Select unit"
                      }
                    />
                  </SelectTrigger>
                  <SelectContent>
                    {getAvailableUnits().length > 0 ? (
                      getAvailableUnits().map((unit) => (
                        <SelectItem key={unit._id} value={unit._id}>
                          <div className="flex items-center justify-between w-full">
                            <span>Unit {unit.unitNumber}</span>
                            <span className="text-xs text-gray-500 ml-2">
                              {unit.bedrooms}BR/{unit.bathrooms}BA - $
                              {unit.rentAmount}/mo
                            </span>
                          </div>
                        </SelectItem>
                      ))
                    ) : leaseData.propertyId ? (
                      <div className="px-2 py-1 text-sm text-gray-500">
                        No available units in this property
                      </div>
                    ) : null}
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="tenant">Tenant</Label>
              <Select
                value={leaseData.tenantId}
                onValueChange={(value) => handleInputChange("tenantId", value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select tenant" />
                </SelectTrigger>
                <SelectContent>
                  {tenants.map((tenant) => (
                    <SelectItem key={tenant._id} value={tenant._id}>
                      {tenant.firstName} {tenant.lastName} - {tenant.email}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </CardContent>
        </Card>

        {/* Selected Unit Details */}
        {leaseData.unitId && getSelectedUnit() && (
          <Card>
            <CardHeader>
              <CardTitle className="text-sm font-medium text-blue-600">
                Selected Unit Details
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4 p-4 bg-blue-50 rounded-lg">
                <div>
                  <div className="text-xs text-gray-500 mb-1">Unit:</div>
                  <div className="font-medium">
                    {getSelectedUnit()?.unitNumber} (
                    {getSelectedUnit()?.unitType})
                  </div>
                </div>
                <div>
                  <div className="text-xs text-gray-500 mb-1">Size:</div>
                  <div className="font-medium">
                    {getSelectedUnit()?.bedrooms}BR/
                    {getSelectedUnit()?.bathrooms}BA
                  </div>
                </div>
                <div>
                  <div className="text-xs text-gray-500 mb-1">Area:</div>
                  <div className="font-medium">
                    {getSelectedUnit()?.squareFootage} sq ft
                  </div>
                </div>
                <div>
                  <div className="text-xs text-gray-500 mb-1">Rent:</div>
                  <div className="font-medium text-green-600">
                    ${getSelectedUnit()?.rentAmount?.toLocaleString()}/month
                  </div>
                </div>
              </div>
              {(getSelectedUnit()?.rentAmount ||
                getSelectedUnit()?.securityDeposit) && (
                <div className="mt-3 text-xs text-blue-600 flex items-center gap-1">
                  <CheckCircle className="h-3 w-3" />
                  Financial terms have been auto-filled from unit details
                </div>
              )}
            </CardContent>
          </Card>
        )}

        {/* Lease Dates */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Calendar className="h-5 w-5" />
              Lease Dates
            </CardTitle>
            <CardDescription>Set the lease start and end dates</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label htmlFor="startDate">Start Date</Label>
                <FormDatePicker
                  key={`start-date-${leaseData.startDate}`}
                  value={
                    leaseData.startDate
                      ? new Date(leaseData.startDate + "T00:00:00")
                      : undefined
                  }
                  onChange={(date) => {
                    if (date) {
                      // Create a new date to avoid timezone issues
                      const localDate = new Date(
                        date.getFullYear(),
                        date.getMonth(),
                        date.getDate()
                      );
                      handleInputChange("startDate", format(localDate, "yyyy-MM-dd"));
                    } else {
                      handleInputChange("startDate", "");
                    }
                  }}
                  placeholder="Select lease start date"
                  disabled={(date) => {
                    const today = new Date();
                    today.setHours(0, 0, 0, 0);
                    const checkDate = new Date(date);
                    checkDate.setHours(0, 0, 0, 0);
                    return checkDate < today;
                  }}
                  fromYear={new Date().getFullYear()}
                  toYear={new Date().getFullYear() + 5}
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="endDate">End Date</Label>
                <FormDatePicker
                  key={`end-date-${leaseData.endDate}-${leaseData.startDate}`}
                  value={
                    leaseData.endDate
                      ? new Date(leaseData.endDate + "T00:00:00")
                      : undefined
                  }
                  onChange={(date) => {
                    if (date) {
                      // Create a new date to avoid timezone issues
                      const localDate = new Date(
                        date.getFullYear(),
                        date.getMonth(),
                        date.getDate()
                      );
                      handleInputChange("endDate", format(localDate, "yyyy-MM-dd"));
                    } else {
                      handleInputChange("endDate", "");
                    }
                  }}
                  placeholder="Select lease end date"
                  disabled={(date) => {
                    const today = new Date();
                    today.setHours(0, 0, 0, 0);
                    const checkDate = new Date(date);
                    checkDate.setHours(0, 0, 0, 0);

                    if (!leaseData.startDate) return checkDate < today;

                    const startDate = new Date(
                      leaseData.startDate + "T00:00:00"
                    );
                    startDate.setHours(0, 0, 0, 0);
                    return checkDate <= startDate;
                  }}
                  fromYear={new Date().getFullYear()}
                  toYear={new Date().getFullYear() + 10}
                />
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Financial Terms */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <DollarSign className="h-5 w-5" />
              Financial Terms
            </CardTitle>
            <CardDescription>
              Set rent amount, security deposit, and payment due date
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-4 md:grid-cols-3">
              <div className="space-y-2">
                <Label htmlFor="rentAmount">Monthly Rent</Label>
                <Input
                  id="rentAmount"
                  type="number"
                  min="0"
                  step="0.01"
                  value={leaseData.rentAmount}
                  onChange={(e) =>
                    handleInputChange(
                      "rentAmount",
                      parseFloat(e.target.value) || 0
                    )
                  }
                  placeholder="2000.00"
                  required
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="securityDeposit">Security Deposit</Label>
                <Input
                  id="securityDeposit"
                  type="number"
                  min="0"
                  step="0.01"
                  value={leaseData.securityDeposit}
                  onChange={(e) =>
                    handleInputChange(
                      "securityDeposit",
                      parseFloat(e.target.value) || 0
                    )
                  }
                  placeholder="2000.00"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="rentDueDay">Rent Due Day</Label>
                <Select
                  value={leaseData.rentDueDay.toString()}
                  onValueChange={(value) =>
                    handleInputChange("rentDueDay", parseInt(value))
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {Array.from({ length: 28 }, (_, i) => i + 1).map((day) => (
                      <SelectItem key={day} value={day.toString()}>
                        {day}
                        {day === 1
                          ? "st"
                          : day === 2
                          ? "nd"
                          : day === 3
                          ? "rd"
                          : "th"}{" "}
                        of month
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Late Fee Configuration */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertTriangle className="h-5 w-5" />
              Late Fee Rules
            </CardTitle>
            <CardDescription>
              Configure late fee amount and grace period
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-4 md:grid-cols-3">
              <div className="space-y-2">
                <Label htmlFor="lateFeeAmount">Late Fee Amount</Label>
                <Input
                  id="lateFeeAmount"
                  type="number"
                  min="0"
                  step="0.01"
                  value={leaseData.lateFeeAmount}
                  onChange={(e) =>
                    handleInputChange(
                      "lateFeeAmount",
                      parseFloat(e.target.value) || 0
                    )
                  }
                  placeholder="50.00"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="lateFeeGracePeriodDays">
                  Grace Period (Days)
                </Label>
                <Input
                  id="lateFeeGracePeriodDays"
                  type="number"
                  min="0"
                  value={leaseData.lateFeeGracePeriodDays}
                  onChange={(e) =>
                    handleInputChange(
                      "lateFeeGracePeriodDays",
                      parseInt(e.target.value) || 0
                    )
                  }
                  placeholder="5"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="lateFeeType">Late Fee Type</Label>
                <Select
                  value={leaseData.lateFeeType}
                  onValueChange={(value: "fixed" | "percentage") =>
                    handleInputChange("lateFeeType", value)
                  }
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="fixed">Fixed Amount</SelectItem>
                    <SelectItem value="percentage">
                      Percentage of Rent
                    </SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Automation Settings */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <CheckCircle className="h-5 w-5" />
              Automation Settings
            </CardTitle>
            <CardDescription>
              Configure automatic invoice generation and email sending
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex items-center justify-between">
              <div className="space-y-0.5">
                <Label htmlFor="autoGenerateInvoices">
                  Auto-Generate Invoices
                </Label>
                <p className="text-sm text-muted-foreground">
                  Automatically create invoices for rent and deposits
                </p>
              </div>
              <Switch
                id="autoGenerateInvoices"
                checked={leaseData.autoGenerateInvoices}
                onCheckedChange={(checked) =>
                  handleInputChange("autoGenerateInvoices", checked)
                }
              />
            </div>

            <Separator />

            <div className="flex items-center justify-between">
              <div className="space-y-0.5">
                <Label htmlFor="autoEmailInvoices">Auto-Email Invoices</Label>
                <p className="text-sm text-muted-foreground">
                  Automatically send invoices to tenant via email
                </p>
              </div>
              <Switch
                id="autoEmailInvoices"
                checked={leaseData.autoEmailInvoices}
                onCheckedChange={(checked) =>
                  handleInputChange("autoEmailInvoices", checked)
                }
                disabled={!leaseData.autoGenerateInvoices}
              />
            </div>
          </CardContent>
        </Card>

        {/* Summary & Submit */}
        <Card>
          <CardHeader>
            <CardTitle>Review & Create</CardTitle>
            <CardDescription>
              Review the lease details and create the lease
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            {leaseData.autoGenerateInvoices && (
              <Alert>
                <CheckCircle className="h-4 w-4" />
                <AlertDescription>
                  Invoices will be automatically generated for this lease
                  including:
                  {leaseData.securityDeposit > 0 && " security deposit,"}
                  {" monthly rent payments"}
                  {leaseData.autoEmailInvoices &&
                    ", and automatically emailed to the tenant"}
                </AlertDescription>
              </Alert>
            )}

            <div className="flex gap-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => {
                  if (isEditMode) {
                    setLeaseData({ ...originalLeaseData });
                  } else {
                    const resetState = createInitialLeaseState();
                    setLeaseData(resetState);
                    setOriginalLeaseData(createInitialLeaseState());
                  }
                }}
                disabled={submitting || (isEditMode && initializingLease)}
              >
                {resetLabel}
              </Button>

              <Button
                type="submit"
                disabled={
                  submitting || (isEditMode && initializingLease)
                }
                className="flex-1"
              >
                {submitting ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    {submitLoadingLabel}
                  </>
                ) : (
                  submitLabel
                )}
              </Button>
            </div>
          </CardContent>
        </Card>
      </form>
    </div>
  );
}
