"use client";

import React, { useState, useEffect } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Alert, AlertDescription } from "@/components/ui/alert";
import {
  DollarSign,
  TrendingUp,
  Calendar,
  CheckCircle,
  Clock,
  AlertTriangle,
  Plus,
  RefreshCw,
  CreditCard,
  Banknote,
} from "lucide-react";
import { PaymentStatus, PaymentMethod, IPayment, ILease } from "@/types";
import { LeaseResponse } from "@/lib/services/lease.service";
import { toast } from "sonner";
import {
  showErrorToast,
  showSuccessToast,
  showWarningToast,
  showInfoToast,
  retryWithBackoff,
  PropertyProError,
  ErrorType,
} from "@/lib/error-handling";
import {
  usePaymentListUpdates,
  useRealTimePayments,
} from "@/hooks/useRealTimePayments";

interface PaymentStatusDashboardProps {
  leaseId: string;
  lease: LeaseResponse;
  onPaymentUpdate?: (payment: IPayment) => void;
  onInvoiceGenerated?: (invoiceId: string) => void;
}

interface PaymentSummary {
  totalDue: number;
  totalPaid: number;
  totalOverdue: number;
  totalUpcoming: number;
  paymentProgress: number;
  nextPaymentDate: string | null;
  nextPaymentAmount: number;
}

export function PaymentStatusDashboard({
  leaseId,
  lease,
  onPaymentUpdate,
  onInvoiceGenerated,
}: PaymentStatusDashboardProps) {
  const [payments, setPayments] = useState<IPayment[]>([]);
  const [summary, setSummary] = useState<PaymentSummary>({
    totalDue: 0,
    totalPaid: 0,
    totalOverdue: 0,
    totalUpcoming: 0,
    paymentProgress: 0,
    nextPaymentDate: null,
    nextPaymentAmount: 0,
  });
  const [loading, setLoading] = useState(true);
  const [refreshing, setRefreshing] = useState(false);

  // Real-time payment updates
  const { isConnected, connectionError, reconnect } = useRealTimePayments({
    leaseId,
    enabled: true,
  });

  // Auto-update payment list when real-time updates arrive
  usePaymentListUpdates(payments, setPayments, { leaseId });

  useEffect(() => {
    fetchPaymentData();
  }, [leaseId]);

  const fetchPaymentData = async () => {
    try {
      setLoading(true);

      const data = await retryWithBackoff(
        async () => {
          const response = await fetch(`/api/payments?leaseId=${leaseId}`);

          if (!response.ok) {
            const errorData = await response.json();
            throw new PropertyProError(
              ErrorType.NETWORK,
              errorData.message || "Failed to fetch payment data",
              {
                code: "PAYMENT_FETCH_FAILED",
                retryable: response.status >= 500,
              }
            );
          }

          return response.json();
        },
        3,
        1000
      );

      if (data?.success) {
        setPayments(data?.data ?? []);
        calculateSummary(data?.data ?? []);
      } else {
        throw new PropertyProError(
          ErrorType.DATABASE,
          data?.message || "Invalid payment data received",
          { code: "INVALID_PAYMENT_DATA" }
        );
      }
    } catch (error) {
      console.error("Error fetching payment data:", error);
      showErrorToast(error);
    } finally {
      setLoading(false);
    }
  };

  const calculateSummary = (paymentData: IPayment[]) => {
    const now = new Date();

    let totalDue = 0;
    let totalPaid = 0;
    let totalOverdue = 0;
    let totalUpcoming = 0;
    let nextPayment: IPayment | null = null;

    paymentData?.forEach((payment) => {
      const dueDate = new Date(payment?.dueDate ?? new Date());
      const isPastDue = dueDate < now;

      totalDue += payment?.amount ?? 0;
      totalPaid += payment?.amountPaid ?? 0;

      if (payment?.status === PaymentStatus.OVERDUE) {
        totalOverdue += (payment?.amount ?? 0) - (payment?.amountPaid ?? 0);
      } else if (payment?.status === PaymentStatus.PENDING && !isPastDue) {
        totalUpcoming += (payment?.amount ?? 0) - (payment?.amountPaid ?? 0);

        if (!nextPayment || dueDate < new Date(nextPayment?.dueDate ?? new Date())) {
          nextPayment = payment;
        }
      }
    });

    const paymentProgress = totalDue > 0 ? (totalPaid / totalDue) * 100 : 0;

    setSummary({
      totalDue,
      totalPaid,
      totalOverdue,
      totalUpcoming,
      paymentProgress,
      nextPaymentDate: nextPayment?.dueDate || null,
      nextPaymentAmount: nextPayment?.amount || 0,
    });
  };

  const handleRefresh = async () => {
    try {
      setRefreshing(true);
      showInfoToast("Refreshing payment data...");
      await fetchPaymentData();
      showSuccessToast("Payment data refreshed successfully");
    } catch (error) {
      console.error("Error refreshing payment data:", error);
      showErrorToast(error);
    } finally {
      setRefreshing(false);
    }
  };

  const handleCreatePayments = async () => {
    try {
      const response = await fetch(`/api/payments`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          tenantId: lease?.tenantId?._id || lease?.tenantId,
          propertyId: lease?.propertyId?._id || lease?.propertyId,
          leaseId: leaseId,
          amount: lease?.terms?.rentAmount ?? 0,
          type: "rent",
          dueDate: (() => {
            const now = new Date();
            const leaseStart = new Date(lease?.startDate ?? new Date());
            const leaseEnd = new Date(lease?.endDate ?? new Date());

            // If current date is within lease period, use it
            if (now >= leaseStart && now <= leaseEnd) {
              return now.toISOString();
            }

            // If current date is before lease start, use lease start date
            if (now < leaseStart) {
              return leaseStart.toISOString();
            }

            // If current date is after lease end, use a date 30 days before lease end
            const thirtyDaysBeforeEnd = new Date(leaseEnd);
            thirtyDaysBeforeEnd.setDate(thirtyDaysBeforeEnd.getDate() - 30);
            return thirtyDaysBeforeEnd.toISOString();
          })(),
          description: "Monthly rent payment",
        }),
      });

      const data = await response.json();

      if (data?.success) {
        toast.success("Payment created successfully");
        fetchPaymentData();
      } else {
        toast.error("Failed to create payment");
      }
    } catch (error) {
      console.error("Error creating payment:", error);
      toast.error("Failed to create payment");
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-US", {
      style: "currency",
      currency: "USD",
    }).format(amount);
  };

  const getStatusColor = (status: PaymentStatus) => {
    switch (status) {
      case PaymentStatus.COMPLETED:
        return "text-green-600";
      case PaymentStatus.PENDING:
        return "text-yellow-600";
      case PaymentStatus.OVERDUE:
        return "text-red-600";
      case PaymentStatus.PROCESSING:
        return "text-blue-600";
      default:
        return "text-gray-600";
    }
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="animate-pulse space-y-4">
            <div className="h-4 bg-gray-200 rounded w-3/4"></div>
            <div className="h-8 bg-gray-200 rounded w-1/2"></div>
            <div className="grid grid-cols-4 gap-4">
              {[...Array(4)].map((_, i) => (
                <div key={i} className="h-20 bg-gray-200 rounded"></div>
              ))}
            </div>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle className="flex items-center gap-2">
                <DollarSign className="h-5 w-5" />
                Payment Status Dashboard
              </CardTitle>
              <CardDescription>
                Overview of payment status for{" "}
                {lease?.propertyId?.name || "this lease"}
              </CardDescription>
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={handleRefresh}
                disabled={refreshing}
              >
                <RefreshCw
                  className={`h-4 w-4 mr-2 ${refreshing ? "animate-spin" : ""}`}
                />
                Refresh
              </Button>
              <Button size="sm" onClick={handleCreatePayments}>
                <Plus className="h-4 w-4 mr-2" />
                Create Payment
              </Button>
            </div>
          </div>
        </CardHeader>
      </Card>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Total Due
                </p>
                <p className="text-2xl font-bold">
                  {formatCurrency(summary?.totalDue ?? 0)}
                </p>
              </div>
              <div className="p-2 bg-blue-100 rounded-lg">
                <DollarSign className="h-6 w-6 text-blue-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Total Paid
                </p>
                <p className="text-2xl font-bold text-green-600">
                  {formatCurrency(summary.totalPaid)}
                </p>
              </div>
              <div className="p-2 bg-green-100 rounded-lg">
                <CheckCircle className="h-6 w-6 text-green-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Overdue
                </p>
                <p className="text-2xl font-bold text-red-600">
                  {formatCurrency(summary.totalOverdue)}
                </p>
              </div>
              <div className="p-2 bg-red-100 rounded-lg">
                <AlertTriangle className="h-6 w-6 text-red-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Upcoming
                </p>
                <p className="text-2xl font-bold text-yellow-600">
                  {formatCurrency(summary.totalUpcoming)}
                </p>
              </div>
              <div className="p-2 bg-yellow-100 rounded-lg">
                <Clock className="h-6 w-6 text-yellow-600" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Payment Progress */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="h-5 w-5" />
            Payment Progress
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium">Overall Progress</span>
              <span className="text-sm text-muted-foreground">
                {summary.paymentProgress.toFixed(1)}%
              </span>
            </div>
            <Progress value={summary.paymentProgress} className="h-2" />
            <div className="flex items-center justify-between text-sm">
              <span className="text-muted-foreground">
                {formatCurrency(summary.totalPaid)} paid
              </span>
              <span className="text-muted-foreground">
                {formatCurrency(summary.totalDue)} total
              </span>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Next Payment Alert */}
      {summary.nextPaymentDate && (
        <Alert>
          <Calendar className="h-4 w-4" />
          <AlertDescription>
            Next payment of {formatCurrency(summary.nextPaymentAmount)} is due
            on {new Date(summary.nextPaymentDate).toLocaleDateString()}
          </AlertDescription>
        </Alert>
      )}

      {/* Recent Payments */}
      <Card>
        <CardHeader>
          <CardTitle>Recent Payments</CardTitle>
          <CardDescription>
            Latest payment activity for this lease
          </CardDescription>
        </CardHeader>
        <CardContent>
          {payments.length === 0 ? (
            <div className="text-center py-8">
              <p className="text-muted-foreground">
                No payments found for this lease
              </p>
              <Button className="mt-4" onClick={handleCreatePayments}>
                Create First Payment
              </Button>
            </div>
          ) : (
            <div className="space-y-4">
              {payments.slice(0, 5).map((payment) => (
                <div
                  key={payment._id.toString()}
                  className="flex items-center justify-between p-4 border rounded-lg"
                >
                  <div className="flex items-center gap-4">
                    <div className="p-2 bg-gray-100 rounded-lg">
                      {payment.paymentMethod === PaymentMethod.CREDIT_CARD ? (
                        <CreditCard className="h-4 w-4" />
                      ) : (
                        <Banknote className="h-4 w-4" />
                      )}
                    </div>
                    <div>
                      <p className="font-medium">{payment.type}</p>
                      <p className="text-sm text-muted-foreground">
                        Due: {new Date(payment.dueDate).toLocaleDateString()}
                      </p>
                    </div>
                  </div>
                  <div className="text-right">
                    <p className="font-medium">
                      {formatCurrency(payment.amount)}
                    </p>
                    <Badge
                      variant={
                        payment.status === PaymentStatus.COMPLETED
                          ? "default"
                          : "secondary"
                      }
                      className={getStatusColor(payment.status)}
                    >
                      {payment.status}
                    </Badge>
                  </div>
                </div>
              ))}

              {payments.length > 5 && (
                <div className="text-center pt-4">
                  <Button variant="outline" size="sm">
                    View All Payments ({payments.length})
                  </Button>
                </div>
              )}
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
