"use client";

import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import {
  Plus,
  Trash2,
  Star,
  Home,
  Car,
  Shield,
  Wifi,
  Zap,
  Trees,
  Utensils,
  Bath,
  Bed,
  Building,
} from "lucide-react";
import { toast } from "sonner";
import { propertyService } from "@/lib/services/property.service";

interface Amenity {
  name: string;
  description?: string;
  category: string;
}

interface PropertyAmenitiesProps {
  amenities: Amenity[];
  canEdit: boolean;
  onAmenitiesUpdate: (newAmenities: Amenity[]) => void;
  propertyId: string;
}

const amenityCategories = [
  "Kitchen",
  "Bathroom",
  "Living",
  "Bedroom",
  "Outdoor",
  "Parking",
  "Security",
  "Utilities",
  "Recreation",
  "Laundry",
  "Climate",
  "Other",
];

const categoryIcons: Record<
  string,
  React.ComponentType<{ className?: string }>
> = {
  Kitchen: Utensils,
  Bathroom: Bath,
  Living: Home,
  Bedroom: Bed,
  Outdoor: Trees,
  Parking: Car,
  Security: Shield,
  Utilities: Zap,
  Recreation: Star,
  Other: Building,
};

const PropertyAmenities: React.FC<PropertyAmenitiesProps> = ({
  amenities,
  canEdit,
  onAmenitiesUpdate,
  propertyId,
}) => {
  const [showAddDialog, setShowAddDialog] = useState(false);
  const [showDeleteDialog, setShowDeleteDialog] = useState(false);
  const [amenityToDelete, setAmenityToDelete] = useState<string | null>(null);
  const [newAmenity, setNewAmenity] = useState({
    name: "",
    description: "",
    category: "",
  });
  const [loading, setLoading] = useState(false);

  const handleAddAmenity = async () => {
    if (!newAmenity.name.trim() || !newAmenity.category) {
      toast.error("Please fill in the required fields");
      return;
    }

    try {
      setLoading(true);
      const amenityData = {
        name: newAmenity.name.trim(),
        description: newAmenity.description.trim() || undefined,
        category: newAmenity.category,
      };

      const response = await propertyService.addPropertyAmenities(propertyId, [
        amenityData,
      ]);
      onAmenitiesUpdate(response.amenities);
      setNewAmenity({ name: "", description: "", category: "" });
      setShowAddDialog(false);
      toast.success("Amenity added successfully");
    } catch (error: any) {
      toast.error(error.message || "Failed to add amenity");
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteAmenity = async () => {
    if (!amenityToDelete) return;

    try {
      setLoading(true);
      const response = await propertyService.removePropertyAmenities(
        propertyId,
        [amenityToDelete]
      );
      onAmenitiesUpdate(response.amenities);
      setAmenityToDelete(null);
      setShowDeleteDialog(false);
      toast.success("Amenity removed successfully");
    } catch (error: any) {
      toast.error(error.message || "Failed to remove amenity");
    } finally {
      setLoading(false);
    }
  };

  const openDeleteDialog = (amenityName: string) => {
    setAmenityToDelete(amenityName);
    setShowDeleteDialog(true);
  };

  const handleDeleteCancel = () => {
    setShowDeleteDialog(false);
    setAmenityToDelete(null);
  };

  const groupedAmenities = amenities.reduce((acc, amenity) => {
    if (!acc[amenity.category]) {
      acc[amenity.category] = [];
    }
    acc[amenity.category].push(amenity);
    return acc;
  }, {} as Record<string, Amenity[]>);

  if (amenities.length === 0 && !canEdit) {
    return (
      <Card>
        <CardContent className="flex flex-col items-center justify-center py-12">
          <Star className="h-12 w-12 text-gray-400 mb-4" />
          <h3 className="text-lg font-semibold text-gray-900 mb-2">
            No Amenities Listed
          </h3>
          <p className="text-gray-600 text-center">
            No amenities have been added to this property yet.
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h3 className="text-lg font-semibold">Property Amenities</h3>
          <p className="text-sm text-gray-600 dark:text-gray-400">
            {amenities.length} amenit{amenities.length !== 1 ? "ies" : "y"}{" "}
            available
          </p>
        </div>
        {canEdit && (
          <Dialog open={showAddDialog} onOpenChange={setShowAddDialog}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Add Amenity
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Add New Amenity</DialogTitle>
                <DialogDescription>
                  Add a new amenity to highlight the features of your property.
                </DialogDescription>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label htmlFor="amenityName">Amenity Name *</Label>
                  <Input
                    id="amenityName"
                    placeholder="e.g., Swimming Pool, Gym, Parking"
                    value={newAmenity.name}
                    onChange={(e) =>
                      setNewAmenity({ ...newAmenity, name: e.target.value })
                    }
                  />
                </div>
                <div>
                  <Label htmlFor="amenityCategory">Category *</Label>
                  <Select
                    value={newAmenity.category}
                    onValueChange={(value) =>
                      setNewAmenity({ ...newAmenity, category: value })
                    }
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select a category" />
                    </SelectTrigger>
                    <SelectContent>
                      {amenityCategories.map((category) => (
                        <SelectItem key={category} value={category}>
                          {category}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="amenityDescription">
                    Description (Optional)
                  </Label>
                  <Textarea
                    id="amenityDescription"
                    placeholder="Provide additional details about this amenity..."
                    value={newAmenity.description}
                    onChange={(e) =>
                      setNewAmenity({
                        ...newAmenity,
                        description: e.target.value,
                      })
                    }
                    rows={3}
                  />
                </div>
              </div>
              <DialogFooter>
                <Button
                  variant="outline"
                  onClick={() => {
                    setNewAmenity({ name: "", description: "", category: "" });
                    setShowAddDialog(false);
                  }}
                >
                  Cancel
                </Button>
                <Button
                  onClick={handleAddAmenity}
                  disabled={
                    loading || !newAmenity.name.trim() || !newAmenity.category
                  }
                >
                  {loading ? "Adding..." : "Add Amenity"}
                </Button>
              </DialogFooter>
            </DialogContent>
          </Dialog>
        )}
      </div>

      {/* Amenities Display */}
      {amenities.length > 0 ? (
        <div className="space-y-6">
          {Object.entries(groupedAmenities).map(
            ([category, categoryAmenities]) => {
              const IconComponent = categoryIcons[category] || Building;

              return (
                <Card key={category}>
                  <CardHeader>
                    <CardTitle className="flex items-center text-lg">
                      <IconComponent className="h-5 w-5 mr-2" />
                      {category}
                      <Badge variant="secondary" className="ml-2">
                        {categoryAmenities.length}
                      </Badge>
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      {categoryAmenities.map((amenity, index) => (
                        <div
                          key={index}
                          className="flex items-start justify-between p-3 border rounded-lg hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors"
                        >
                          <div className="flex-1">
                            <h4 className="font-medium text-gray-900 dark:text-gray-100">
                              {amenity.name}
                            </h4>
                            {amenity.description && (
                              <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                {amenity.description}
                              </p>
                            )}
                          </div>
                          {canEdit && (
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => openDeleteDialog(amenity.name)}
                              className="text-red-600 hover:text-red-700 hover:bg-red-50"
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          )}
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              );
            }
          )}
        </div>
      ) : (
        <Card>
          <CardContent className="flex flex-col items-center justify-center py-12">
            <Star className="h-12 w-12 text-gray-400 mb-4" />
            <h3 className="text-lg font-semibold text-gray-900 mb-2">
              No Amenities Yet
            </h3>
            <p className="text-gray-600 text-center mb-4">
              Start highlighting your property features by adding some
              amenities.
            </p>
            {canEdit && (
              <Button onClick={() => setShowAddDialog(true)}>
                <Plus className="h-4 w-4 mr-2" />
                Add First Amenity
              </Button>
            )}
          </CardContent>
        </Card>
      )}

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={showDeleteDialog} onOpenChange={setShowDeleteDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Remove Amenity</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to remove &quot;{amenityToDelete}&quot; from
              the property amenities? This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel onClick={handleDeleteCancel} disabled={loading}>
              Cancel
            </AlertDialogCancel>
            <AlertDialogAction
              onClick={handleDeleteAmenity}
              disabled={loading}
              className="bg-red-600 hover:bg-red-700 focus:ring-red-600"
            >
              {loading ? "Removing..." : "Remove Amenity"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
};

export default PropertyAmenities;
