"use client";

import React, { useState, useEffect } from "react";
import {
  X,
  MoreVertical,
  Edit,
  Trash2,
  DollarSign,
  Home,
  MapPin,
  Calendar,
  Bed,
  Bath,
  Car,
  Wifi,
  Zap,
  Droplets,
  Thermometer,
  Camera,
  User,
  FileText,
  Building2,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog";
import { PropertyStatus } from "@/types";
import { formatCurrency } from "@/lib/utils/formatting";

interface UnitDetails {
  id: string;
  unitNumber: string;
  unitType: string;
  bedrooms: number;
  bathrooms: number;
  squareFootage: number;
  rentAmount: number;
  securityDeposit: number;
  status: PropertyStatus;
  floor?: string;
  balcony: boolean;
  patio: boolean;
  garden: boolean;
  parking: {
    included: boolean;
    spaces?: number;
    type?: string;
  };
  utilities: {
    electricity: boolean;
    water: boolean;
    gas: boolean;
    internet: boolean;
    heating: boolean;
    cooling: boolean;
  };
  images: string[];
  notes?: string;
  availableFrom?: string;
  lastRenovated?: string;
}

interface PropertyInfo {
  id: string;
  name: string;
  address?: {
    street?: string;
    city?: string;
    state?: string;
    zipCode?: string;
  };
  type: string;
}

interface UnitDetailsModalProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  propertyId: string;
  unitId: string;
  onUnitUpdated?: () => void;
  onUnitDeleted?: () => void;
}

export default function UnitDetailsModal({
  open,
  onOpenChange,
  propertyId,
  unitId,
  onUnitUpdated,
  onUnitDeleted,
}: UnitDetailsModalProps) {
  const [unit, setUnit] = useState<UnitDetails | null>(null);
  const [property, setProperty] = useState<PropertyInfo | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState("overview");

  useEffect(() => {
    if (open && propertyId && unitId) {
      fetchUnitDetails();
      fetchPropertyDetails();
    }
  }, [open, propertyId, unitId]);

  const fetchUnitDetails = async () => {
    setLoading(true);
    setError(null);
    try {
      const response = await fetch(
        `/api/properties/${propertyId}/units/${unitId}`
      );
      if (!response.ok) {
        throw new Error("Failed to fetch unit details");
      }
      const data = await response.json();
      setUnit(data);
    } catch (error) {
      setError("Failed to load unit details");
    } finally {
      setLoading(false);
    }
  };

  const fetchPropertyDetails = async () => {
    try {
      const response = await fetch(`/api/properties/${propertyId}`);
      if (!response.ok) {
        throw new Error("Failed to fetch property details");
      }
      const data = await response.json();
      setProperty(data);
    } catch (error) {}
  };

  const handleDeleteUnit = async () => {
    try {
      const response = await fetch(
        `/api/properties/${propertyId}/units/${unitId}`,
        {
          method: "DELETE",
        }
      );

      if (!response.ok) {
        throw new Error("Failed to delete unit");
      }

      onUnitDeleted?.();
      onOpenChange(false);
    } catch (error) {}
  };

  const getStatusColor = (status: PropertyStatus) => {
    switch (status) {
      case PropertyStatus.AVAILABLE:
        return "bg-green-100 text-green-800";
      case PropertyStatus.OCCUPIED:
        return "bg-blue-100 text-blue-800";
      case PropertyStatus.MAINTENANCE:
        return "bg-yellow-100 text-yellow-800";
      case PropertyStatus.UNAVAILABLE:
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusIcon = (status: PropertyStatus) => {
    switch (status) {
      case PropertyStatus.AVAILABLE:
        return "🟢";
      case PropertyStatus.OCCUPIED:
        return "🔵";
      case PropertyStatus.MAINTENANCE:
        return "🟡";
      case PropertyStatus.UNAVAILABLE:
        return "🔴";
      default:
        return "⚪";
    }
  };

  if (!open) return null;

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center">
      {/* Backdrop */}
      <div
        className="absolute inset-0 bg-black/50 backdrop-blur-sm"
        onClick={() => onOpenChange(false)}
      />

      {/* Modal Content */}
      <div className="relative bg-white rounded-lg shadow-2xl w-[96vw] max-w-[1400px] h-[68vh] max-h-[750px] overflow-hidden flex flex-col">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b bg-white">
          <div>
            {loading ? (
              <div className="space-y-2">
                <div className="h-8 bg-gray-200 rounded animate-pulse w-48" />
                <div className="h-4 bg-gray-200 rounded animate-pulse w-64" />
              </div>
            ) : unit && property ? (
              <>
                <h2 className="text-2xl font-bold text-gray-900">
                  Unit {unit.unitNumber}
                </h2>
                <p className="text-gray-600">
                  {property?.name}
                  {property?.address?.city && property?.address?.state && (
                    <>
                      {" "}
                      • {property.address.city}, {property.address.state}
                    </>
                  )}
                </p>
              </>
            ) : null}
          </div>

          <div className="flex items-center space-x-3">
            {unit && (
              <Badge className={`${getStatusColor(unit.status)} border`}>
                {getStatusIcon(unit.status)}
                <span className="ml-1">{unit.status}</span>
              </Badge>
            )}

            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="outline" size="sm">
                  <MoreVertical className="h-4 w-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                <DropdownMenuItem>
                  <Edit className="mr-2 h-4 w-4" />
                  Edit Unit
                </DropdownMenuItem>
                <AlertDialog>
                  <AlertDialogTrigger asChild>
                    <DropdownMenuItem
                      className="text-red-600 focus:text-red-600"
                      onSelect={(e) => e.preventDefault()}
                    >
                      <Trash2 className="mr-2 h-4 w-4" />
                      Delete Unit
                    </DropdownMenuItem>
                  </AlertDialogTrigger>
                  <AlertDialogContent>
                    <AlertDialogHeader>
                      <AlertDialogTitle>Delete Unit</AlertDialogTitle>
                      <AlertDialogDescription>
                        Are you sure you want to delete this unit? This action
                        cannot be undone.
                      </AlertDialogDescription>
                    </AlertDialogHeader>
                    <AlertDialogFooter>
                      <AlertDialogCancel>Cancel</AlertDialogCancel>
                      <AlertDialogAction
                        onClick={handleDeleteUnit}
                        className="bg-red-600 hover:bg-red-700"
                      >
                        Delete
                      </AlertDialogAction>
                    </AlertDialogFooter>
                  </AlertDialogContent>
                </AlertDialog>
              </DropdownMenuContent>
            </DropdownMenu>

            <Button
              variant="ghost"
              size="sm"
              onClick={() => onOpenChange(false)}
              className="h-8 w-8 p-0"
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
        </div>

        {/* Content */}
        <div className="flex-1 overflow-y-auto p-6">
          {loading ? (
            <div className="space-y-6">
              <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                {Array.from({ length: 4 }).map((_, i) => (
                  <Card key={i}>
                    <CardContent className="p-6">
                      <div className="space-y-3">
                        <div className="h-4 bg-gray-200 rounded animate-pulse w-full" />
                        <div className="h-8 bg-gray-200 rounded animate-pulse w-24" />
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>
          ) : error ? (
            <div className="text-center py-12">
              <p className="text-red-600 mb-4">{error}</p>
              <Button onClick={fetchUnitDetails} variant="outline">
                Try Again
              </Button>
            </div>
          ) : unit && property ? (
            <Tabs
              value={activeTab}
              onValueChange={setActiveTab}
              className="space-y-6"
            >
              <TabsList className="grid w-full grid-cols-4">
                <TabsTrigger value="overview">Overview</TabsTrigger>
                <TabsTrigger value="features">Features</TabsTrigger>
                <TabsTrigger value="tenant">Tenant Info</TabsTrigger>
                <TabsTrigger value="documents">Documents</TabsTrigger>
              </TabsList>

              <TabsContent value="overview" className="space-y-6">
                {/* Quick Stats */}
                <div className="grid grid-cols-2 lg:grid-cols-4 gap-6">
                  <Card className="shadow-sm hover:shadow-md transition-shadow">
                    <CardContent className="p-6">
                      <div className="flex items-center space-x-3">
                        <div className="p-3 bg-green-100 dark:bg-green-950/30 rounded-lg">
                          <DollarSign className="h-6 w-6 text-green-600 dark:text-green-400" />
                        </div>
                        <div>
                          <p className="text-sm font-medium text-gray-600 dark:text-gray-400 mb-1">
                            Monthly Rent
                          </p>
                          <p className="text-2xl font-bold text-gray-900 dark:text-gray-100">
                            {formatCurrency(unit.rentAmount)}
                          </p>
                        </div>
                      </div>
                    </CardContent>
                  </Card>

                  <Card className="shadow-sm hover:shadow-md transition-shadow">
                    <CardContent className="p-6">
                      <div className="flex items-center space-x-3">
                        <div className="p-3 bg-blue-100 dark:bg-blue-950/30 rounded-lg">
                          <DollarSign className="h-6 w-6 text-blue-600 dark:text-blue-400" />
                        </div>
                        <div>
                          <p className="text-sm font-medium text-gray-600 dark:text-gray-400 mb-1">
                            Security Deposit
                          </p>
                          <p className="text-2xl font-bold text-gray-900 dark:text-gray-100">
                            {formatCurrency(unit.securityDeposit)}
                          </p>
                        </div>
                      </div>
                    </CardContent>
                  </Card>

                  <Card className="shadow-sm hover:shadow-md transition-shadow">
                    <CardContent className="p-6">
                      <div className="flex items-center space-x-3">
                        <div className="p-3 bg-purple-100 rounded-lg">
                          <Home className="h-6 w-6 text-purple-600" />
                        </div>
                        <div>
                          <p className="text-sm font-medium text-gray-600 mb-1">
                            Square Footage
                          </p>
                          <p className="text-2xl font-bold text-gray-900">
                            {unit.squareFootage.toLocaleString()} ft²
                          </p>
                        </div>
                      </div>
                    </CardContent>
                  </Card>

                  <Card className="shadow-sm hover:shadow-md transition-shadow">
                    <CardContent className="p-6">
                      <div className="flex items-center space-x-3">
                        <div className="p-3 bg-orange-100 rounded-lg">
                          <Home className="h-6 w-6 text-orange-600" />
                        </div>
                        <div>
                          <p className="text-sm font-medium text-gray-600 mb-1">
                            Unit Type
                          </p>
                          <p className="text-2xl font-bold text-gray-900 capitalize">
                            {unit.unitType}
                          </p>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                </div>

                {/* Unit Details Grid */}
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                  {/* Basic Information */}
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <Building2 className="h-5 w-5" />
                        <span>Basic Information</span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div className="grid grid-cols-2 gap-4">
                        <div>
                          <p className="text-sm font-medium text-gray-600">
                            Unit Number
                          </p>
                          <p className="text-lg font-semibold">
                            {unit.unitNumber}
                          </p>
                        </div>
                        <div>
                          <p className="text-sm font-medium text-gray-600">
                            Floor
                          </p>
                          <p className="text-lg font-semibold">
                            {unit.floor || "N/A"}
                          </p>
                        </div>
                        <div>
                          <p className="text-sm font-medium text-gray-600">
                            Bedrooms
                          </p>
                          <p className="text-lg font-semibold flex items-center">
                            <Bed className="h-4 w-4 mr-1" />
                            {unit.bedrooms}
                          </p>
                        </div>
                        <div>
                          <p className="text-sm font-medium text-gray-600">
                            Bathrooms
                          </p>
                          <p className="text-lg font-semibold flex items-center">
                            <Bath className="h-4 w-4 mr-1" />
                            {unit.bathrooms}
                          </p>
                        </div>
                      </div>

                      {unit.availableFrom && (
                        <div>
                          <p className="text-sm font-medium text-gray-600">
                            Available From
                          </p>
                          <p className="text-lg font-semibold flex items-center">
                            <Calendar className="h-4 w-4 mr-1" />
                            {new Date(unit.availableFrom).toLocaleDateString()}
                          </p>
                        </div>
                      )}

                      {unit.lastRenovated && (
                        <div>
                          <p className="text-sm font-medium text-gray-600">
                            Last Renovated
                          </p>
                          <p className="text-lg font-semibold">
                            {new Date(unit.lastRenovated).toLocaleDateString()}
                          </p>
                        </div>
                      )}
                    </CardContent>
                  </Card>

                  {/* Property Information */}
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <MapPin className="h-5 w-5" />
                        <span>Property Information</span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div>
                        <p className="text-sm font-medium text-gray-600">
                          Property Name
                        </p>
                        <p className="text-lg font-semibold">
                          {property?.name}
                        </p>
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-600">
                          Address
                        </p>
                        {property?.address ? (
                          <>
                            <p className="text-lg font-semibold">
                              {property.address.street ||
                                "Address not available"}
                            </p>
                            <p className="text-sm text-gray-600">
                              {[
                                property.address.city,
                                property.address.state,
                                property.address.zipCode,
                              ]
                                .filter(Boolean)
                                .join(", ") || "Location not available"}
                            </p>
                          </>
                        ) : (
                          <p className="text-lg font-semibold text-gray-400">
                            Address not available
                          </p>
                        )}
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-600">
                          Property Type
                        </p>
                        <p className="text-lg font-semibold capitalize">
                          {property?.type}
                        </p>
                      </div>
                    </CardContent>
                  </Card>

                  {/* Additional Details */}
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <FileText className="h-5 w-5" />
                        <span>Additional Details</span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div>
                        <p className="text-sm font-medium text-gray-600">
                          Status
                        </p>
                        <Badge
                          className={`${getStatusColor(unit.status)} mt-1`}
                        >
                          {getStatusIcon(unit.status)}
                          <span className="ml-1">{unit.status}</span>
                        </Badge>
                      </div>

                      {unit.notes && (
                        <div>
                          <p className="text-sm font-medium text-gray-600">
                            Notes
                          </p>
                          <p className="text-sm text-gray-700 mt-1">
                            {unit.notes}
                          </p>
                        </div>
                      )}
                    </CardContent>
                  </Card>
                </div>

                {/* Unit Images */}
                {unit.images && unit.images.length > 0 && (
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <Camera className="h-5 w-5" />
                        <span>Unit Images</span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="grid grid-cols-3 lg:grid-cols-6 gap-4">
                        {unit.images.map((image, index) => (
                          <div
                            key={index}
                            className="relative aspect-video rounded-lg overflow-hidden"
                          >
                            <img
                              src={image}
                              alt={`Unit ${unit.unitNumber} - Image ${
                                index + 1
                              }`}
                              className="w-full h-full object-cover hover:scale-105 transition-transform cursor-pointer"
                            />
                          </div>
                        ))}
                      </div>
                    </CardContent>
                  </Card>
                )}
              </TabsContent>

              <TabsContent value="features" className="space-y-6">
                <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-4 gap-6">
                  {/* Unit Features */}
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <Home className="h-5 w-5" />
                        <span>Unit Features</span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium">Balcony</span>
                        <span
                          className={
                            unit.balcony ? "text-green-600" : "text-gray-400"
                          }
                        >
                          {unit.balcony ? "✓" : "✗"}
                        </span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium">Patio</span>
                        <span
                          className={
                            unit.patio ? "text-green-600" : "text-gray-400"
                          }
                        >
                          {unit.patio ? "✓" : "✗"}
                        </span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium">Garden</span>
                        <span
                          className={
                            unit.garden ? "text-green-600" : "text-gray-400"
                          }
                        >
                          {unit.garden ? "✓" : "✗"}
                        </span>
                      </div>
                    </CardContent>
                  </Card>

                  {/* Parking */}
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <Car className="h-5 w-5" />
                        <span>Parking</span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium">Included</span>
                        <span
                          className={
                            unit.parking.included
                              ? "text-green-600"
                              : "text-gray-400"
                          }
                        >
                          {unit.parking.included ? "✓" : "✗"}
                        </span>
                      </div>
                      {unit.parking.spaces && (
                        <div className="flex items-center justify-between">
                          <span className="text-sm font-medium">Spaces</span>
                          <span className="text-sm font-semibold">
                            {unit.parking.spaces}
                          </span>
                        </div>
                      )}
                      {unit.parking.type && (
                        <div className="flex items-center justify-between">
                          <span className="text-sm font-medium">Type</span>
                          <span className="text-sm font-semibold capitalize">
                            {unit.parking.type}
                          </span>
                        </div>
                      )}
                    </CardContent>
                  </Card>

                  {/* Utilities */}
                  <Card>
                    <CardHeader>
                      <CardTitle className="flex items-center space-x-2">
                        <Droplets className="h-5 w-5" />
                        <span>Utilities</span>
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="grid grid-cols-2 gap-4">
                        {[
                          {
                            key: "electricity",
                            label: "Electricity",
                            icon: Zap,
                          },
                          { key: "water", label: "Water", icon: Droplets },
                          { key: "gas", label: "Gas", icon: Thermometer },
                          { key: "internet", label: "Internet", icon: Wifi },
                          {
                            key: "heating",
                            label: "Heating",
                            icon: Thermometer,
                          },
                          {
                            key: "cooling",
                            label: "Cooling",
                            icon: Thermometer,
                          },
                        ].map(({ key, label, icon: Icon }) => (
                          <div
                            key={key}
                            className="flex items-center space-x-2"
                          >
                            <Icon className="h-4 w-4 text-gray-500" />
                            <span className="text-sm font-medium">{label}</span>
                            <span
                              className={
                                unit.utilities[
                                  key as keyof typeof unit.utilities
                                ]
                                  ? "text-green-600"
                                  : "text-gray-400"
                              }
                            >
                              {unit.utilities[
                                key as keyof typeof unit.utilities
                              ]
                                ? "✓"
                                : "✗"}
                            </span>
                          </div>
                        ))}
                      </div>
                    </CardContent>
                  </Card>
                </div>
              </TabsContent>

              <TabsContent value="tenant" className="space-y-6">
                {unit.status === PropertyStatus.OCCUPIED ? (
                  <div className="grid gap-6 md:grid-cols-2">
                    <Card>
                      <CardHeader>
                        <CardTitle className="flex items-center space-x-2">
                          <User className="h-5 w-5" />
                          <span>Current Tenant</span>
                        </CardTitle>
                      </CardHeader>
                      <CardContent>
                        <div className="text-center py-8">
                          <User className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                          <p className="text-gray-600">
                            Tenant information will be displayed here
                          </p>
                          <p className="text-sm text-gray-500 mt-2">
                            Integration with tenant management system required
                          </p>
                        </div>
                      </CardContent>
                    </Card>

                    <Card>
                      <CardHeader>
                        <CardTitle className="flex items-center space-x-2">
                          <FileText className="h-5 w-5" />
                          <span>Lease Information</span>
                        </CardTitle>
                      </CardHeader>
                      <CardContent>
                        <div className="text-center py-8">
                          <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                          <p className="text-gray-600">
                            Lease details will be displayed here
                          </p>
                          <p className="text-sm text-gray-500 mt-2">
                            Integration with lease management system required
                          </p>
                        </div>
                      </CardContent>
                    </Card>
                  </div>
                ) : (
                  <Card>
                    <CardContent className="text-center py-12">
                      <User className="h-16 w-16 text-gray-300 mx-auto mb-4" />
                      <h3 className="text-lg font-semibold text-gray-600 mb-2">
                        No Current Tenant
                      </h3>
                      <p className="text-gray-500">
                        This unit is currently {unit.status.toLowerCase()}
                      </p>
                    </CardContent>
                  </Card>
                )}
              </TabsContent>

              <TabsContent value="documents" className="space-y-6">
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center space-x-2">
                      <FileText className="h-5 w-5" />
                      <span>Unit Documents</span>
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-center py-12">
                      <FileText className="h-16 w-16 text-gray-300 mx-auto mb-4" />
                      <h3 className="text-lg font-semibold text-gray-600 mb-2">
                        No Documents Available
                      </h3>
                      <p className="text-gray-500 mb-4">
                        Unit documents such as floor plans, inspection reports,
                        and maintenance records will appear here
                      </p>
                      <Button variant="outline">
                        <FileText className="mr-2 h-4 w-4" />
                        Upload Documents
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              </TabsContent>
            </Tabs>
          ) : null}
        </div>
      </div>
    </div>
  );
}
