"use client";

import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Bed,
  Bath,
  Square,
  MapPin,
  Edit,
  Trash2,
  Car,
  Wifi,
  Droplets,
  Zap,
  Calendar,
  User,
  Building,
} from "lucide-react";
import { PropertyStatus } from "@/types";
import { EditUnitDialog } from "./EditUnitDialog";
import UnitDetailsModal from "./UnitDetailsModal";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";

interface Unit {
  _id: string;
  unitNumber: string;
  unitType: "apartment" | "studio" | "penthouse" | "loft" | "room";
  floor?: number;
  bedrooms: number;
  bathrooms: number;
  squareFootage: number;
  rentAmount: number;
  securityDeposit: number;
  status: PropertyStatus;
  balcony?: boolean;
  patio?: boolean;
  garden?: boolean;
  parking?: {
    included: boolean;
    spaces?: number;
    type?: "garage" | "covered" | "open" | "street";
  };
  utilities?: {
    electricity?: boolean;
    water?: boolean;
    gas?: boolean;
    internet?: boolean;
    heating?: boolean;
    cooling?: boolean;
  };
  currentTenantId?: string;
  currentLeaseId?: string;
  availableFrom?: string;
  lastRenovated?: string;
  notes?: string;
  images?: string[];
}

interface EnhancedUnitDisplayProps {
  units: Unit[];
  propertyId: string;
  onDeleteUnit: (unitId: string) => void;
  onAddUnit: () => void;
  onUnitsChange: () => void; // Callback to refresh units after edit
  isLoading?: boolean;
}

const getStatusColor = (status: PropertyStatus) => {
  switch (status) {
    case "available":
      return "bg-green-100 text-green-800 border-green-200";
    case "occupied":
      return "bg-blue-100 text-blue-800 border-blue-200";
    case "maintenance":
      return "bg-yellow-100 text-yellow-800 border-yellow-200";
    case "unavailable":
      return "bg-red-100 text-red-800 border-red-200";
    default:
      return "bg-gray-100 text-gray-800 border-gray-200";
  }
};

const getStatusIcon = (status: PropertyStatus) => {
  switch (status) {
    case "available":
      return "🟢";
    case "occupied":
      return "🔵";
    case "maintenance":
      return "🟡";
    case "unavailable":
      return "🔴";
    default:
      return "⚪";
  }
};

export function EnhancedUnitDisplay({
  units,
  propertyId,
  onDeleteUnit,
  onAddUnit,
  onUnitsChange,
  isLoading = false,
}: EnhancedUnitDisplayProps) {
  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [typeFilter, setTypeFilter] = useState<string>("all");
  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [selectedUnit, setSelectedUnit] = useState<Unit | null>(null);
  const [detailsModalOpen, setDetailsModalOpen] = useState(false);
  const [selectedUnitForDetails, setSelectedUnitForDetails] =
    useState<Unit | null>(null);

  const handleEditUnit = (unit: Unit) => {
    setSelectedUnit(unit);
    setEditDialogOpen(true);
  };

  const handleViewUnitDetails = (unit: Unit) => {
    setSelectedUnitForDetails(unit);
    setDetailsModalOpen(true);
  };

  const handleUnitUpdated = () => {
    onUnitsChange();
    setEditDialogOpen(false);
    setSelectedUnit(null);
  };

  // Filter units based on search and filters
  const filteredUnits = units.filter((unit) => {
    const matchesSearch = unit.unitNumber
      .toLowerCase()
      .includes(searchTerm.toLowerCase());
    const matchesStatus =
      statusFilter === "all" || unit.status === statusFilter;
    const matchesType = typeFilter === "all" || unit.unitType === typeFilter;

    return matchesSearch && matchesStatus && matchesType;
  });

  const unitTypes = Array.from(new Set(units.map((unit) => unit.unitType)));

  return (
    <div className="space-y-6">
      {/* Header with Search and Filters */}
      <Card className="border-blue-100 p-2">
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center">
              <Building className="h-5 w-5 mr-2 text-blue-600" />
              Unit Management ({filteredUnits.length} of {units.length} units)
            </CardTitle>
            <Button
              onClick={onAddUnit}
              className="bg-blue-600 hover:bg-blue-700"
            >
              <Building className="h-4 w-4 mr-2" />
              Add Unit
            </Button>
          </div>
        </CardHeader>
      </Card>

      {/* Units Grid */}
      {filteredUnits.length === 0 ? (
        <Card>
          <CardContent className="p-12 text-center">
            <Building className="h-12 w-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-900 mb-2">
              {units.length === 0 ? "No Units Added" : "No Units Found"}
            </h3>
            <p className="text-gray-600 mb-6">
              {units.length === 0
                ? "Start by adding your first unit to this property."
                : "Try adjusting your search or filter criteria."}
            </p>
            {units.length === 0 && (
              <Button
                onClick={onAddUnit}
                className="bg-blue-600 hover:bg-blue-700"
              >
                <Building className="h-4 w-4 mr-2" />
                Add First Unit
              </Button>
            )}
          </CardContent>
        </Card>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {filteredUnits
            .filter((unit) => unit && unit.unitNumber)
            .map((unit) => (
              <Card
                key={unit._id}
                className="border-gray-200 hover:border-blue-300 hover:shadow-lg transition-all duration-200 cursor-pointer"
                onClick={() => handleViewUnitDetails(unit)}
              >
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-2">
                      <span className="text-2xl">
                        {getStatusIcon(unit.status)}
                      </span>
                      <div>
                        <h3 className="text-lg font-bold text-gray-900">
                          Unit {unit.unitNumber}
                        </h3>
                        {unit.floor !== undefined && (
                          <p className="text-sm text-gray-600 flex items-center">
                            <MapPin className="h-3 w-3 mr-1" />
                            Floor {unit.floor}
                          </p>
                        )}
                      </div>
                    </div>
                    <Badge className={getStatusColor(unit.status)}>
                      {unit.status.charAt(0).toUpperCase() +
                        unit.status.slice(1)}
                    </Badge>
                  </div>
                </CardHeader>

                <CardContent className="space-y-4">
                  {/* Unit Type */}
                  <div className="flex items-center justify-between">
                    <Badge variant="outline" className="capitalize">
                      {unit.unitType}
                    </Badge>
                    <div className="text-right">
                      <div className="text-2xl font-bold text-blue-600">
                        ${unit.rentAmount.toLocaleString()}
                      </div>
                      <div className="text-xs text-gray-600">per month</div>
                    </div>
                  </div>

                  {/* Unit Details */}
                  <div className="grid grid-cols-3 gap-3 py-3 border-t border-gray-100">
                    <div className="text-center">
                      <div className="flex items-center justify-center mb-1">
                        <Bed className="h-4 w-4 text-blue-600" />
                      </div>
                      <div className="text-lg font-semibold">
                        {unit.bedrooms}
                      </div>
                      <div className="text-xs text-gray-600">Beds</div>
                    </div>
                    <div className="text-center">
                      <div className="flex items-center justify-center mb-1">
                        <Bath className="h-4 w-4 text-green-600" />
                      </div>
                      <div className="text-lg font-semibold">
                        {unit.bathrooms}
                      </div>
                      <div className="text-xs text-gray-600">Baths</div>
                    </div>
                    <div className="text-center">
                      <div className="flex items-center justify-center mb-1">
                        <Square className="h-4 w-4 text-purple-600" />
                      </div>
                      <div className="text-lg font-semibold">
                        {unit.squareFootage.toLocaleString()}
                      </div>
                      <div className="text-xs text-gray-600">Sq Ft</div>
                    </div>
                  </div>

                  {/* Features */}
                  <div className="space-y-2">
                    <div className="flex flex-wrap gap-1">
                      {unit.balcony && (
                        <Badge variant="secondary" className="text-xs">
                          Balcony
                        </Badge>
                      )}
                      {unit.patio && (
                        <Badge variant="secondary" className="text-xs">
                          Patio
                        </Badge>
                      )}
                      {unit.garden && (
                        <Badge variant="secondary" className="text-xs">
                          Garden
                        </Badge>
                      )}
                      {unit.parking?.included && (
                        <Badge variant="secondary" className="text-xs">
                          <Car className="h-3 w-3 mr-1" />
                          Parking
                        </Badge>
                      )}
                    </div>

                    {/* Utilities */}
                    <div className="flex flex-wrap gap-1">
                      {unit.utilities?.internet && (
                        <Badge variant="outline" className="text-xs">
                          <Wifi className="h-3 w-3 mr-1" />
                          Internet
                        </Badge>
                      )}
                      {unit.utilities?.water && (
                        <Badge variant="outline" className="text-xs">
                          <Droplets className="h-3 w-3 mr-1" />
                          Water
                        </Badge>
                      )}
                      {unit.utilities?.electricity && (
                        <Badge variant="outline" className="text-xs">
                          <Zap className="h-3 w-3 mr-1" />
                          Electric
                        </Badge>
                      )}
                    </div>
                  </div>

                  {/* Tenant Info */}
                  {unit.status === "occupied" && unit.currentTenantId && (
                    <div className="bg-blue-50 p-3 rounded-lg border border-blue-100">
                      <div className="flex items-center text-sm text-blue-700">
                        <User className="h-4 w-4 mr-2" />
                        <span className="font-medium">Occupied</span>
                      </div>
                      {unit.currentLeaseId && (
                        <div className="text-xs text-blue-600 mt-1">
                          Active lease on file
                        </div>
                      )}
                    </div>
                  )}

                  {/* Available Date */}
                  {unit.status === "available" && unit.availableFrom && (
                    <div className="bg-green-50 p-3 rounded-lg border border-green-100">
                      <div className="flex items-center text-sm text-green-700">
                        <Calendar className="h-4 w-4 mr-2" />
                        <span className="font-medium">
                          Available from{" "}
                          {new Date(unit.availableFrom).toLocaleDateString()}
                        </span>
                      </div>
                    </div>
                  )}

                  {/* Actions */}
                  <div className="flex gap-2 pt-3 border-t border-gray-100">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={(e) => {
                        e.stopPropagation();
                        handleEditUnit(unit);
                      }}
                      className="flex-1"
                    >
                      <Edit className="h-4 w-4 mr-2" />
                      Edit
                    </Button>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={(e) => {
                        e.stopPropagation();
                        onDeleteUnit(unit._id);
                      }}
                      className="text-red-600 border-red-200 hover:bg-red-50"
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
        </div>
      )}

      {/* Edit Unit Dialog */}
      <EditUnitDialog
        open={editDialogOpen}
        onOpenChange={setEditDialogOpen}
        propertyId={propertyId}
        unit={selectedUnit}
        onUnitUpdated={handleUnitUpdated}
      />

      {/* Unit Details Modal */}
      {selectedUnitForDetails && (
        <UnitDetailsModal
          open={detailsModalOpen}
          onOpenChange={setDetailsModalOpen}
          propertyId={propertyId}
          unitId={selectedUnitForDetails._id}
          onUnitUpdated={onUnitsChange}
          onUnitDeleted={() => {
            onDeleteUnit(selectedUnitForDetails._id);
            setDetailsModalOpen(false);
            setSelectedUnitForDetails(null);
          }}
        />
      )}
    </div>
  );
}
