"use client";

import { useState, useEffect } from "react";
import { useForm, Controller } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { Upload, User, Mail, Phone, Calendar, MapPin } from "lucide-react";
import { isValidPhoneNumber } from "@/lib/utils";
import { logClientError, logClientWarn } from "@/utils/logger";

const profileSchema = z.object({
  firstName: z
    .string()
    .min(1, "First name is required")
    .max(50, "First name too long"),
  lastName: z
    .string()
    .min(1, "Last name is required")
    .max(50, "Last name too long"),
  email: z.string().email("Invalid email address"),
  phone: z
    .string()
    .optional()
    .refine(
      (phone) => {
        if (!phone || phone.trim() === "") return true;
        return isValidPhoneNumber(phone);
      },
      {
        message: "Invalid phone number format",
      }
    ),
  bio: z.string().max(500, "Bio too long").optional(),
  location: z.string().max(100, "Location too long").optional(),
  city: z.string().max(50, "City too long").optional(),
  website: z
    .string()
    .optional()
    .refine(
      (website) => {
        if (!website || website.trim() === "") return true;
        try {
          new URL(website);
          return true;
        } catch {
          return false;
        }
      },
      {
        message: "Invalid website URL format",
      }
    ),
  address: z.string().max(200, "Address too long").optional(),
  jobTitle: z.string().max(100, "Job title too long").optional(),
  company: z.string().max(100, "Company name too long").optional(),
  dateOfBirth: z.string().optional(),
  gender: z.enum(["male", "female", "other", "prefer_not_to_say"]).optional(),
  emergencyContact: z
    .object({
      name: z.string().max(100, "Emergency contact name too long").optional(),
      phone: z.string().optional(),
      relationship: z.string().max(50, "Relationship too long").optional(),
    })
    .optional(),
  socialLinks: z
    .object({
      linkedin: z
        .string()
        .url("Invalid LinkedIn URL")
        .optional()
        .or(z.literal("")),
      twitter: z
        .string()
        .url("Invalid Twitter URL")
        .optional()
        .or(z.literal("")),
      facebook: z
        .string()
        .url("Invalid Facebook URL")
        .optional()
        .or(z.literal("")),
      instagram: z
        .string()
        .url("Invalid Instagram URL")
        .optional()
        .or(z.literal("")),
    })
    .optional(),
  preferences: z
    .object({
      preferredContactMethod: z.enum(["email", "phone", "sms"]).optional(),
      language: z.string().optional(),
      timezone: z.string().optional(),
    })
    .optional(),
});

type ProfileFormData = z.infer<typeof profileSchema>;

interface ProfileSettingsProps {
  user: any;
  onUpdate: (data: any) => void;
  onAlert: (type: "success" | "error" | "info", message: string) => void;
}

export function ProfileSettings({
  user,
  onUpdate,
  onAlert,
}: ProfileSettingsProps) {
  const [isLoading, setIsLoading] = useState(false);
  const [isUploading, setIsUploading] = useState(false);
  const [avatarKey, setAvatarKey] = useState(Date.now());

  // Show loading state if user data is not available
  if (!user) {
    return (
      <div className="flex h-64 items-center justify-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
      </div>
    );
  }

  // Helper function to format date for HTML date input
  const formatDateForInput = (date: any): string => {
    if (!date) return "";

    try {
      // Handle different date formats
      let dateObj: Date;

      if (date instanceof Date) {
        dateObj = date;
      } else if (typeof date === "string") {
        dateObj = new Date(date);
      } else {
        return "";
      }

      // Check if date is valid
      if (isNaN(dateObj.getTime())) {
        return "";
      }

      // Format as YYYY-MM-DD for HTML date input
      return dateObj.toISOString().split("T")[0];
    } catch (error) {
      logClientWarn("Error formatting date:", error);
      return "";
    }
  };

  // Create default values from user data
  const getDefaultValues = (userData: any): ProfileFormData => ({
    firstName: userData?.firstName || "",
    lastName: userData?.lastName || "",
    email: userData?.email || "",
    phone: userData?.phone || "",
    bio: userData?.bio || "",
    location: userData?.location || "",
    city: userData?.city || "",
    website: userData?.website || "",
    address: userData?.address || "",
    jobTitle: userData?.jobTitle || "",
    company: userData?.company || "",
    dateOfBirth: formatDateForInput(userData?.dateOfBirth),
    gender: userData?.gender || undefined,
    emergencyContact: {
      name: userData?.emergencyContact?.name || "",
      phone: userData?.emergencyContact?.phone || "",
      relationship: userData?.emergencyContact?.relationship || "",
    },
    socialLinks: {
      linkedin: userData?.socialLinks?.linkedin || "",
      twitter: userData?.socialLinks?.twitter || "",
      facebook: userData?.socialLinks?.facebook || "",
      instagram: userData?.socialLinks?.instagram || "",
    },
    preferences: {
      preferredContactMethod:
        userData?.preferences?.preferredContactMethod || "email",
      language: userData?.preferences?.language || "en",
      timezone: userData?.preferences?.timezone || "America/New_York",
    },
  });

  const form = useForm<ProfileFormData>({
    resolver: zodResolver(profileSchema),
    defaultValues: getDefaultValues(user),
    mode: "onChange",
  });

  // Update form when user data changes
  useEffect(() => {
    if (user && Object.keys(user).length > 0) {
      const formData = getDefaultValues(user);

      // Use setTimeout to ensure form reset happens after component is fully mounted
      setTimeout(() => {
        // Reset form first
        form.reset(formData);

        // Then set individual values to ensure they're properly set
        Object.entries(formData).forEach(([key, value]) => {
          if (value !== undefined && value !== null && value !== "") {
            form.setValue(key as keyof ProfileFormData, value, {
              shouldValidate: false,
              shouldDirty: false,
              shouldTouch: false,
            });
          }
        });

        form.clearErrors();
      }, 100); // Increased timeout to ensure proper mounting
    }
  }, [user, form]);

  const onSubmit = async (data: ProfileFormData) => {
    try {
      setIsLoading(true);

      // Helper function to validate and clean URL
      const cleanUrl = (url?: string) => {
        if (!url || url.trim() === "") return "";
        const trimmed = url.trim();
        // If it doesn't start with http/https, add https://
        if (
          trimmed &&
          !trimmed.startsWith("http://") &&
          !trimmed.startsWith("https://")
        ) {
          return `https://${trimmed}`;
        }
        return trimmed;
      };

      // Prepare data for the profile settings API
      const profileData = {
        firstName: data.firstName?.trim(),
        lastName: data.lastName?.trim(),
        email: data.email?.trim(),
        phone: data.phone?.trim() || undefined,
        bio: data.bio?.trim() || undefined,
        location: data.location?.trim() || undefined,
        city: data.city?.trim() || undefined,
        website: cleanUrl(data.website) || undefined,
        address: data.address?.trim() || undefined,
        jobTitle: data.jobTitle?.trim() || undefined,
        company: data.company?.trim() || undefined,
        // Convert dateOfBirth to ISO string if provided
        dateOfBirth: data.dateOfBirth?.trim()
          ? new Date(data.dateOfBirth.trim()).toISOString()
          : undefined,
        gender: data.gender || undefined,
        // Keep emergencyContact as object (ProfileSettings model expects this format)
        emergencyContact: data.emergencyContact?.name?.trim()
          ? {
              name: data.emergencyContact.name.trim(),
              phone: data.emergencyContact.phone?.trim() || "",
              relationship: data.emergencyContact.relationship?.trim() || "",
            }
          : undefined,
        // Clean social links URLs
        socialLinks: {
          linkedin: cleanUrl(data.socialLinks?.linkedin) || undefined,
          twitter: cleanUrl(data.socialLinks?.twitter) || undefined,
          facebook: cleanUrl(data.socialLinks?.facebook) || undefined,
          instagram: cleanUrl(data.socialLinks?.instagram) || undefined,
        },
        preferences: data.preferences,
      };

      // Remove undefined values and empty objects to avoid validation issues
      const cleanData = Object.fromEntries(
        Object.entries(profileData).filter(([_, value]) => {
          if (value === undefined || value === null || value === "")
            return false;
          if (typeof value === "object" && value !== null) {
            // For objects, check if they have any non-undefined values
            const hasValues = Object.values(value).some(
              (v) => v !== undefined && v !== null && v !== ""
            );
            return hasValues;
          }
          return true;
        })
      );

      const response = await fetch("/api/settings/profile", {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(cleanData),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "Failed to update profile");
      }

      const result = await response.json();

      const updatedSettings =
        result?.data?.settings ?? result?.settings ?? result?.data ?? result;

      if (!updatedSettings || typeof updatedSettings !== "object") {
        throw new Error("Unexpected response from profile update");
      }

      const mergedUserData = { ...user, ...updatedSettings };

      form.reset(getDefaultValues(mergedUserData));

      onUpdate(mergedUserData);
      onAlert("success", "Profile updated successfully");
    } catch (error) {
      onAlert(
        "error",
        error instanceof Error ? error.message : "Failed to update profile"
      );
    } finally {
      setIsLoading(false);
    }
  };

  const handleAvatarUpload = async (
    event: React.ChangeEvent<HTMLInputElement>
  ) => {
    const file = event.target.files?.[0];
    if (!file) return;

    // Validate file type
    if (!file.type.startsWith("image/")) {
      onAlert("error", "Please select an image file");
      return;
    }

    // Validate file size (5MB max)
    if (file.size > 5 * 1024 * 1024) {
      onAlert("error", "Image size must be less than 5MB");
      return;
    }

    try {
      setIsUploading(true);

      // Upload to R2 first
      const formData = new FormData();
      formData.append("files", file);
      formData.append("folder", "PropertyPro/avatars");
      formData.append("quality", "85");
      formData.append("maxWidth", "400");
      formData.append("maxHeight", "400");

      const uploadResponse = await fetch("/api/upload/images", {
        method: "POST",
        body: formData,
        credentials: "include",
      });

      const uploadResult = await uploadResponse.json();

      if (!uploadResponse.ok || !uploadResult?.success) {
        const errorMessage =
          uploadResult?.error ||
          uploadResult?.details?.join(", ") ||
          "Failed to upload image";
        console.error("Upload error:", uploadResult);
        throw new Error(errorMessage);
      }

      if (!uploadResult?.images?.[0]?.url) {
        throw new Error("Invalid upload response from R2");
      }

      const avatarUrl = uploadResult.images[0].url;

      // Update user avatar with R2 URL
      const updateResponse = await fetch("/api/user/avatar", {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ avatar: avatarUrl }),
      });

      if (!updateResponse.ok) {
        const errorText = await updateResponse.text();
        logClientError(
          "Avatar update error:",
          updateResponse.status,
          errorText
        );
        let errorData;
        try {
          errorData = JSON.parse(errorText);
        } catch {
          errorData = { error: errorText };
        }
        throw new Error(
          errorData.error || `Failed to update avatar: ${updateResponse.status}`
        );
      }

      const result = await updateResponse.json();

      // Validate response structure
      if (!result?.success || !result?.data?.user) {
        throw new Error("Invalid response from server");
      }

      // Update avatar key to force re-render
      setAvatarKey(Date.now());

      // Pass the complete updated user object to onUpdate
      onUpdate(result.data.user);
      onAlert("success", "Profile picture updated successfully");
    } catch (error) {
      onAlert(
        "error",
        error instanceof Error
          ? error.message
          : "Failed to upload profile picture"
      );
    } finally {
      setIsUploading(false);
    }
  };

  return (
    <div className="space-y-6">
      {/* Avatar Section */}
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">Profile Picture</CardTitle>
          <CardDescription>
            Upload a profile picture to personalize your account
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex items-center gap-6">
            <Avatar className="h-20 w-20" key={avatarKey}>
              <AvatarImage
                src={user?.avatar || user?.image || ""}
                alt={user?.name || `${user?.firstName} ${user?.lastName}`}
              />
              <AvatarFallback className="text-lg">
                {user?.firstName?.[0] || "N"}
                {user?.lastName?.[0] || "S"}
              </AvatarFallback>
            </Avatar>
            <div className="space-y-2">
              <Label htmlFor="avatar-upload" className="cursor-pointer">
                <Button
                  variant="outline"
                  disabled={isUploading}
                  className="cursor-pointer"
                  asChild
                >
                  <span>
                    <Upload className="h-4 w-4 mr-2" />
                    {isUploading ? "Uploading..." : "Upload Photo"}
                  </span>
                </Button>
              </Label>
              <input
                id="avatar-upload"
                type="file"
                accept="image/*"
                onChange={handleAvatarUpload}
                className="hidden"
              />
              <p className="text-sm text-muted-foreground">
                JPG, PNG or GIF. Max size 5MB.
              </p>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Single Unified Form */}
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
        {/* Basic Information */}
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Basic Information</CardTitle>
            <CardDescription>
              Update your personal information and contact details
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="firstName">First Name</Label>
                <Input
                  id="firstName"
                  {...form.register("firstName")}
                  error={form.formState.errors.firstName?.message}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="lastName">Last Name</Label>
                <Input
                  id="lastName"
                  {...form.register("lastName")}
                  error={form.formState.errors.lastName?.message}
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="email">Email Address</Label>
              <Input
                id="email"
                type="email"
                {...form.register("email")}
                error={form.formState.errors.email?.message}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="phone">Phone Number</Label>
              <Input
                id="phone"
                type="tel"
                placeholder="+1 (555) 123-4567"
                {...form.register("phone")}
                error={form.formState.errors.phone?.message}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="bio">Bio</Label>
              <Textarea
                id="bio"
                placeholder="Tell us about yourself..."
                {...form.register("bio")}
                error={form.formState.errors.bio?.message}
              />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="location">Location</Label>
                <Input
                  id="location"
                  placeholder="State, Country"
                  {...form.register("location")}
                  error={form.formState.errors.location?.message}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="city">City</Label>
                <Input
                  id="city"
                  placeholder="Your city"
                  {...form.register("city")}
                  error={form.formState.errors.city?.message}
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="website">Website</Label>
              <Input
                id="website"
                type="url"
                placeholder="https://example.com"
                {...form.register("website")}
                error={form.formState.errors.website?.message}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="address">Address</Label>
              <Controller
                name="address"
                control={form.control}
                render={({ field }) => (
                  <Input
                    id="address"
                    placeholder="Your full address"
                    {...field}
                    value={field.value || ""}
                    error={form.formState.errors.address?.message}
                  />
                )}
              />
            </div>

            {/* Professional Information */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="jobTitle">Job Title</Label>
                <Input
                  id="jobTitle"
                  placeholder="Your job title"
                  {...form.register("jobTitle")}
                  error={form.formState.errors.jobTitle?.message}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="company">Company</Label>
                <Input
                  id="company"
                  placeholder="Your company"
                  {...form.register("company")}
                  error={form.formState.errors.company?.message}
                />
              </div>
            </div>

            {/* Personal Information */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="dateOfBirth">Date of Birth</Label>
                <Input
                  id="dateOfBirth"
                  type="date"
                  {...form.register("dateOfBirth")}
                  error={form.formState.errors.dateOfBirth?.message}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="gender">Gender</Label>
                <Controller
                  name="gender"
                  control={form.control}
                  render={({ field }) => (
                    <select
                      id="gender"
                      {...field}
                      className="flex h-10 w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background file:border-0 file:bg-transparent file:text-sm file:font-medium placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:cursor-not-allowed disabled:opacity-50"
                    >
                      <option value="">Select gender</option>
                      <option value="male">Male</option>
                      <option value="female">Female</option>
                      <option value="other">Other</option>
                      <option value="prefer_not_to_say">
                        Prefer not to say
                      </option>
                    </select>
                  )}
                />
              </div>
            </div>

            <Separator />
          </CardContent>
        </Card>

        {/* Emergency Contact */}
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Emergency Contact</CardTitle>
            <CardDescription>
              Provide emergency contact information for safety purposes
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="emergencyContact.name">Contact Name</Label>
                <Input
                  id="emergencyContact.name"
                  placeholder="Emergency contact name"
                  {...form.register("emergencyContact.name")}
                  error={form.formState.errors.emergencyContact?.name?.message}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="emergencyContact.relationship">
                  Relationship
                </Label>
                <Input
                  id="emergencyContact.relationship"
                  placeholder="e.g., Spouse, Parent, Sibling"
                  {...form.register("emergencyContact.relationship")}
                  error={
                    form.formState.errors.emergencyContact?.relationship
                      ?.message
                  }
                />
              </div>
            </div>
            <div className="space-y-2">
              <Label htmlFor="emergencyContact.phone">Contact Phone</Label>
              <Input
                id="emergencyContact.phone"
                type="tel"
                placeholder="+1 (555) 123-4567"
                {...form.register("emergencyContact.phone")}
                error={form.formState.errors.emergencyContact?.phone?.message}
              />
            </div>
          </CardContent>
        </Card>

        {/* Social Links */}
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Social Links</CardTitle>
            <CardDescription>
              Connect your social media profiles
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="socialLinks.linkedin">LinkedIn</Label>
                <Input
                  id="socialLinks.linkedin"
                  type="url"
                  placeholder="https://linkedin.com/in/username"
                  {...form.register("socialLinks.linkedin")}
                  error={form.formState.errors.socialLinks?.linkedin?.message}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="socialLinks.twitter">Twitter</Label>
                <Input
                  id="socialLinks.twitter"
                  type="url"
                  placeholder="https://twitter.com/username"
                  {...form.register("socialLinks.twitter")}
                  error={form.formState.errors.socialLinks?.twitter?.message}
                />
              </div>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="socialLinks.facebook">Facebook</Label>
                <Input
                  id="socialLinks.facebook"
                  type="url"
                  placeholder="https://facebook.com/username"
                  {...form.register("socialLinks.facebook")}
                  error={form.formState.errors.socialLinks?.facebook?.message}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="socialLinks.instagram">Instagram</Label>
                <Input
                  id="socialLinks.instagram"
                  type="url"
                  placeholder="https://instagram.com/username"
                  {...form.register("socialLinks.instagram")}
                  error={form.formState.errors.socialLinks?.instagram?.message}
                />
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Form Actions */}
        <div className="flex justify-end gap-3">
          <Button
            type="button"
            variant="outline"
            onClick={() => form.reset()}
            disabled={isLoading}
          >
            Reset
          </Button>
          <Button type="submit" disabled={isLoading}>
            {isLoading ? "Saving..." : "Save Changes"}
          </Button>
        </div>
      </form>

      {/* Account Information */}
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">Account Information</CardTitle>
          <CardDescription>
            View your account details and role information
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="flex items-center gap-3">
              <User className="h-4 w-4 text-muted-foreground" />
              <div>
                <p className="text-sm font-medium">Role</p>
                <Badge variant="secondary" className="mt-1">
                  {user?.role?.replace("_", " ").toUpperCase()}
                </Badge>
              </div>
            </div>
            <div className="flex items-center gap-3">
              <Calendar className="h-4 w-4 text-muted-foreground" />
              <div>
                <p className="text-sm font-medium">Member Since</p>
                <p className="text-sm text-muted-foreground">
                  {user?.createdAt
                    ? new Date(user.createdAt).toLocaleDateString()
                    : "N/A"}
                </p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
