"use client";

import { z } from "zod";
import { toast } from "sonner";
import { useForm } from "react-hook-form";
import { useRouter } from "next/navigation";
import React, { useState, useEffect } from "react";
import { zodResolver } from "@hookform/resolvers/zod";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Loader2, Settings, AlertTriangle } from "lucide-react";

const MAINTENANCE_CATEGORIES = [
  "Plumbing",
  "Electrical",
  "HVAC",
  "Appliances",
  "Flooring",
  "Painting",
  "Roofing",
  "Windows",
  "Doors",
  "Landscaping",
  "Pest Control",
  "Cleaning",
  "Security",
  "General Repair",
  "Emergency",
  "Other",
 ] as const;

// Validation schema for tenant maintenance requests
const tenantMaintenanceSchema = z.object({
  title: z.string().min(1, "Title is required").max(100, "Title too long"),
  description: z.string().min(10, "Description must be at least 10 characters"),
  category: z.enum(MAINTENANCE_CATEGORIES),
  priority: z.enum(["low", "medium", "high", "emergency"]),
  leaseId: z.string().min(1, "Please select a lease"),
  unitId: z.string().optional(),
  contactPhone: z
    .string()
    .trim()
    .optional()
    .refine(
      (value) => !value || value.length >= 7,
      "Contact phone must be at least 7 characters"
    )
    .refine(
      (value) => !value || value.length <= 20,
      "Contact phone cannot exceed 20 characters"
    ),
});

type TenantMaintenanceFormData = z.infer<typeof tenantMaintenanceSchema>;

interface Lease {
  _id: string;
  propertyId: {
    _id: string;
    name: string;
    address: string;
    units?: Array<{
      _id: string;
      unitNumber: string;
      unitType: string;
    }>;
  };
  unitId?: string;
  startDate: string;
  endDate: string;
  status: string;
}

interface TenantMaintenanceRequestFormProps {
  onSuccess?: () => void;
  onCancel?: () => void;
  isLoading?: boolean;
}

export default function TenantMaintenanceRequestForm({
  onSuccess,
  onCancel,
  isLoading = false,
}: TenantMaintenanceRequestFormProps) {
  const router = useRouter();
  const [leases, setLeases] = useState<Lease[]>([]);
  const [loadingLeases, setLoadingLeases] = useState(true);
  const [submitting, setSubmitting] = useState(false);
  const effectiveSubmitting = submitting || isLoading;

  const form = useForm<TenantMaintenanceFormData>({
    resolver: zodResolver(tenantMaintenanceSchema),
    defaultValues: {
      title: "",
      description: "",
      category: "Other",
      priority: "medium",
      leaseId: "",
      unitId: "",
      contactPhone: "",
    },
  });

  const watchedLeaseId = form.watch("leaseId");
  const selectedLease = leases.find((lease) => lease._id === watchedLeaseId);

  // Format address helper function
  const formatAddress = (address: any) => {
    if (!address) return "Address not available";
    if (typeof address === "string") return address;
    if (typeof address === "object") {
      const { street, city, state, zipCode } = address;
      return `${street || ""}, ${city || ""}, ${state || ""} ${zipCode || ""}`
        .replace(/,\s*,/g, ",")
        .replace(/^\s*,\s*|\s*,\s*$/g, "");
    }
    return "Address not available";
  };

  // Fetch tenant's leases
  useEffect(() => {
    fetchTenantLeases();
  }, []);

  const fetchTenantLeases = async () => {
    try {
      setLoadingLeases(true);
      const response = await fetch("/api/tenant/dashboard");
      const data = await response.json();

      if (data.success) {
        setLeases(data.data.allLeases || []);

        // Auto-select the current lease if there's only one
        if (data.data.allLeases?.length === 1) {
          form.setValue("leaseId", data.data.allLeases[0]._id);
        }
      } else {
        toast.error("Failed to fetch your leases");
      }
    } catch (error) {
      console.error("Error fetching leases:", error);
      toast.error("Failed to fetch your leases");
    } finally {
      setLoadingLeases(false);
    }
  };

  const handleSubmit = async (data: TenantMaintenanceFormData) => {
    toast.info("Processing your maintenance request...");

    try {
      setSubmitting(true);

      const selectedLease = leases.find((lease) => lease._id === data.leaseId);
      if (!selectedLease) {
        toast.error("Please select a valid lease");
        return;
      }

      const contactPhone = data.contactPhone?.trim() || undefined;

      const normalizeId = (value: unknown): string | undefined => {
        if (!value || value === "") return undefined;
        if (typeof value === "string") return value;
        if (typeof value === "object") {
          const objectValue = value as { _id?: unknown; toString?: () => string };
          if (typeof objectValue._id === "string") {
            return objectValue._id;
          }
          if (
            objectValue._id &&
            typeof (objectValue._id as { toString?: () => string }).toString === "function"
          ) {
            return (objectValue._id as { toString: () => string }).toString();
          }
        }
        return undefined;
      };

      const resolvedPropertyId = normalizeId(selectedLease.propertyId);
      const resolvedUnitId = normalizeId(data.unitId || selectedLease.unitId);

      const requestBody: Record<string, unknown> = {
        title: data.title,
        description: data.description,
        category: data.category,
        priority: data.priority,
        leaseId: data.leaseId,
        ...(resolvedPropertyId ? { propertyId: resolvedPropertyId } : {}),
        ...(resolvedUnitId ? { unitId: resolvedUnitId } : {}),
        ...(contactPhone ? { contactPhone } : {}),
      };

      const response = await fetch("/api/tenant/maintenance", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(requestBody),
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`HTTP ${response.status}: ${errorText}`);
      }

      await response.json();

      toast.success("Maintenance request submitted successfully!");
      form.reset({
        title: "",
        description: "",
        category: data.category,
        priority: data.priority,
        leaseId: data.leaseId,
        unitId: "",
        contactPhone: "",
      });

      if (onSuccess) {
        onSuccess();
      } else {
        router.push("/dashboard/maintenance/my-requests");
      }
    } catch (error) {
      toast.error(
        error instanceof Error ? error.message : "Failed to submit request"
      );
    } finally {
      setSubmitting(false);
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case "emergency":
        return "destructive";
      case "high":
        return "destructive";
      case "medium":
        return "default";
      case "low":
        return "secondary";
      default:
        return "default";
    }
  };

  if (loadingLeases) {
    return (
      <Card>
        <CardContent className="flex items-center justify-center py-8">
          <Loader2 className="h-8 w-8 animate-spin" />
          <span className="ml-2">Loading your leases...</span>
        </CardContent>
      </Card>
    );
  }

  if (leases.length === 0) {
    return (
      <Card>
        <CardContent className="text-center py-8">
          <AlertTriangle className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
          <h3 className="text-lg font-semibold mb-2">No Active Leases</h3>
          <p className="text-muted-foreground">
            You need an active lease to submit maintenance requests.
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Settings className="h-5 w-5" />
          Submit Maintenance Request
        </CardTitle>
        <CardDescription>
          Report maintenance issues for your property
        </CardDescription>
      </CardHeader>
      <CardContent>
        <Form {...form}>
          <form
            onSubmit={form.handleSubmit(handleSubmit)}
            className="space-y-6"
          >
            {/* Lease Selection */}
            <FormField
              control={form.control}
              name="leaseId"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Property/Lease</FormLabel>
                  <Select onValueChange={field.onChange} value={field.value}>
                    <FormControl>
                      <SelectTrigger>
                        <SelectValue placeholder="Select your lease" />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      {leases.map((lease) => (
                        <SelectItem key={lease._id} value={lease._id}>
                          <div className="flex flex-col">
                            <span className="font-medium">
                              {lease.propertyId.name}
                            </span>
                            <span className="text-sm text-muted-foreground">
                              {formatAddress(lease.propertyId.address)}
                            </span>
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* Unit Selection (if property has multiple units) */}
            {selectedLease?.propertyId?.units &&
              selectedLease.propertyId.units.length > 1 && (
                <FormField
                  control={form.control}
                  name="unitId"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Unit</FormLabel>
                      <Select
                        onValueChange={field.onChange}
                        value={field.value}
                      >
                        <FormControl>
                          <SelectTrigger>
                            <SelectValue placeholder="Select unit" />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          {selectedLease.propertyId.units?.map((unit) => (
                            <SelectItem key={unit._id} value={unit._id}>
                              Unit {unit.unitNumber} ({unit.unitType})
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              )}

            {/* Title */}
            <FormField
              control={form.control}
              name="title"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Title</FormLabel>
                  <FormControl>
                    <Input
                      placeholder="Brief description of the issue"
                      {...field}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* Category and Priority */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <FormField
                control={form.control}
                name="category"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Category</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value}>
                      <FormControl>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {MAINTENANCE_CATEGORIES.map((category) => (
                          <SelectItem key={category} value={category}>
                            {category}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="priority"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Priority</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value}>
                      <FormControl>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="low">
                          <div className="flex items-center gap-2">
                            <Badge variant="secondary">Low</Badge>
                            <span>Can wait a few days</span>
                          </div>
                        </SelectItem>
                        <SelectItem value="medium">
                          <div className="flex items-center gap-2">
                            <Badge variant="default">Medium</Badge>
                            <span>Should be addressed soon</span>
                          </div>
                        </SelectItem>
                        <SelectItem value="high">
                          <div className="flex items-center gap-2">
                            <Badge variant="destructive">High</Badge>
                            <span>Needs prompt attention</span>
                          </div>
                        </SelectItem>
                        <SelectItem value="emergency">
                          <div className="flex items-center gap-2">
                            <Badge variant="destructive">Emergency</Badge>
                            <span>Immediate attention required</span>
                          </div>
                        </SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            {/* Description */}
            <FormField
              control={form.control}
              name="description"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Description</FormLabel>
                  <FormControl>
                    <Textarea
                      placeholder="Please provide detailed information about the maintenance issue..."
                      className="min-h-[100px]"
                      {...field}
                    />
                  </FormControl>
                  <FormDescription>
                    Include as much detail as possible to help us address the
                    issue quickly
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* Contact Phone */}
            <FormField
              control={form.control}
              name="contactPhone"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Contact Phone (Optional)</FormLabel>
                  <FormControl>
                    <Input
                      placeholder="Phone number for urgent contact"
                      {...field}
                    />
                  </FormControl>
                  <FormDescription>
                    Provide a phone number if this is urgent and you need
                    immediate contact
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* Submit Buttons */}
            <div className="flex gap-4">
              <Button
                type="submit"
                disabled={
                  effectiveSubmitting || loadingLeases || !selectedLease
                }
                className="flex-1 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white font-medium py-2.5 px-4 rounded-lg transition-all duration-200 shadow-lg hover:shadow-xl disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {effectiveSubmitting ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Creating Request...
                  </>
                ) : (
                  "Create Maintenance Request"
                )}
              </Button>
              {onCancel && (
                <Button
                  type="button"
                  variant="outline"
                  onClick={onCancel}
                  disabled={effectiveSubmitting || loadingLeases}
                >
                  Cancel
                </Button>
              )}
            </div>
          </form>
        </Form>
      </CardContent>
    </Card>
  );
}
