"use client";

import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Checkbox } from "@/components/ui/checkbox";
import { toast } from "sonner";
import {
  CheckCircle,
  XCircle,
  Clock,
  UserCheck,
  UserX,
  AlertTriangle,
  Loader2,
  Users,
} from "lucide-react";
import type { TenantStatus } from "./types";

interface BulkStatusDialogProps {
  isOpen: boolean;
  onClose: () => void;
  selectedTenants: Array<{
    _id: string;
    firstName: string;
    lastName: string;
    tenantStatus?: TenantStatus;
  }>;
  onBulkStatusChange: (tenantIds: string[], newStatus: TenantStatus) => void;
  userRole: string;
}

const statusOptions = [
  {
    value: "under_review" as TenantStatus,
    label: "Start Review",
    icon: Clock,
    color: "outline" as const,
    description: "Begin application review process",
  },
  {
    value: "approved" as TenantStatus,
    label: "Approve Applications",
    icon: CheckCircle,
    color: "default" as const,
    description: "Approve tenant applications",
  },
  {
    value: "terminated" as TenantStatus,
    label: "Terminate",
    icon: XCircle,
    color: "destructive" as const,
    description: "Terminate tenant applications/leases",
  },
];

const getStatusInfo = (status?: TenantStatus) => {
  const statusMap = {
    application_submitted: {
      label: "Application Submitted",
      color: "secondary" as const,
      icon: Clock,
    },
    under_review: {
      label: "Under Review",
      color: "outline" as const,
      icon: Clock,
    },
    approved: {
      label: "Approved",
      color: "default" as const,
      icon: CheckCircle,
    },
    active: {
      label: "Active",
      color: "default" as const,
      icon: UserCheck,
    },
    inactive: {
      label: "Inactive",
      color: "secondary" as const,
      icon: UserX,
    },
    moved_out: {
      label: "Moved Out",
      color: "secondary" as const,
      icon: UserX,
    },
    terminated: {
      label: "Terminated",
      color: "destructive" as const,
      icon: XCircle,
    },
  };

  return statusMap[status || "application_submitted"];
};

export default function BulkStatusDialog({
  isOpen,
  onClose,
  selectedTenants,
  onBulkStatusChange,
  userRole,
}: BulkStatusDialogProps) {
  const [selectedStatus, setSelectedStatus] = useState<TenantStatus | "">("");
  const [reason, setReason] = useState("");
  const [notes, setNotes] = useState("");
  const [confirmAction, setConfirmAction] = useState(false);
  const [isLoading, setIsLoading] = useState(false);

  const canManageStatus = ["admin", "manager"].includes(userRole);

  const handleClose = () => {
    setSelectedStatus("");
    setReason("");
    setNotes("");
    setConfirmAction(false);
    onClose();
  };

  const handleBulkStatusChange = async () => {
    if (!selectedStatus) {
      toast.error("Please select a status");
      return;
    }

    if (!confirmAction) {
      toast.error("Please confirm the bulk action");
      return;
    }

    setIsLoading(true);
    try {
      const tenantIds = selectedTenants.map((t) => t._id);

      const batchSize = 5;
      const batches = [];
      for (let i = 0; i < tenantIds.length; i += batchSize) {
        batches.push(tenantIds.slice(i, i + batchSize));
      }

      let successCount = 0;
      let errorCount = 0;
      const failedTenantIds: string[] = [];
      const tenantNameMap = new Map(
        selectedTenants.map((tenant) => [
          tenant._id,
          `${tenant.firstName} ${tenant.lastName}`.trim(),
        ])
      );

      for (const batch of batches) {
        const promises = batch.map(async (tenantId) => {
          try {
            const response = await fetch(`/api/tenants/${tenantId}/status`, {
              method: "POST",
              headers: {
                "Content-Type": "application/json",
              },
              body: JSON.stringify({
                newStatus: selectedStatus,
                reason: reason.trim() || "Bulk status update by admin",
                notes: notes.trim(),
              }),
            });

            if (!response.ok) {
              throw new Error(`Failed to update tenant ${tenantId}`);
            }

            successCount++;
            return tenantId;
          } catch (error) {
            errorCount++;
            failedTenantIds.push(tenantId);
            return null;
          }
        });

        await Promise.all(promises);
      }

      if (successCount > 0) {
        const selectedStatusInfo = getStatusInfo(selectedStatus);
        const failureSummary =
          errorCount > 0
            ? (() => {
                const names = failedTenantIds
                  .map((id) => tenantNameMap.get(id) || id)
                  .slice(0, 3)
                  .join(", ");
                return errorCount > 3
                  ? `${errorCount} failed (e.g., ${names}…)`
                  : `${errorCount} failed${names ? ` (${names})` : ""}`;
              })()
            : "";

        toast.success(`Bulk status update completed`, {
          description: `${successCount} tenant(s) updated to ${
            selectedStatusInfo.label
          }${failureSummary ? `, ${failureSummary}` : ""}`,
        });

        onBulkStatusChange(tenantIds, selectedStatus);
      } else {
        toast.error("Failed to update any tenants", {
          description:
            "Please check the individual tenant statuses and try again",
        });
      }

      handleClose();
    } catch (error) {
      toast.error("Failed to update tenant statuses", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
    } finally {
      setIsLoading(false);
    }
  };

  if (!canManageStatus || selectedTenants.length === 0) {
    return null;
  }

  const selectedStatusInfo = selectedStatus
    ? getStatusInfo(selectedStatus)
    : null;

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-[600px]">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Users className="h-5 w-5" />
            Bulk Status Update
          </DialogTitle>
          <DialogDescription>
            Update the status for {selectedTenants.length} selected tenant(s)
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-6">
          <div className="space-y-2">
            <Label>Selected Tenants ({selectedTenants.length})</Label>
            <div className="max-h-32 overflow-y-auto border rounded-lg p-3 bg-muted/50">
              <div className="space-y-1">
                {selectedTenants.map((tenant) => {
                  const statusInfo = getStatusInfo(tenant.tenantStatus);
                  return (
                    <div
                      key={tenant._id}
                      className="flex items-center justify-between text-sm"
                    >
                      <span className="font-medium">
                        {tenant.firstName} {tenant.lastName}
                      </span>
                      <Badge variant={statusInfo.color} className="text-xs">
                        {statusInfo.label}
                      </Badge>
                    </div>
                  );
                })}
              </div>
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="status">New Status *</Label>
            <Select
              value={selectedStatus}
              onValueChange={(value) =>
                setSelectedStatus(value as TenantStatus)
              }
            >
              <SelectTrigger>
                <SelectValue placeholder="Select new status for all tenants" />
              </SelectTrigger>
              <SelectContent>
                {statusOptions.map((option) => {
                  const Icon = option.icon;
                  return (
                    <SelectItem key={option.value} value={option.value}>
                      <div className="flex items-center gap-2">
                        <Icon className="h-4 w-4" />
                        <div>
                          <div className="font-medium">{option.label}</div>
                          <div className="text-xs text-muted-foreground">
                            {option.description}
                          </div>
                        </div>
                      </div>
                    </SelectItem>
                  );
                })}
              </SelectContent>
            </Select>
          </div>

          <div className="space-y-2">
            <Label htmlFor="reason">Reason *</Label>
            <Input
              id="reason"
              placeholder="Enter reason for bulk status change"
              value={reason}
              onChange={(e) => setReason(e.target.value)}
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="notes">Additional Notes</Label>
            <Textarea
              id="notes"
              placeholder="Enter any additional notes (optional)"
              value={notes}
              onChange={(e) => setNotes(e.target.value)}
              rows={3}
            />
          </div>

          <div className="flex items-start gap-2 p-3 border rounded-lg bg-muted/50">
            <Checkbox
              id="confirm"
              checked={confirmAction}
              onCheckedChange={(checked) =>
                setConfirmAction(checked as boolean)
              }
            />
            <div className="space-y-1">
              <Label
                htmlFor="confirm"
                className="text-sm font-medium cursor-pointer"
              >
                I confirm this bulk action
              </Label>
              <p className="text-xs text-muted-foreground">
                This will update the status for all {selectedTenants.length}{" "}
                selected tenant(s).
                {selectedStatus === "terminated" &&
                  " This action will terminate all selected tenants."}
              </p>
            </div>
          </div>

          {selectedStatus === "terminated" && (
            <div className="flex items-start gap-2 p-3 border border-destructive/20 bg-destructive/5 rounded-lg">
              <AlertTriangle className="h-5 w-5 text-destructive mt-0.5" />
              <div className="text-sm">
                <p className="font-medium text-destructive">Warning</p>
                <p className="text-muted-foreground">
                  Terminating tenants will mark them as permanently inactive.
                  This action should only be used for lease terminations or
                  rejected applications.
                </p>
              </div>
            </div>
          )}
        </div>

        <DialogFooter>
          <Button variant="outline" onClick={handleClose} disabled={isLoading}>
            Cancel
          </Button>
          <Button
            onClick={handleBulkStatusChange}
            disabled={
              isLoading || !selectedStatus || !reason.trim() || !confirmAction
            }
            variant={
              selectedStatus === "terminated" ? "destructive" : "default"
            }
          >
            {isLoading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
            Update {selectedTenants.length} Tenant(s)
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
