"use client";

import { useMemo } from "react";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import {
  Users,
  UserCheck,
  UserX,
  Clock,
  CheckCircle,
  XCircle,
  CreditCard,
  TrendingUp,
  TrendingDown,
  Calendar,
} from "lucide-react";
import {
  AnalyticsCard,
  AnalyticsCardGrid,
} from "@/components/analytics/AnalyticsCard";
import { formatCurrency } from "@/lib/utils/formatting";
import type { TenantRecord } from "./types";

const currencyDisplayOptions = {
  minimumFractionDigits: 0,
  maximumFractionDigits: 0,
} as const;

type Tenant = TenantRecord;

interface TenantStatsProps {
  tenants: Tenant[];
  isLoading?: boolean;
}

export default function TenantStats({ tenants, isLoading }: TenantStatsProps) {
  const stats = useMemo(() => {
    if (!tenants.length) {
      return {
        total: 0,
        active: 0,
        pending: 0,
        movedOut: 0,
        approved: 0,
        rejected: 0,
        averageCreditScore: 0,
        averageIncome: 0,
        thisMonthApplications: 0,
        lastMonthApplications: 0,
      };
    }

    const now = new Date();
    const thisMonth = new Date(now.getFullYear(), now.getMonth(), 1);
    const lastMonth = new Date(now.getFullYear(), now.getMonth() - 1, 1);
    const lastMonthEnd = new Date(now.getFullYear(), now.getMonth(), 0);

    const active = tenants.filter((t) => t.tenantStatus === "active").length;
    const pending = tenants.filter(
      (t) =>
        t.tenantStatus === "application_submitted" ||
        t.tenantStatus === "under_review"
    ).length;
    const movedOut = tenants.filter(
      (t) => t.tenantStatus === "moved_out"
    ).length;
    const approved = tenants.filter(
      (t) => t.tenantStatus === "approved"
    ).length;
    const rejected = tenants.filter(
      (t) => t.tenantStatus === "terminated"
    ).length;

    const tenantsWithCreditScore = tenants.filter((t) => t.creditScore);
    const averageCreditScore =
      tenantsWithCreditScore.length > 0
        ? Math.round(
            tenantsWithCreditScore.reduce(
              (sum, t) => sum + (t.creditScore || 0),
              0
            ) / tenantsWithCreditScore.length
          )
        : 0;

    const tenantsWithIncome = tenants.filter((t) => t.employmentInfo?.income);
    const averageIncome =
      tenantsWithIncome.length > 0
        ? Math.round(
            tenantsWithIncome.reduce(
              (sum, t) => sum + (t.employmentInfo?.income || 0),
              0
            ) / tenantsWithIncome.length
          )
        : 0;

    const thisMonthApplications = tenants.filter(
      (t) => new Date(t.applicationDate) >= thisMonth
    ).length;

    const lastMonthApplications = tenants.filter((t) => {
      const appDate = new Date(t.applicationDate);
      return appDate >= lastMonth && appDate <= lastMonthEnd;
    }).length;

    return {
      total: tenants.length,
      active,
      pending,
      movedOut,
      approved,
      rejected,
      averageCreditScore,
      averageIncome,
      thisMonthApplications,
      lastMonthApplications,
    };
  }, [tenants]);

  const getApplicationTrend = () => {
    if (stats.lastMonthApplications === 0) {
      return { trend: "neutral", percentage: 0 };
    }

    const change =
      ((stats.thisMonthApplications - stats.lastMonthApplications) /
        stats.lastMonthApplications) *
      100;
    return {
      trend: change > 0 ? "up" : change < 0 ? "down" : "neutral",
      percentage: Math.abs(Math.round(change)),
    };
  };

  const applicationTrend = getApplicationTrend();

  if (isLoading) {
    return (
      <AnalyticsCardGrid className="lg:grid-cols-4">
        {[...Array(8)].map((_, i) => (
          <Card key={i} className="hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                <div className="h-4 bg-muted rounded animate-pulse" />
              </CardTitle>
              <div className="h-4 w-4 bg-muted rounded animate-pulse" />
            </CardHeader>
            <CardContent>
              <div className="h-8 bg-muted rounded animate-pulse mb-2" />
              <div className="h-3 bg-muted rounded animate-pulse w-2/3" />
            </CardContent>
          </Card>
        ))}
      </AnalyticsCardGrid>
    );
  }

  return (
    <AnalyticsCardGrid className="lg:grid-cols-4">
      <AnalyticsCard
        title="Total Tenants"
        value={stats.total}
        description="All tenant profiles"
        icon={Users}
        iconColor="primary"
      />

      <AnalyticsCard
        title="Active Tenants"
        value={stats.active}
        description="Currently living in properties"
        icon={UserCheck}
        iconColor="success"
      />

      <AnalyticsCard
        title="Pending Review"
        value={stats.pending}
        description="Awaiting background check"
        icon={Clock}
        iconColor="warning"
      />

      <AnalyticsCard
        title="Avg Credit Score"
        value={stats.averageCreditScore}
        description={`${
          stats.averageCreditScore >= 700
            ? "Excellent"
            : stats.averageCreditScore >= 600
            ? "Good"
            : "Fair"
        } average`}
        icon={CreditCard}
        iconColor="info"
      />

      <AnalyticsCard
        title="Approved"
        value={stats.approved}
        description="Ready to move in"
        icon={CheckCircle}
        iconColor="success"
      />

      <AnalyticsCard
        title="Rejected"
        value={stats.rejected}
        description="Applications declined"
        icon={XCircle}
        iconColor="error"
      />

      <AnalyticsCard
        title="Avg Income"
        value={
          stats.averageIncome > 0
            ? formatCurrency(stats.averageIncome, undefined, currencyDisplayOptions)
            : "N/A"
        }
        description="Annual household income"
        icon={TrendingUp}
        iconColor="success"
      />

      <AnalyticsCard
        title="This Month"
        value={stats.thisMonthApplications}
        description="New applications"
        icon={Calendar}
        iconColor="info"
        trend={
          applicationTrend.trend !== "neutral"
            ? {
                value: `${applicationTrend.percentage}% from last month`,
                isPositive: applicationTrend.trend === "up",
                icon:
                  applicationTrend.trend === "up" ? TrendingUp : TrendingDown,
              }
            : undefined
        }
      />
    </AnalyticsCardGrid>
  );
}
