"use client";

import { useState } from "react";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { toast } from "sonner";
import {
  CheckCircle,
  XCircle,
  Clock,
  UserCheck,
  UserX,
  AlertTriangle,
  Loader2,
} from "lucide-react";
import { DatePicker } from "@/components/ui/date-picker";
import type { TenantStatus } from "./types";
import { formatDate } from "@/lib/utils/formatting";

interface TenantStatusDialogProps {
  isOpen: boolean;
  onClose: () => void;
  tenant: {
    _id: string;
    firstName: string;
    lastName: string;
    tenantStatus?: TenantStatus;
    lastStatusUpdate?: string;
  };
  onStatusChange: (newStatus: TenantStatus) => void;
  userRole: string;
}

const statusOptions = [
  {
    value: "under_review" as TenantStatus,
    label: "Start Review",
    icon: Clock,
    color: "outline" as const,
    description: "Begin application review process",
  },
  {
    value: "approved" as TenantStatus,
    label: "Approve Application",
    icon: CheckCircle,
    color: "default" as const,
    description: "Approve tenant application",
  },
  {
    value: "active" as TenantStatus,
    label: "Activate Tenant",
    icon: UserCheck,
    color: "default" as const,
    description: "Mark tenant as active (moved in)",
  },
  {
    value: "inactive" as TenantStatus,
    label: "Deactivate",
    icon: UserX,
    color: "secondary" as const,
    description: "Temporarily deactivate tenant",
  },
  {
    value: "moved_out" as TenantStatus,
    label: "Mark Moved Out",
    icon: UserX,
    color: "secondary" as const,
    description: "Tenant has moved out",
  },
  {
    value: "terminated" as TenantStatus,
    label: "Terminate",
    icon: XCircle,
    color: "destructive" as const,
    description: "Terminate tenant application/lease",
  },
];

const getAvailableTransitions = (
  currentStatus?: TenantStatus
): TenantStatus[] => {
  const transitions: Record<TenantStatus, TenantStatus[]> = {
    application_submitted: ["under_review", "approved", "terminated"],
    under_review: ["approved", "terminated"],
    approved: ["active", "terminated"],
    active: ["inactive", "moved_out", "terminated"],
    inactive: ["active", "moved_out", "terminated"],
    moved_out: ["terminated"],
    terminated: [],
  };

  return transitions[currentStatus || "application_submitted"] || [];
};

const getStatusInfo = (status?: TenantStatus) => {
  const statusMap = {
    application_submitted: {
      label: "Application Submitted",
      color: "secondary" as const,
      icon: Clock,
    },
    under_review: {
      label: "Under Review",
      color: "outline" as const,
      icon: Clock,
    },
    approved: {
      label: "Approved",
      color: "default" as const,
      icon: CheckCircle,
    },
    active: {
      label: "Active",
      color: "default" as const,
      icon: UserCheck,
    },
    inactive: {
      label: "Inactive",
      color: "secondary" as const,
      icon: UserX,
    },
    moved_out: {
      label: "Moved Out",
      color: "secondary" as const,
      icon: UserX,
    },
    terminated: {
      label: "Terminated",
      color: "destructive" as const,
      icon: XCircle,
    },
  };

  return statusMap[status || "application_submitted"];
};

export default function TenantStatusDialog({
  isOpen,
  onClose,
  tenant,
  onStatusChange,
  userRole,
}: TenantStatusDialogProps) {
  const [selectedStatus, setSelectedStatus] = useState<TenantStatus | "">("");
  const [reason, setReason] = useState("");
  const [notes, setNotes] = useState("");
  const [moveDate, setMoveDate] = useState<Date | undefined>(undefined);
  const [isLoading, setIsLoading] = useState(false);

  const canManageStatus = ["admin", "manager"].includes(userRole);
  const currentStatus = tenant.tenantStatus || "application_submitted";
  const availableTransitions = getAvailableTransitions(currentStatus);
  const currentStatusInfo = getStatusInfo(currentStatus);
  const selectedStatusInfo = selectedStatus
    ? getStatusInfo(selectedStatus)
    : null;
  const lastUpdated = tenant.lastStatusUpdate
    ? formatDate(tenant.lastStatusUpdate, { format: "medium" })
    : "Recently updated";

  const handleClose = () => {
    setSelectedStatus("");
    setReason("");
    setNotes("");
    setMoveDate(undefined);
    onClose();
  };

  const handleStatusChange = async () => {
    if (!selectedStatus) {
      toast.error("Please select a status");
      return;
    }

    if (!reason.trim()) {
      toast.error("Please provide a reason for the status change");
      return;
    }

    if (
      (selectedStatus === "active" || selectedStatus === "moved_out") &&
      !moveDate
    ) {
      toast.error(
        `${
          selectedStatus === "active" ? "Move-in" : "Move-out"
        } date is required`
      );
      return;
    }

    setIsLoading(true);
    try {
      const response = await fetch(`/api/tenants/${tenant._id}/status`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          newStatus: selectedStatus,
          reason: reason.trim(),
          notes: notes.trim() || undefined,
          moveDate: moveDate?.toISOString().split("T")[0] || undefined,
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || "Failed to update status");
      }

      toast.success(`Tenant status updated to ${selectedStatusInfo?.label}`, {
        description: `${tenant.firstName} ${tenant.lastName} status changed successfully`,
      });

      onStatusChange(selectedStatus);
      handleClose();
    } catch (error) {
      toast.error("Failed to update tenant status", {
        description:
          error instanceof Error ? error.message : "Please try again",
      });
    } finally {
      setIsLoading(false);
    }
  };

  if (!canManageStatus) {
    return null;
  }

  const availableOptions = statusOptions.filter((option) =>
    availableTransitions.includes(option.value)
  );

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-[500px]">
        <DialogHeader>
          <DialogTitle>Change Tenant Status</DialogTitle>
          <DialogDescription>
            Update the status for {tenant.firstName} {tenant.lastName}
          </DialogDescription>
        </DialogHeader>

          <div className="space-y-4">
            <div className="flex items-center gap-3 p-3 rounded-lg bg-muted/30 border">
              <div className="flex items-center justify-center w-8 h-8 rounded-full bg-background border">
                <Clock className="h-4 w-4 text-muted-foreground" />
              </div>
              <div>
                <p className="text-sm font-medium text-muted-foreground">
                  Current Status
                </p>
                <p className="font-semibold">{currentStatusInfo.label}</p>
                <p className="text-xs text-muted-foreground">{lastUpdated}</p>
              </div>
            </div>

            <div className="space-y-2">
            <Label htmlFor="status" className="text-sm font-medium">
              New Status *
            </Label>
            <Select
              value={selectedStatus}
              onValueChange={(value) =>
                setSelectedStatus(value as TenantStatus)
              }
            >
              <SelectTrigger className="h-11">
                <SelectValue placeholder="Select new status" />
              </SelectTrigger>
              <SelectContent>
                {availableOptions.map((option) => {
                  const Icon = option.icon;
                  return (
                    <SelectItem
                      key={option.value}
                      value={option.value}
                      className="py-3"
                    >
                      <div className="flex items-center gap-3">
                        <div className="flex items-center justify-center w-8 h-8 rounded-full bg-muted">
                          <Icon className="h-4 w-4" />
                        </div>
                        <div>
                          <div className="font-medium">{option.label}</div>
                          <div className="text-xs text-muted-foreground">
                            {option.description}
                          </div>
                        </div>
                      </div>
                    </SelectItem>
                  );
                })}
              </SelectContent>
            </Select>
          </div>

          {(selectedStatus === "active" || selectedStatus === "moved_out") && (
            <div className="space-y-2">
              <Label htmlFor="moveDate" className="text-sm font-medium">
                {selectedStatus === "active" ? "Move-in Date" : "Move-out Date"}{" "}
                *
              </Label>
              <DatePicker
                date={moveDate}
                onSelect={setMoveDate}
                placeholder={`Select ${
                  selectedStatus === "active" ? "move-in" : "move-out"
                } date`}
                disabled={(date) => date > new Date()}
              />
            </div>
          )}

          <div className="space-y-2">
            <Label htmlFor="reason" className="text-sm font-medium">
              Reason for Change *
            </Label>
            <Textarea
              id="reason"
              placeholder="Enter the reason for this status change..."
              value={reason}
              onChange={(e) => setReason(e.target.value)}
              rows={2}
              className="resize-none"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="notes" className="text-sm font-medium">
              Additional Notes
            </Label>
            <Textarea
              id="notes"
              placeholder="Enter any additional notes (optional)"
              value={notes}
              onChange={(e) => setNotes(e.target.value)}
              rows={2}
              className="resize-none"
            />
          </div>

          {selectedStatus === "terminated" && (
            <div className="flex items-start gap-3 p-4 border border-destructive/20 bg-destructive/5 rounded-lg">
              <div className="flex items-center justify-center w-8 h-8 rounded-full bg-destructive/10 border border-destructive/20">
                <AlertTriangle className="h-4 w-4 text-destructive" />
              </div>
              <div className="text-sm">
                <p className="font-medium text-destructive mb-1">Warning</p>
                <p className="text-muted-foreground">
                  Terminating a tenant will mark them as permanently inactive.
                  This action should only be used for lease terminations or
                  rejected applications.
                </p>
              </div>
            </div>
          )}
        </div>

        <DialogFooter className="gap-2 sm:gap-2">
          <Button
            variant="outline"
            onClick={handleClose}
            disabled={isLoading}
            className="h-11"
          >
            Cancel
          </Button>
          <Button
            onClick={handleStatusChange}
            disabled={isLoading || !selectedStatus || !reason.trim()}
            variant={
              selectedStatus === "terminated" ? "destructive" : "default"
            }
            className="h-11"
          >
            {isLoading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
            Update Status
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
