/**
 * PropertyPro - Role Permissions Hook
 * Enhanced permission management supporting both system and custom roles
 */

"use client";

import { useState, useEffect, useCallback } from "react";
import { useSession } from "next-auth/react";
import { UserRole, IRoleConfig } from "@/types";

// ============================================================================
// TYPES
// ============================================================================

interface RolePermissionsData {
  permissions: string[];
  roleInfo: IRoleConfig | null;
  isLoading: boolean;
  error: string | null;
}

interface UseRolePermissionsReturn extends RolePermissionsData {
  hasPermission: (permission: string) => boolean;
  hasAnyPermission: (permissions: string[]) => boolean;
  hasAllPermissions: (permissions: string[]) => boolean;
  refreshPermissions: () => Promise<void>;
  isSystemRole: boolean;
  isCustomRole: boolean;
}

// ============================================================================
// SYSTEM ROLE PERMISSIONS
// ============================================================================

const SYSTEM_ROLE_PERMISSIONS: Record<UserRole, string[]> = {
  [UserRole.ADMIN]: [
    "user_management",
    "role_management",
    "property_management",
    "tenant_management",
    "lease_management",
    "maintenance_management",
    "financial_management",
    "system_settings",
    "audit_logs",
    "backup_restore",
    "reports_all",
    "bulk_operations",
    "company_settings",
    "data_export",
    "advanced_analytics",
  ],
  [UserRole.MANAGER]: [
    "property_management",
    "tenant_management",
    "lease_management",
    "maintenance_management",
    "financial_management",
    "reports_all",
    "bulk_operations",
    "property_create",
    "tenant_create",
    "lease_create",
    "maintenance_assign",
    "payment_processing",
    "document_management",
  ],
  [UserRole.TENANT]: [
    "profile_management",
    "maintenance_requests",
    "payment_portal",
    "document_access",
    "lease_view",
    "payment_history",
    "maintenance_history",
  ],
};

// ============================================================================
// PERMISSION CACHE
// ============================================================================

const permissionCache = new Map<
  string,
  { permissions: string[]; timestamp: number; roleInfo: IRoleConfig | null }
>();
const CACHE_DURATION = 5 * 60 * 1000; // 5 minutes

// ============================================================================
// HOOK IMPLEMENTATION
// ============================================================================

export function useRolePermissions(
  targetRole?: string
): UseRolePermissionsReturn {
  const { data: session } = useSession();
  const userRole = targetRole || session?.user?.role;

  const [permissionsData, setPermissionsData] = useState<RolePermissionsData>({
    permissions: [],
    roleInfo: null,
    isLoading: true,
    error: null,
  });

  // Check if role is a system role
  const isSystemRole = userRole
    ? Object.values(UserRole).includes(userRole as UserRole)
    : false;
  const isCustomRole = userRole ? !isSystemRole : false;

  // Fetch permissions for custom roles
  const fetchCustomRolePermissions = useCallback(
    async (
      roleName: string
    ): Promise<{ permissions: string[]; roleInfo: IRoleConfig | null }> => {
      try {
        // Check cache first
        const cached = permissionCache.get(roleName);
        if (cached && Date.now() - cached.timestamp < CACHE_DURATION) {
          return { permissions: cached.permissions, roleInfo: cached.roleInfo };
        }

        const response = await fetch(
          `/api/roles?search=${encodeURIComponent(roleName)}&limit=1`
        );
        if (!response.ok) {
          throw new Error(
            `Failed to fetch role permissions: ${response.statusText}`
          );
        }

        const data = await response.json();
        const roles = data?.data?.roles ?? data?.roles ?? [];
        const role = roles.find((r: IRoleConfig) => r.name === roleName);

        if (!role) {
          throw new Error(`Role '${roleName}' not found`);
        }

        const result = { permissions: role.permissions, roleInfo: role };

        // Cache the result
        permissionCache.set(roleName, {
          permissions: role.permissions,
          roleInfo: role,
          timestamp: Date.now(),
        });

        return result;
      } catch (error) {
        console.error("Error fetching custom role permissions:", error);
        throw error;
      }
    },
    []
  );

  // Load permissions
  const loadPermissions = useCallback(async () => {
    if (!userRole) {
      setPermissionsData({
        permissions: [],
        roleInfo: null,
        isLoading: false,
        error: "No role specified",
      });
      return;
    }

    setPermissionsData((prev) => ({ ...prev, isLoading: true, error: null }));

    try {
      if (isSystemRole) {
        // Handle system roles
        const permissions = SYSTEM_ROLE_PERMISSIONS[userRole as UserRole] || [];
        const roleInfo: IRoleConfig = {
          name: userRole,
          label: userRole.charAt(0).toUpperCase() + userRole.slice(1),
          description: `System role: ${userRole}`,
          permissions,
          isSystem: true,
          isActive: true,
          color:
            userRole === UserRole.ADMIN
              ? "destructive"
              : userRole === UserRole.MANAGER
              ? "default"
              : "outline",
          userCount: 0,
          canEdit: false,
          canDelete: false,
        };

        setPermissionsData({
          permissions,
          roleInfo,
          isLoading: false,
          error: null,
        });
      } else {
        // Handle custom roles
        const { permissions, roleInfo } = await fetchCustomRolePermissions(
          userRole
        );
        setPermissionsData({
          permissions,
          roleInfo,
          isLoading: false,
          error: null,
        });
      }
    } catch (error) {
      setPermissionsData({
        permissions: [],
        roleInfo: null,
        isLoading: false,
        error:
          error instanceof Error ? error.message : "Failed to load permissions",
      });
    }
  }, [userRole, isSystemRole, fetchCustomRolePermissions]);

  // Refresh permissions (clears cache)
  const refreshPermissions = useCallback(async () => {
    if (userRole && !isSystemRole) {
      permissionCache.delete(userRole);
    }
    await loadPermissions();
  }, [userRole, isSystemRole, loadPermissions]);

  // Permission checking functions
  const hasPermission = useCallback(
    (permission: string): boolean => {
      return permissionsData.permissions.includes(permission);
    },
    [permissionsData.permissions]
  );

  const hasAnyPermission = useCallback(
    (permissions: string[]): boolean => {
      return permissions.some((permission) =>
        permissionsData.permissions.includes(permission)
      );
    },
    [permissionsData.permissions]
  );

  const hasAllPermissions = useCallback(
    (permissions: string[]): boolean => {
      return permissions.every((permission) =>
        permissionsData.permissions.includes(permission)
      );
    },
    [permissionsData.permissions]
  );

  // Load permissions on mount and when role changes
  useEffect(() => {
    loadPermissions();
  }, [loadPermissions]);

  return {
    ...permissionsData,
    hasPermission,
    hasAnyPermission,
    hasAllPermissions,
    refreshPermissions,
    isSystemRole,
    isCustomRole,
  };
}

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================

/**
 * Get permissions for a role (cached version)
 */
export async function getRolePermissions(roleName: string): Promise<string[]> {
  // Check if it's a system role
  if (Object.values(UserRole).includes(roleName as UserRole)) {
    return SYSTEM_ROLE_PERMISSIONS[roleName as UserRole] || [];
  }

  // Check cache for custom roles
  const cached = permissionCache.get(roleName);
  if (cached && Date.now() - cached.timestamp < CACHE_DURATION) {
    return cached.permissions;
  }

  // Fetch from API for custom roles
  try {
    const response = await fetch(
      `/api/roles?search=${encodeURIComponent(roleName)}&limit=1`
    );
    if (!response.ok) {
      throw new Error(
        `Failed to fetch role permissions: ${response.statusText}`
      );
    }

    const data = await response.json();
    const role = data.roles?.find((r: IRoleConfig) => r.name === roleName);

    if (!role) {
      return [];
    }

    // Cache the result
    permissionCache.set(roleName, {
      permissions: role.permissions,
      roleInfo: role,
      timestamp: Date.now(),
    });

    return role.permissions;
  } catch (error) {
    console.error("Error fetching role permissions:", error);
    return [];
  }
}

/**
 * Clear permission cache for a specific role or all roles
 */
export function clearPermissionCache(roleName?: string): void {
  if (roleName) {
    permissionCache.delete(roleName);
  } else {
    permissionCache.clear();
  }
}
