/**
 * PropertyPro - Database Schema Updates
 * Essential database fields for payment system functionality
 */

import { Schema, model, models } from "mongoose";

// Enhanced Payment Schema with new fields
const enhancedPaymentSchema = new Schema(
  {
    // Existing fields
    tenantId: { type: Schema.Types.ObjectId, ref: "User", required: true },
    propertyId: {
      type: Schema.Types.ObjectId,
      ref: "Property",
      required: true,
    },
    amount: { type: Number, required: true },
    dueDate: { type: Date, required: true },
    paidDate: { type: Date },
    status: {
      type: String,
      enum: ["pending", "processing", "paid", "failed", "overdue", "cancelled"],
      default: "pending",
    },
    paymentMethod: {
      type: String,
      enum: [
        "credit_card",
        "debit_card",
        "bank_transfer",
        "cash",
        "check",
        "money_order",
      ],
      required: true,
    },

    // NEW FIELDS FOR ENHANCED PAYMENT SYSTEM

    // Late Fee Fields
    lateFee: {
      amount: { type: Number, default: 0 },
      appliedDate: { type: Date },
      gracePeriodEnd: { type: Date },
      autoApplied: { type: Boolean, default: false },
      waived: { type: Boolean, default: false },
      waivedBy: { type: Schema.Types.ObjectId, ref: "User" },
      waivedReason: { type: String },
      feeStructure: {
        type: {
          type: String,
          enum: ["fixed", "percentage", "tiered", "daily"],
        },
        rate: { type: Number }, // Fixed amount or percentage
        maxAmount: { type: Number },
        dailyRate: { type: Number },
      },
    },

    // Proration Fields
    proration: {
      isProrated: { type: Boolean, default: false },
      originalAmount: { type: Number },
      proratedAmount: { type: Number },
      prorationType: {
        type: String,
        enum: ["move_in", "move_out", "mid_month_change"],
      },
      periodStart: { type: Date },
      periodEnd: { type: Date },
      daysInPeriod: { type: Number },
      daysOccupied: { type: Number },
      calculationMethod: { type: String, enum: ["daily", "calendar_month"] },
      calculationDetails: { type: Schema.Types.Mixed },
    },

    // Scheduling Fields
    scheduling: {
      isRecurring: { type: Boolean, default: false },
      recurringType: {
        type: String,
        enum: ["monthly", "quarterly", "annually"],
      },
      nextDueDate: { type: Date },
      autoGenerated: { type: Boolean, default: false },
      parentPaymentId: { type: Schema.Types.ObjectId, ref: "Payment" },
      generatedPayments: [{ type: Schema.Types.ObjectId, ref: "Payment" }],
      schedulingRules: {
        dayOfMonth: { type: Number }, // 1-31
        monthsInterval: { type: Number, default: 1 },
        endDate: { type: Date },
        maxOccurrences: { type: Number },
      },
    },

    // Payment Processing Fields
    processing: {
      stripePaymentIntentId: { type: String },
      stripeChargeId: { type: String },
      processingFee: { type: Number, default: 0 },
      netAmount: { type: Number },
      processingTime: { type: Number }, // milliseconds
      retryCount: { type: Number, default: 0 },
      lastRetryDate: { type: Date },
      failureReason: { type: String },
      refundId: { type: String },
      refundAmount: { type: Number },
      refundDate: { type: Date },
    },

    // Communication Tracking
    communications: [
      {
        type: {
          type: String,
          enum: [
            "reminder",
            "overdue_notice",
            "confirmation",
            "failure_notice",
          ],
        },
        channel: { type: String, enum: ["email", "sms", "push"] },
        sentDate: { type: Date },
        status: {
          type: String,
          enum: ["sent", "delivered", "failed", "opened"],
        },
        templateId: { type: String },
        messageId: { type: String },
      },
    ],

    // Audit Trail
    auditTrail: [
      {
        action: { type: String, required: true },
        performedBy: { type: Schema.Types.ObjectId, ref: "User" },
        timestamp: { type: Date, default: Date.now },
        details: { type: Schema.Types.Mixed },
        ipAddress: { type: String },
        userAgent: { type: String },
      },
    ],

    // Metadata
    metadata: {
      source: {
        type: String,
        enum: ["manual", "auto_generated", "imported", "api"],
      },
      tags: [{ type: String }],
      notes: { type: String },
      customFields: { type: Schema.Types.Mixed },
    },
  },
  {
    timestamps: true,
    toJSON: { virtuals: true },
    toObject: { virtuals: true },
  }
);

// Enhanced Tenant Schema with payment preferences
const enhancedTenantSchema = new Schema(
  {
    // Existing user fields...

    // NEW PAYMENT-RELATED FIELDS
    paymentPreferences: {
      preferredMethod: {
        type: String,
        enum: ["credit_card", "debit_card", "bank_transfer"],
      },
      autoPayEnabled: { type: Boolean, default: false },
      autoPayMethod: { type: String },
      gracePeriodDays: { type: Number, default: 5 },
      communicationPreferences: {
        email: { type: Boolean, default: true },
        sms: { type: Boolean, default: false },
        push: { type: Boolean, default: false },
      },
      reminderSchedule: {
        firstReminder: { type: Number, default: 7 }, // days before due
        secondReminder: { type: Number, default: 3 },
        finalReminder: { type: Number, default: 1 },
      },
    },

    // Payment Methods
    paymentMethods: [
      {
        id: { type: String, required: true },
        type: {
          type: String,
          enum: ["credit_card", "debit_card", "bank_account"],
          required: true,
        },
        isDefault: { type: Boolean, default: false },
        last4: { type: String },
        brand: { type: String }, // Visa, Mastercard, etc.
        expiryMonth: { type: Number },
        expiryYear: { type: Number },
        bankName: { type: String },
        accountType: { type: String, enum: ["checking", "savings"] },
        stripePaymentMethodId: { type: String },
        isActive: { type: Boolean, default: true },
        addedDate: { type: Date, default: Date.now },
        lastUsed: { type: Date },
      },
    ],

    // Lease Information
    leaseInfo: {
      currentLeaseId: { type: Schema.Types.ObjectId, ref: "Lease" },
      rentAmount: { type: Number },
      securityDeposit: { type: Number },
      moveInDate: { type: Date },
      moveOutDate: { type: Date },
      leaseStartDate: { type: Date },
      leaseEndDate: { type: Date },
      renewalDate: { type: Date },
      isActive: { type: Boolean, default: true },
    },

    // Payment History Summary
    paymentSummary: {
      totalPaid: { type: Number, default: 0 },
      totalOutstanding: { type: Number, default: 0 },
      totalLateFees: { type: Number, default: 0 },
      onTimePayments: { type: Number, default: 0 },
      latePayments: { type: Number, default: 0 },
      averageDaysToPayment: { type: Number, default: 0 },
      lastPaymentDate: { type: Date },
      nextPaymentDue: { type: Date },
    },
  },
  {
    timestamps: true,
  }
);

// Enhanced Property Schema with payment configuration
const enhancedPropertySchema = new Schema(
  {
    // Existing property fields...

    // NEW PAYMENT CONFIGURATION FIELDS
    paymentConfiguration: {
      gracePeriods: {
        default: { type: Number, default: 5 },
        firstTimeRenters: { type: Number, default: 7 },
        longTermTenants: { type: Number, default: 3 },
        autoPayUsers: { type: Number, default: 2 },
      },

      lateFees: {
        enabled: { type: Boolean, default: true },
        gracePeriodDays: { type: Number, default: 5 },
        feeStructure: {
          type: {
            type: String,
            enum: ["fixed", "percentage", "tiered", "daily"],
            default: "fixed",
          },
          fixedAmount: { type: Number, default: 50 },
          percentage: { type: Number },
          dailyAmount: { type: Number },
          tieredRates: [
            {
              daysLate: { type: Number },
              amount: { type: Number },
            },
          ],
        },
        maximumFee: { type: Number, default: 200 },
        autoApplication: { type: Boolean, default: true },
      },

      communicationSettings: {
        reminderSchedule: {
          firstReminder: { type: Number, default: 7 },
          secondReminder: { type: Number, default: 3 },
          finalReminder: { type: Number, default: 1 },
          overdueNotice: { type: Number, default: 1 },
          escalationNotice: { type: Number, default: 7 },
        },
        channels: {
          email: { type: Boolean, default: true },
          sms: { type: Boolean, default: true },
          push: { type: Boolean, default: false },
        },
        businessHours: {
          enabled: { type: Boolean, default: true },
          start: { type: String, default: "09:00" },
          end: { type: String, default: "17:00" },
          timezone: { type: String, default: "America/New_York" },
          weekendsEnabled: { type: Boolean, default: false },
        },
      },

      autoPayIncentives: {
        enabled: { type: Boolean, default: true },
        discountType: {
          type: String,
          enum: ["fixed", "percentage"],
          default: "fixed",
        },
        discountAmount: { type: Number, default: 10 },
        gracePeriodExtension: { type: Number, default: 2 },
      },
    },

    // Financial Summary
    financialSummary: {
      monthlyRent: { type: Number },
      totalCollected: { type: Number, default: 0 },
      totalOutstanding: { type: Number, default: 0 },
      collectionRate: { type: Number, default: 0 },
      averageDaysToPayment: { type: Number, default: 0 },
      lateFeeRevenue: { type: Number, default: 0 },
      lastUpdated: { type: Date, default: Date.now },
    },
  },
  {
    timestamps: true,
  }
);

// System Configuration Schema
const systemConfigurationSchema = new Schema(
  {
    configType: { type: String, required: true, unique: true },
    version: { type: String, required: true },
    configuration: { type: Schema.Types.Mixed, required: true },
    isActive: { type: Boolean, default: true },
    createdBy: { type: Schema.Types.ObjectId, ref: "User" },
    updatedBy: { type: Schema.Types.ObjectId, ref: "User" },
    changeLog: [
      {
        version: { type: String },
        changes: { type: String },
        changedBy: { type: Schema.Types.ObjectId, ref: "User" },
        changeDate: { type: Date, default: Date.now },
        reason: { type: String },
      },
    ],
  },
  {
    timestamps: true,
  }
);

// Indexes for performance
enhancedPaymentSchema.index({ tenantId: 1, dueDate: -1 });
enhancedPaymentSchema.index({ propertyId: 1, status: 1 });
enhancedPaymentSchema.index({ status: 1, dueDate: 1 });
enhancedPaymentSchema.index({
  "scheduling.nextDueDate": 1,
  "scheduling.isRecurring": 1,
});
enhancedPaymentSchema.index({
  "lateFee.gracePeriodEnd": 1,
  "lateFee.autoApplied": 1,
});

enhancedTenantSchema.index({ "paymentPreferences.autoPayEnabled": 1 });
enhancedTenantSchema.index({ "leaseInfo.currentLeaseId": 1 });
enhancedTenantSchema.index({ "paymentSummary.nextPaymentDue": 1 });

enhancedPropertySchema.index({ "paymentConfiguration.lateFees.enabled": 1 });

// Export models
export const EnhancedPayment =
  models.EnhancedPayment || model("EnhancedPayment", enhancedPaymentSchema);
export const EnhancedTenant =
  models.EnhancedTenant || model("EnhancedTenant", enhancedTenantSchema);
export const EnhancedProperty =
  models.EnhancedProperty || model("EnhancedProperty", enhancedPropertySchema);
export const SystemConfiguration =
  models.SystemConfiguration ||
  model("SystemConfiguration", systemConfigurationSchema);

// Migration helper functions
export const migrationHelpers = {
  /**
   * Migrate existing payments to enhanced schema
   */
  async migratePayments() {

    // Migration logic would go here

  },

  /**
   * Migrate existing tenants to enhanced schema
   */
  async migrateTenants() {

    // Migration logic would go here

  },

  /**
   * Migrate existing properties to enhanced schema
   */
  async migrateProperties() {

    // Migration logic would go here

  },

  /**
   * Initialize default system configuration
   */
  async initializeSystemConfig() {

    const defaultConfig = {
      configType: "payment_system",
      version: "1.0.0",
      configuration: {
        gracePeriods: {
          default: 5,
          firstTimeRenters: 7,
          longTermTenants: 3,
          autoPayUsers: 2,
        },
        lateFees: {
          enabled: true,
          gracePeriodDays: 5,
          feeStructure: {
            type: "fixed",
            fixedAmount: 50,
          },
          maximumFee: 200,
          autoApplication: true,
        },
        communicationTiming: {
          paymentReminders: {
            firstReminder: 7,
            secondReminder: 3,
            finalReminder: 1,
            overdueNotice: 1,
            escalationNotice: 7,
          },
          channels: {
            email: true,
            sms: true,
            pushNotification: false,
          },
          businessHours: {
            enabled: true,
            start: "09:00",
            end: "17:00",
            timezone: "America/New_York",
            weekendsEnabled: false,
          },
        },
        autoPayIncentives: {
          enabled: true,
          discountType: "fixed",
          discountAmount: 10,
          gracePeriodExtension: 2,
        },
      },
    };

    await SystemConfiguration.findOneAndUpdate(
      { configType: "payment_system" },
      defaultConfig,
      { upsert: true, new: true }
    );


  },
};
