import { InvoiceStatus } from "@/types";

export interface InvoiceCompanyInfo {
  name: string;
  address: string;
  phone: string;
  email: string;
  website?: string;
  logo?: string;
  [key: string]: unknown;
}

export interface InvoiceAddress {
  street?: string;
  city?: string;
  state?: string;
  zipCode?: string;
  country?: string;
  line1?: string;
  line2?: string;
  [key: string]: unknown;
}

export interface InvoicePartyInfo {
  firstName?: string;
  lastName?: string;
  email?: string;
  phone?: string;
  name?: string;
  companyName?: string;
  [key: string]: unknown;
}

export interface InvoicePropertyInfo {
  name?: string;
  address?: InvoiceAddress | string;
  unit?: string;
  [key: string]: unknown;
}

export interface InvoiceLineItemInfo {
  description: string;
  quantity: number;
  unitPrice: number;
  total: number;
  amount?: number;
  type?: string;
  dueDate?: Date | null;
  metadata?: Record<string, unknown>;
}

export interface InvoiceStatusMeta {
  value: string;
  label: string;
  badgeColor: string;
  badgeBackground: string;
  textColor: string;
  pdfFillColor: [number, number, number];
}

export interface InvoiceTotalsInfo {
  subtotal: number;
  taxAmount: number;
  discountAmount: number;
  shippingAmount: number;
  adjustmentsAmount: number;
  total: number;
  amountPaid: number;
  balanceDue: number;
}

export interface NormalizedInvoice {
  _id?: string;
  invoiceNumber: string;
  issueDate: Date;
  dueDate: Date;
  status: string;
  statusMeta: InvoiceStatusMeta;
  companyInfo: InvoiceCompanyInfo;
  tenant: InvoicePartyInfo;
  property: InvoicePropertyInfo;
  lineItems: InvoiceLineItemInfo[];
  totals: InvoiceTotalsInfo;
  subtotal: number;
  taxAmount: number;
  totalAmount: number;
  amountPaid: number;
  balanceRemaining: number;
  discountAmount: number;
  shippingAmount: number;
  adjustmentsAmount: number;
  notes: string;
  metadata?: Record<string, unknown>;
  tenantId?: InvoicePartyInfo;
  propertyId?: InvoicePropertyInfo;
  leaseId?: Record<string, unknown>;
  raw?: unknown;
}

export interface NormalizeInvoiceOptions {
  companyInfo?: Partial<InvoiceCompanyInfo>;
  defaultNotes?: string;
  fallbackStatus?: InvoiceStatus | string;
}

export const DEFAULT_COMPANY_INFO: InvoiceCompanyInfo = {
  name: "PropertyPro Management",
  address: "123 Business Ave, Suite 100, City, State 12345",
  phone: "+1 (555) 123-4567",
  email: "info@PropertyPro.com",
  website: "www.PropertyPro.com",
};

export const DEFAULT_INVOICE_NOTES =
  "We appreciate your business. Should you need us to add VAT or extra notes let us know!";

const STATUS_META_MAP: Record<string, InvoiceStatusMeta> = {
  paid: {
    value: "paid",
    label: "Paid",
    badgeBackground: "#dcfce7",
    badgeColor: "#166534",
    textColor: "#166534",
    pdfFillColor: [16, 185, 129],
  },
  partial: {
    value: "partial",
    label: "Partially Paid",
    badgeBackground: "#fef3c7",
    badgeColor: "#b45309",
    textColor: "#b45309",
    pdfFillColor: [234, 179, 8],
  },
  overdue: {
    value: "overdue",
    label: "Overdue",
    badgeBackground: "#fee2e2",
    badgeColor: "#991b1b",
    textColor: "#991b1b",
    pdfFillColor: [239, 68, 68],
  },
  cancelled: {
    value: "cancelled",
    label: "Cancelled",
    badgeBackground: "#e5e7eb",
    badgeColor: "#374151",
    textColor: "#374151",
    pdfFillColor: [156, 163, 175],
  },
  scheduled: {
    value: "scheduled",
    label: "Scheduled",
    badgeBackground: "#dbeafe",
    badgeColor: "#1d4ed8",
    textColor: "#1d4ed8",
    pdfFillColor: [59, 130, 246],
  },
  issued: {
    value: "issued",
    label: "Issued",
    badgeBackground: "#dbeafe",
    badgeColor: "#1d4ed8",
    textColor: "#1d4ed8",
    pdfFillColor: [59, 130, 246],
  },
};

function toNumber(value: unknown, fallback = 0): number {
  if (value === null || value === undefined) return fallback;
  const num = Number(value);
  return Number.isFinite(num) ? num : fallback;
}

function parseDate(value: unknown): Date {
  if (value instanceof Date) {
    return value;
  }

  if (typeof value === "string" || typeof value === "number") {
    const parsed = new Date(value);
    if (!Number.isNaN(parsed.getTime())) {
      return parsed;
    }
  }

  return new Date();
}

function ensureCompanyInfo(
  invoiceCompanyInfo?: Partial<InvoiceCompanyInfo>,
  overrideCompanyInfo?: Partial<InvoiceCompanyInfo>
): InvoiceCompanyInfo {
  return {
    ...DEFAULT_COMPANY_INFO,
    ...invoiceCompanyInfo,
    ...overrideCompanyInfo,
  };
}

function ensureParty(party: unknown): InvoicePartyInfo {
  if (!party || typeof party !== "object") {
    return {};
  }

  const source = party as Record<string, unknown>;
  const nestedUser = source["userId"];
  const fromNested =
    nestedUser && typeof nestedUser === "object"
      ? ensureParty(nestedUser)
      : undefined;

  const firstName =
    typeof source.firstName === "string"
      ? source.firstName
      : typeof source["first_name"] === "string"
      ? (source["first_name"] as string)
      : fromNested?.firstName;

  const lastName =
    typeof source.lastName === "string"
      ? source.lastName
      : typeof source["last_name"] === "string"
      ? (source["last_name"] as string)
      : fromNested?.lastName;

  const email =
    typeof source.email === "string"
      ? source.email
      : typeof source["contactEmail"] === "string"
      ? (source["contactEmail"] as string)
      : fromNested?.email;

  const phone =
    typeof source.phone === "string"
      ? source.phone
      : typeof source["contactPhone"] === "string"
      ? (source["contactPhone"] as string)
      : fromNested?.phone;

  const name =
    typeof source.name === "string"
      ? source.name
      : typeof source["fullName"] === "string"
      ? (source["fullName"] as string)
      : fromNested?.name;

  return {
    ...fromNested,
    ...source,
    firstName,
    lastName,
    email,
    phone,
    name,
  } as InvoicePartyInfo;
}

function ensureProperty(property: unknown): InvoicePropertyInfo {
  if (!property || typeof property !== "object") {
    return {};
  }

  const source = property as Record<string, unknown>;
  const address = source.address ?? source["propertyAddress"];

  return {
    ...source,
    name:
      typeof source.name === "string"
        ? source.name
        : typeof source["propertyName"] === "string"
        ? (source["propertyName"] as string)
        : undefined,
    address: address as InvoiceAddress | string | undefined,
  } as InvoicePropertyInfo;
}

function normalizeLineItems(items: unknown[]): InvoiceLineItemInfo[] {
  if (!Array.isArray(items) || items.length === 0) {
    return [];
  }

  return items.map((item) => {
    if (!item || typeof item !== "object") {
      return {
        description: "Item",
        quantity: 1,
        unitPrice: 0,
        total: 0,
        amount: 0,
      };
    }

    const source = item as Record<string, unknown>;
    const quantityRaw = source.quantity ?? 1;
    const quantity = toNumber(quantityRaw, 1) || 1;
    const unitPriceExplicit = source.unitPrice ?? source["unit_price"];
    const amountRaw = source.amount ?? source["total"];
    const amount = toNumber(amountRaw, 0);
    const unitPrice = toNumber(
      unitPriceExplicit,
      quantity > 0 ? amount / quantity : amount
    );
    const total = amount !== 0 ? amount : unitPrice * quantity;

    const dueDate =
      source.dueDate || source["due_date"] || source["due"] || undefined;

    return {
      description:
        typeof source.description === "string"
          ? source.description
          : typeof source["name"] === "string"
          ? (source["name"] as string)
          : "Item",
      quantity,
      unitPrice,
      total,
      amount: total,
      type:
        typeof source.type === "string"
          ? source.type
          : typeof source["category"] === "string"
          ? (source["category"] as string)
          : undefined,
      dueDate: dueDate ? parseDate(dueDate) : null,
      metadata: source,
    };
  });
}

export function getInvoiceStatusMeta(status?: string): InvoiceStatusMeta {
  const normalized = (status || "issued").toString().toLowerCase();
  const fromMap = STATUS_META_MAP[normalized];

  if (fromMap) {
    return { ...fromMap };
  }

  if (Object.values(InvoiceStatus).includes(normalized as InvoiceStatus)) {
    const fallback = STATUS_META_MAP[normalized] ?? STATUS_META_MAP.issued;
    return { ...fallback };
  }

  return { ...STATUS_META_MAP.issued };
}

export function normalizeInvoiceForPrint(
  invoice: unknown,
  options?: NormalizeInvoiceOptions
): NormalizedInvoice {
  const source = (invoice ?? {}) as Record<string, unknown>;

  const statusMeta = getInvoiceStatusMeta(
    (source.status as string | undefined) || options?.fallbackStatus
  );

  const issueDate = parseDate(
    source.issueDate ?? source["issuedAt"] ?? source["createdAt"]
  );
  const dueDate = parseDate(source.dueDate ?? source["due"] ?? issueDate);

  const lineItems = normalizeLineItems(
    (source.lineItems as unknown[]) ||
      (source["items"] as unknown[]) ||
      (source["line_items"] as unknown[]) ||
      []
  );

  const subtotalFromLines = lineItems.reduce((acc, item) => acc + item.total, 0);
  const subtotal = toNumber(source.subtotal, subtotalFromLines);
  const taxAmount = toNumber(
    source.taxAmount ?? source["tax"] ?? source["tax_total"],
    0
  );
  const discountAmount = toNumber(
    source["discountAmount"] ?? source["discount"] ?? source["discount_total"],
    0
  );
  const shippingAmount = toNumber(
    source["shippingAmount"] ?? source["shipping"] ?? source["shipping_total"],
    0
  );
  const adjustmentsAmount = toNumber(
    source["adjustmentsAmount"] ??
      source["adjustment"] ??
      source["adjustments"] ??
      0,
    0
  );

  const computedTotal =
    subtotal - discountAmount - adjustmentsAmount + taxAmount + shippingAmount;
  const totalAmount = toNumber(source.totalAmount, computedTotal);
  const amountPaid = toNumber(source.amountPaid ?? source["paid"], 0);
  const balanceRemaining = toNumber(
    source.balanceRemaining ?? source["balance"] ?? totalAmount - amountPaid,
    totalAmount - amountPaid
  );

  const companyInfoSource = source["companyInfo"] as
    | Partial<InvoiceCompanyInfo>
    | undefined;
  const companyInfo = ensureCompanyInfo(
    companyInfoSource,
    options?.companyInfo
  );

  const tenant = ensureParty(
    source.tenantId ?? source["tenant"] ?? source["customer"]
  );
  const property = ensureProperty(
    source.propertyId ?? source["property"] ?? source["unit"]
  );

  const notesSource =
    typeof source.notes === "string" && source.notes.trim().length > 0
      ? (source.notes as string)
      : options?.defaultNotes ?? DEFAULT_INVOICE_NOTES;

  const normalized: NormalizedInvoice = {
    _id:
      typeof source._id === "string"
        ? source._id
        : typeof source._id === "object" && source._id !== null
        ? (source._id as { toString?: () => string }).toString?.()
        : undefined,
    invoiceNumber:
      (source.invoiceNumber as string | undefined) ??
      (source["number"] as string | undefined) ??
      (source._id as string | undefined) ??
      "Invoice",
    issueDate,
    dueDate,
    status: statusMeta.value,
    statusMeta,
    companyInfo,
    tenant,
    property,
    lineItems,
    totals: {
      subtotal,
      taxAmount,
      discountAmount,
      shippingAmount,
      adjustmentsAmount,
      total: totalAmount,
      amountPaid,
      balanceDue: balanceRemaining,
    },
    subtotal,
    taxAmount,
    totalAmount,
    amountPaid,
    balanceRemaining,
    discountAmount,
    shippingAmount,
    adjustmentsAmount,
    notes: notesSource,
    metadata:
      (source.metadata as Record<string, unknown> | undefined) || undefined,
    tenantId: tenant,
    propertyId: property,
    leaseId: (source.leaseId as Record<string, unknown> | undefined) || undefined,
    raw: invoice,
  };

  return normalized;
}
