import mongoose, { Schema, Model } from "mongoose";

// Document version interface
export interface IDocumentVersion {
  version: number;
  fileUrl: string;
  fileSize: number;
  uploadedBy: mongoose.Types.ObjectId;
  uploadedAt: Date;
  changeLog?: string;
  r2ObjectKey?: string;
}

// Digital signature interface
export interface IDigitalSignature {
  signerId: mongoose.Types.ObjectId;
  signerName: string;
  signerEmail: string;
  signedAt: Date;
  signatureData: string; // Base64 encoded signature
  ipAddress?: string;
  userAgent?: string;
  verified: boolean;
  certificateId?: string;
}

// Document access log interface
export interface IDocumentAccessLog {
  userId: mongoose.Types.ObjectId;
  action: "view" | "download" | "edit" | "share" | "sign";
  timestamp: Date;
  ipAddress?: string;
  userAgent?: string;
  details?: string;
}

// Document sharing settings interface
export interface IDocumentSharing {
  isPublic: boolean;
  allowDownload: boolean;
  allowPrint: boolean;
  passwordProtected: boolean;
  password?: string;
  expiresAt?: Date;
  maxDownloads?: number;
  currentDownloads: number;
}

export interface IDocument {
  _id: string;
  name: string;
  description?: string;
  type: string;
  category: string;
  fileUrl: string;
  fileSize: number;
  mimeType: string;
  uploadedBy: mongoose.Types.ObjectId;
  tenantId?: mongoose.Types.ObjectId;
  propertyId?: mongoose.Types.ObjectId;
  leaseId?: mongoose.Types.ObjectId;
  sharedWith?: mongoose.Types.ObjectId[];
  tags: string[];
  status: "active" | "expired" | "pending" | "archived" | "signed" | "draft";
  isRequired: boolean;
  expiresAt?: Date;
  r2ObjectKey?: string;
  downloadCount: number;
  lastDownloadedAt?: Date;
  uploadedAt: Date;

  // Enhanced features
  versions: IDocumentVersion[];
  currentVersion: number;
  digitalSignatures: IDigitalSignature[];
  requiresSignature: boolean;
  signatureDeadline?: Date;
  accessLogs: IDocumentAccessLog[];
  sharing: IDocumentSharing;
  templateId?: mongoose.Types.ObjectId;
  isTemplate: boolean;
  autoGenerated: boolean;
  parentDocumentId?: mongoose.Types.ObjectId;

  // Metadata
  metadata: {
    author?: string;
    subject?: string;
    keywords?: string[];
    language?: string;
    pageCount?: number;
    wordCount?: number;
  };

  // Security
  encryption: {
    isEncrypted: boolean;
    algorithm?: string;
    keyId?: string;
  };

  // Compliance
  compliance: {
    retentionPeriod?: number; // in days
    retentionStartDate?: Date;
    legalHold: boolean;
    complianceNotes?: string;
  };

  createdAt: Date;
  updatedAt: Date;
  deletedAt?: Date;
}

const DocumentSchema = new Schema<IDocument>(
  {
    name: {
      type: String,
      required: [true, "Document name is required"],
      trim: true,
      maxlength: [255, "Document name too long"],
    },
    description: {
      type: String,
      trim: true,
      maxlength: [1000, "Description too long"],
    },
    type: {
      type: String,
      required: [true, "Document type is required"],
      enum: [
        "lease",
        "receipt",
        "notice",
        "insurance",
        "identification",
        "income",
        "maintenance",
        "inspection",
        "other",
      ],
      default: "other",
    },
    category: {
      type: String,
      required: [true, "Document category is required"],
      enum: [
        "lease",
        "payments",
        "maintenance",
        "insurance",
        "identification",
        "notices",
        "general",
      ],
      default: "general",
    },
    fileUrl: {
      type: String,
      required: [true, "File URL is required"],
      trim: true,
    },
    fileSize: {
      type: Number,
      required: [true, "File size is required"],
      min: [0, "File size cannot be negative"],
      max: [50 * 1024 * 1024, "File size cannot exceed 50MB"],
    },
    mimeType: {
      type: String,
      required: [true, "MIME type is required"],
      trim: true,
    },
    uploadedBy: {
      type: Schema.Types.ObjectId,
      ref: "User",
      required: [true, "Uploader is required"],
      index: true,
    },
    tenantId: {
      type: Schema.Types.ObjectId,
      ref: "Tenant",
      index: true,
    },
    propertyId: {
      type: Schema.Types.ObjectId,
      ref: "Property",
      index: true,
    },
    leaseId: {
      type: Schema.Types.ObjectId,
      ref: "Lease",
      index: true,
    },
    sharedWith: {
      type: [Schema.Types.ObjectId],
      ref: "User",
      default: [],
    },
    tags: {
      type: [String],
      default: [],
      validate: {
        validator: function (tags: string[]) {
          return tags.length <= 20;
        },
        message: "Cannot have more than 20 tags",
      },
    },
    status: {
      type: String,
      enum: ["active", "expired", "pending", "archived", "signed", "draft"],
      default: "active",
      required: [true, "Status is required"],
    },
    isRequired: {
      type: Boolean,
      default: false,
    },
    expiresAt: {
      type: Date,
      validate: {
        validator: function (date: Date) {
          if (!date) return true;
          return date > new Date();
        },
        message: "Expiration date must be in the future",
      },
    },
    r2ObjectKey: {
      type: String,
      trim: true,
    },
    downloadCount: {
      type: Number,
      default: 0,
      min: [0, "Download count cannot be negative"],
    },
    lastDownloadedAt: {
      type: Date,
    },
    uploadedAt: {
      type: Date,
      default: Date.now,
    },
    deletedAt: {
      type: Date,
      default: null,
    },

    // Enhanced features
    versions: {
      type: [
        {
          version: { type: Number, required: true },
          fileUrl: { type: String, required: true },
          fileSize: { type: Number, required: true },
          uploadedBy: {
            type: Schema.Types.ObjectId,
            ref: "User",
            required: true,
          },
          uploadedAt: { type: Date, default: Date.now },
          changeLog: { type: String },
          r2ObjectKey: { type: String },
        },
      ],
      default: [],
    },
    currentVersion: {
      type: Number,
      default: 1,
      min: 1,
    },
    digitalSignatures: {
      type: [
        {
          signerId: {
            type: Schema.Types.ObjectId,
            ref: "User",
            required: true,
          },
          signerName: { type: String, required: true },
          signerEmail: { type: String, required: true },
          signedAt: { type: Date, default: Date.now },
          signatureData: { type: String, required: true },
          ipAddress: { type: String },
          userAgent: { type: String },
          verified: { type: Boolean, default: false },
          certificateId: { type: String },
        },
      ],
      default: [],
    },
    requiresSignature: {
      type: Boolean,
      default: false,
    },
    signatureDeadline: {
      type: Date,
    },
    accessLogs: {
      type: [
        {
          userId: { type: Schema.Types.ObjectId, ref: "User", required: true },
          action: {
            type: String,
            enum: ["view", "download", "edit", "share", "sign"],
            required: true,
          },
          timestamp: { type: Date, default: Date.now },
          ipAddress: { type: String },
          userAgent: { type: String },
          details: { type: String },
        },
      ],
      default: [],
    },
    sharing: {
      isPublic: { type: Boolean, default: false },
      allowDownload: { type: Boolean, default: true },
      allowPrint: { type: Boolean, default: true },
      passwordProtected: { type: Boolean, default: false },
      password: { type: String },
      expiresAt: { type: Date },
      maxDownloads: { type: Number },
      currentDownloads: { type: Number, default: 0 },
    },
    templateId: {
      type: Schema.Types.ObjectId,
      ref: "Document",
    },
    isTemplate: {
      type: Boolean,
      default: false,
    },
    autoGenerated: {
      type: Boolean,
      default: false,
    },
    parentDocumentId: {
      type: Schema.Types.ObjectId,
      ref: "Document",
    },
    metadata: {
      author: { type: String },
      subject: { type: String },
      keywords: { type: [String], default: [] },
      language: { type: String, default: "en" },
      pageCount: { type: Number },
      wordCount: { type: Number },
    },
    encryption: {
      isEncrypted: { type: Boolean, default: false },
      algorithm: { type: String },
      keyId: { type: String },
    },
    compliance: {
      retentionPeriod: { type: Number },
      retentionStartDate: { type: Date },
      legalHold: { type: Boolean, default: false },
      complianceNotes: { type: String },
    },
  },
  {
    timestamps: true,
    toJSON: { virtuals: true },
    toObject: { virtuals: true },
  }
);

// Indexes for better query performance - with error handling to prevent emitWarning issues
try {
  DocumentSchema.index({ uploadedBy: 1, createdAt: -1 });
  DocumentSchema.index({ tenantId: 1, type: 1 });
  DocumentSchema.index({ propertyId: 1, category: 1 });
  DocumentSchema.index({ type: 1, status: 1 });
  DocumentSchema.index({ category: 1, status: 1 });
  DocumentSchema.index({ tags: 1 });
  DocumentSchema.index({ expiresAt: 1 });
  DocumentSchema.index({ deletedAt: 1 });
  DocumentSchema.index({ uploadedAt: -1 });

  // Text index for search functionality
  DocumentSchema.index({
    name: "text",
    description: "text",
    tags: "text",
  });
} catch (error) {
  // Silently handle index creation errors in development
  if (process.env.NODE_ENV !== "production") {
    console.warn("Document index creation warning:", error);
  }
}

// Virtual for file size in human readable format
DocumentSchema.virtual("fileSizeFormatted").get(function () {
  const bytes = this.fileSize;
  if (bytes === 0) return "0 Bytes";
  const k = 1024;
  const sizes = ["Bytes", "KB", "MB", "GB"];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
});

// Virtual for checking if document is expired
DocumentSchema.virtual("isExpired").get(function () {
  return this.expiresAt && this.expiresAt < new Date();
});

// Instance method to record download
DocumentSchema.methods.recordDownload = function () {
  this.downloadCount += 1;
  this.lastDownloadedAt = new Date();
  return this.save();
};

// Query middleware to exclude soft deleted documents
DocumentSchema.pre(/^find/, function () {
  // @ts-ignore
  this.find({ deletedAt: null });
});

// Create and export the model with safer initialization
let Document: Model<IDocument>;

try {
  // Try to get existing model first
  Document = mongoose.model<IDocument>("Document");
} catch (error) {
  // Model doesn't exist, create it
  Document = mongoose.model<IDocument>("Document", DocumentSchema);
}

export default Document;

DocumentSchema.statics.findByUploader = function (uploadedBy: string) {
  return this.find({ uploadedBy, deletedAt: null });
};

DocumentSchema.statics.searchDocuments = function (searchTerm: string) {
  return this.find({
    $text: { $search: searchTerm },
    deletedAt: null,
  }).sort({ score: { $meta: "textScore" } });
};

// Instance methods
DocumentSchema.methods.softDelete = function () {
  this.deletedAt = new Date();
  return this.save();
};

DocumentSchema.methods.restore = function () {
  this.deletedAt = null;
  return this.save();
};

DocumentSchema.methods.approve = function () {
  this.status = "active";
  return this.save();
};

DocumentSchema.methods.reject = function () {
  this.status = "expired";
  return this.save();
};

DocumentSchema.methods.archive = function () {
  this.status = "archived";
  return this.save();
};

DocumentSchema.methods.addTag = function (tag: string) {
  if (!this.tags.includes(tag.toLowerCase())) {
    this.tags.push(tag.toLowerCase());
  }
  return this.save();
};

DocumentSchema.methods.removeTag = function (tag: string) {
  this.tags = this.tags.filter((t) => t !== tag.toLowerCase());
  return this.save();
};

// Enhanced document methods
DocumentSchema.methods.createVersion = function (
  fileUrl: string,
  fileSize: number,
  uploadedBy: mongoose.Types.ObjectId,
  changeLog?: string,
  r2ObjectKey?: string
) {
  const newVersion = this.currentVersion + 1;
  this.versions.push({
    version: newVersion,
    fileUrl,
    fileSize,
    uploadedBy,
    uploadedAt: new Date(),
    changeLog,
    r2ObjectKey,
  });
  this.currentVersion = newVersion;
  this.fileUrl = fileUrl;
  this.fileSize = fileSize;
  this.r2ObjectKey = r2ObjectKey;
  return this.save();
};

DocumentSchema.methods.addDigitalSignature = function (
  signerId: mongoose.Types.ObjectId,
  signerName: string,
  signerEmail: string,
  signatureData: string,
  ipAddress?: string,
  userAgent?: string
) {
  this.digitalSignatures.push({
    signerId,
    signerName,
    signerEmail,
    signedAt: new Date(),
    signatureData,
    ipAddress,
    userAgent,
    verified: false,
  });

  // Check if all required signatures are collected
  if (this.requiresSignature && this.digitalSignatures.length > 0) {
    this.status = "signed";
  }

  return this.save();
};

DocumentSchema.methods.logAccess = function (
  userId: mongoose.Types.ObjectId,
  action: string,
  ipAddress?: string,
  userAgent?: string,
  details?: string
) {
  this.accessLogs.push({
    userId,
    action,
    timestamp: new Date(),
    ipAddress,
    userAgent,
    details,
  });

  // Update download count if action is download
  if (action === "download") {
    this.downloadCount += 1;
    this.lastDownloadedAt = new Date();
    if (this.sharing.maxDownloads) {
      this.sharing.currentDownloads += 1;
    }
  }

  return this.save();
};

DocumentSchema.methods.shareDocument = function (settings: {
  isPublic?: boolean;
  allowDownload?: boolean;
  allowPrint?: boolean;
  passwordProtected?: boolean;
  password?: string;
  expiresAt?: Date;
  maxDownloads?: number;
}) {
  Object.assign(this.sharing, settings);
  return this.save();
};

DocumentSchema.methods.verifySignature = function (signatureIndex: number) {
  if (this.digitalSignatures[signatureIndex]) {
    this.digitalSignatures[signatureIndex].verified = true;
    return this.save();
  }
  throw new Error("Signature not found");
};

DocumentSchema.methods.isSignedBy = function (userId: mongoose.Types.ObjectId) {
  return this.digitalSignatures.some(
    (sig) => sig.signerId.toString() === userId.toString()
  );
};

DocumentSchema.methods.getVersion = function (version: number) {
  return this.versions.find((v) => v.version === version);
};

DocumentSchema.methods.getCurrentVersionData = function () {
  return (
    this.versions.find((v) => v.version === this.currentVersion) || {
      version: this.currentVersion,
      fileUrl: this.fileUrl,
      fileSize: this.fileSize,
      uploadedBy: this.uploadedBy,
      uploadedAt: this.uploadedAt,
      r2ObjectKey: this.r2ObjectKey,
    }
  );
};
